﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Diagnostics.Eventing.Reader;
using EffectMaker.BusinessLogic.Manager;
using EffectMaker.BusinessLogic.Options;
using EffectMaker.DataModel.Attributes;
using EffectMaker.Foundation.Log;
using EffectMaker.Foundation.Texture;
using EffectMaker.Foundation.Utility;

namespace EffectMaker.DataModelLogic.BinaryConverters
{
    /// <summary>
    /// テクスチャが他のバイナリに属しているか、自身のバイナリで持っているかを通知するコンバータです。
    /// </summary>
    public class TextureLocationConverter : ConverterBase
    {
        /// <summary>
        /// Constructor.
        /// </summary>
        public TextureLocationConverter() : base(null)
        {
        }

        /// <summary>
        /// Get the maximum number of input values.
        /// </summary>
        public override int MaxInputValueCount
        {
            get { return 1; }
        }

        /// <summary>
        /// Get the description of the converter.
        /// </summary>
        public override string Description
        {
            get { return Properties.Resources.TextureLocationConverterDesc; }
        }

        /// <summary>
        /// Get the output value description.
        /// </summary>
        public override string OutputValueDescription
        {
            get { return Properties.Resources.TextureLocationConverterOutputValueDesc; }
        }

        /// <summary>
        /// Get the description of the input value.
        /// </summary>
        /// <param name="index">The index to the input value.</param>
        /// <returns>The description.</returns>
        public override string GetInputValueDescription(int index)
        {
            if (index == 0)
            {
                return Properties.Resources.TextureIDConverterInputValue0Desc;
            }
            else
            {
                return Properties.Resources.WarningInvalidConverterInputValueIndex;
            }
        }

        /// <summary>
        /// Convert values.
        /// </summary>
        /// <returns>True on success.</returns>
        public override bool Convert()
        {
            object inputValue = this.GetInputValue(0);

            var path = inputValue as string;
            if (string.IsNullOrEmpty(path) || !OptionStore.RuntimeOptions.IsSubBinaryConverting)
            {
                this.SetOutputValue((byte)0x00);
            }
            else
            {
                ulong guid = TextureManager.Instance.GetGuid(path);
                if (TextureManager.Instance.IsLocatedOnAnotherBinary(guid))
                {
                    this.SetOutputValue((byte)0x01);
                }
                else
                {
                    this.SetOutputValue((byte)0x00);
                }
            }

            return true;
        }
    }
}
