﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;

using EffectMaker.DataModel.Attributes;

using EffectMaker.Foundation.Log;
using EffectMaker.Foundation.Utility;

namespace EffectMaker.DataModelLogic.BinaryConverters
{
    /// <summary>
    /// Converter for selecting a single channel in the input primitive and
    /// feed the selected channel value to output.
    /// </summary>
    public class SelectPrimitiveChannelConverter : ConverterBase
    {
        /// <summary>
        /// 円周率です.
        /// </summary>
        private readonly float pi = 3.1415926535897932384626433832795f;

        /// <summary>Constant parameter name for the selected channel index.</summary>
        private const string ChannelIndexParamName = "ChannelIndex";

        /// <summary>Constant parameter name for radian convert.</summary>
        private const string RadianConvertModeName = "RadianConvertMode";

        /// <summary>The converter parameter holder.</summary>
        private static ConverterParamHolder paramHolder = new ConverterParamHolder();

        /// <summary>
        /// Static constructor.
        /// </summary>
        static SelectPrimitiveChannelConverter()
        {
            // Register parameters.
            paramHolder.RegisterParameter(
                ChannelIndexParamName,
                "The channel index of the primitive to output.",
                ValidateChannelIndexParamter);
            paramHolder.RegisterParameter(
                RadianConvertModeName,
                "Convert to radian.",
                ValidateRadianConvertModeParameter);
        }

        /// <summary>
        /// Constructor.
        /// </summary>
        public SelectPrimitiveChannelConverter() :
            base(paramHolder)
        {
        }

        /// <summary>
        /// Get the maximum number of input values.
        /// </summary>
        public override int MaxInputValueCount
        {
            get { return 1; }
        }

        /// <summary>
        /// Get the description of the converter.
        /// </summary>
        public override string Description
        {
            get { return Properties.Resources.SelectPrimitiveChannelConverterDesc; }
        }

        /// <summary>
        /// Get the output value description.
        /// </summary>
        public override string OutputValueDescription
        {
            get { return Properties.Resources.SelectPrimitiveChannelConverterOutputValueDesc; }
        }

        /// <summary>
        /// Get the description of the input value.
        /// </summary>
        /// <param name="index">The index to the input value.</param>
        /// <returns>The description.</returns>
        public override string GetInputValueDescription(int index)
        {
            if (index == 0)
            {
                return Properties.Resources.SelectPrimitiveChannelConverterInputValue0Desc;
            }
            else
            {
                return Properties.Resources.WarningInvalidConverterInputValueIndex;
            }
        }

        /// <summary>
        /// Convert data model field values.
        /// </summary>
        /// <returns>True on success.</returns>
        public override bool Convert()
        {
            object param;

            // Get the channel index from parameter.
            int channelIndex = 0;
            if (this.GetParameter(ChannelIndexParamName, out param) == true)
            {
                object convertedValue = param;
                TypeConversionUtility.TryConvert(param.GetType(), typeof(int), ref convertedValue);

                channelIndex = (int)convertedValue;
            }

            bool isRadianConvert = false;
            if (this.GetParameter(RadianConvertModeName, out param) == true)
            {
                object convertedValue = param;
                TypeConversionUtility.TryConvert(param.GetType(), typeof(int), ref convertedValue);

                isRadianConvert = (int)convertedValue != 0;
            }

            if (this.IsInputValueAssigned(0) == false)
            {
                Logger.Log(LogLevels.Warning, "SelectPrimitiveChannelConverter.Convert : There is no input value assigned.");
                return false;
            }

            var input = this.GetInputValue(0) as EffectMaker.Foundation.Interfaces.IPrimitive;
            if (input == null)
            {
                Logger.Log(LogLevels.Warning, "SelectPrimitiveChannelConverter.Convert : The assigned input value is not a primitive.");
                return false;
            }

            if (channelIndex < 0 || channelIndex >= input.Count)
            {
                Logger.Log(LogLevels.Warning, "SelectPrimitiveChannelConverter.Convert : The assigned channel index '{0}' is invalid for a primitive with {1} elements.", channelIndex, input.Count);
                return false;
            }

            if (isRadianConvert == true)
            {
                // ラジアンにコンバートする
                this.SetOutputValue((float)input[channelIndex] * (this.pi / 180.0f));
            }
            else
            {
                this.SetOutputValue(input[channelIndex]);
            }
            return true;
        }

        /// <summary>
        /// Validate channel index parameter.
        /// </summary>
        /// <param name="value">The parameter value.</param>
        /// <returns>True if the value is valid.</returns>
        private static bool ValidateChannelIndexParamter(object value)
        {
            int index;
            if (value is string)
            {
                if (int.TryParse((string)value, out index) == false)
                {
                    return false;
                }
            }
            else if (value != null)
            {
                object convertedValue = value;

                bool result = TypeConversionUtility.TryConvert(
                    value.GetType(),
                    typeof(int),
                    ref convertedValue);

                if (result == false)
                {
                    return false;
                }

                index = (int)convertedValue;
            }
            else
            {
                return false;
            }

            return true;
        }

        /// <summary>
        /// Validate frame convert mode parameter.
        /// </summary>
        /// <param name="value">The parameter value.</param>
        /// <returns>True if the value is valid.</returns>
        private static bool ValidateRadianConvertModeParameter(object value)
        {
            int mode;
            if (value is string)
            {
                if (int.TryParse((string)value, out mode) == false)
                {
                    return false;
                }
            }
            else if (value != null)
            {
                object convertedValue = value;

                bool result = TypeConversionUtility.TryConvert(
                    value.GetType(),
                    typeof(int),
                    ref convertedValue);

                if (result == false)
                {
                    return false;
                }

                mode = (int)convertedValue;
            }
            else
            {
                return false;
            }

            return true;
        }
    }
}
