﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;

using EffectMaker.DataModel.Attributes;

using EffectMaker.Foundation.Log;
using EffectMaker.Foundation.Utility;

namespace EffectMaker.DataModelLogic.BinaryConverters
{
    /// <summary>
    /// Converter for converting value into specified type.
    /// </summary>
    public class PrimitiveComponentTypeConversionConverter : ConverterBase
    {
        /// <summary>Constant for target type parameter name.</summary>
        private const string TargetTypeParamName = "TargetType";

        /// <summary>The converter parameter holder.</summary>
        private static ConverterParamHolder paramHolder = new ConverterParamHolder();

        /// <summary>
        /// Static constructor.
        /// </summary>
        static PrimitiveComponentTypeConversionConverter()
        {
            // Register parameters.
            paramHolder.RegisterParameter(
                TargetTypeParamName,
                "The target output value type.",
                ValidateTargetTypeParamter);
        }

        /// <summary>
        /// Constructor.
        /// </summary>
        public PrimitiveComponentTypeConversionConverter() :
            base(paramHolder)
        {
        }

        /// <summary>
        /// Get the maximum number of input values.
        /// </summary>
        public override int MaxInputValueCount
        {
            get { return 1; }
        }

        /// <summary>
        /// Get the description of the converter.
        /// </summary>
        public override string Description
        {
            get { return Properties.Resources.TypeConversionConverterDesc; }
        }

        /// <summary>
        /// Get the output value description.
        /// </summary>
        public override string OutputValueDescription
        {
            get { return Properties.Resources.TypeConversionConverterOutputValueDesc; }
        }

        /// <summary>
        /// Get the description of the input value.
        /// </summary>
        /// <param name="index">The index to the input value.</param>
        /// <returns>The description.</returns>
        public override string GetInputValueDescription(int index)
        {
            if (index == 0)
            {
                return Properties.Resources.TypeConversionConverterInputValue0Desc;
            }
            else
            {
                return Properties.Resources.WarningInvalidConverterInputValueIndex;
            }
        }

        /// <summary>
        /// Convert data model field values.
        /// </summary>
        /// <returns>True on success.</returns>
        public override bool Convert()
        {
            // Is the input value assigned at the index?
            if (this.IsInputValueAssigned(0) == false)
            {
                Logger.Log(LogLevels.Error, "PrimitiveComponentTypeConversionConverter.Convert : The input value has not been assigned yet.");
                return false;
            }

            object param;

            // Get the element type from the parameter.
            Type targetType = typeof(byte);
            if (this.GetParameter(TargetTypeParamName, out param) == true)
            {
                targetType = (Type)param;
            }

            // The input is assigned, get it.
            var input = this.GetInputValue(0) as EffectMaker.Foundation.Interfaces.IPrimitive;
            if (input == null)
            {
                Logger.Log(LogLevels.Error, "PrimitiveComponentTypeConversionConverter.Convert : The input value is null and cannot be converted.");
                return false;
            }

            // 出力先変数を生成.
            Type listType = typeof(System.Collections.Generic.List<>);
            Type outputType = listType.MakeGenericType(targetType);
            System.Collections.IList output = Activator.CreateInstance(outputType) as System.Collections.IList;

            for (int i = 0; i < input.Count; ++i)
            {
                // Convert the input value to the desired type.
                object convertedInputValue = input[i];
                if (TypeConversionUtility.TryConvert(
                    input[i].GetType(),
                    targetType,
                    ref convertedInputValue) == false)
                {
                    Logger.Log(LogLevels.Error, "PrimitiveComponentTypeConversionConverter.Convert : Failed converting the input value to the target type.");
                    return false;
                }

                output.Add(convertedInputValue);
            }

            // Set the converted input value to the output.
            this.SetOutputValue(output);

            return true;
        }

        /// <summary>
        /// Validate target type parameter.
        /// </summary>
        /// <param name="value">The parameter value.</param>
        /// <returns>True if the value is valid.</returns>
        private static bool ValidateTargetTypeParamter(object value)
        {
            Type type;
            if (value is Type)
            {
                type = (Type)value;
            }
            else if (value is string)
            {
                string[] matches =
                    RegexUtility.ExtractStrings("^typeof\\(([^\\)]+)\\)$", (string)value);
                if (matches == null || matches.Length <= 0 || matches[0] == null)
                {
                    return false;
                }

                type = Type.GetType(matches[0], false);
                if (type == null)
                {
                    return false;
                }
            }
            else
            {
                return false;
            }

            return true;
        }
    }
}
