﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections;
using System.Linq;

using EffectMaker.DataModel.AnimationTable;
using EffectMaker.DataModel.Attributes;

using EffectMaker.Foundation.Log;
using EffectMaker.Foundation.Utility;

namespace EffectMaker.DataModelLogic.BinaryConverters
{
    /// <summary>
    /// Convert animation tables into an array of their key frame count by combiner project path.
    /// </summary>
    class CombinerShaderAnimationTableCountConverter : ConverterBase
    {
        /// <summary>
        /// Constructor.
        /// </summary>
        public CombinerShaderAnimationTableCountConverter()
            : base(null)
        {
        }

        /// <summary>
        /// Get the maximum number of input values.
        /// </summary>
        public override int MaxInputValueCount
        {
            get { return 4; }
        }

        /// <summary>
        /// Get the description of the converter.
        /// </summary>
        public override string Description
        {
            get { return Properties.Resources.AnimationTableCountConverterDesc; }
        }

        /// <summary>
        /// Get the output value description.
        /// </summary>
        public override string OutputValueDescription
        {
            get { return Properties.Resources.AnimationTableCountConverterOutputValueDesc; }
        }

        /// <summary>
        /// Get the description of the input value.
        /// </summary>
        /// <param name="index">The index to the input value.</param>
        /// <returns>The description.</returns>
        public override string GetInputValueDescription(int index)
        {
            if (index == 0)
            {
                return Properties.Resources.AnimationTableCountConverterInputValueDesc;
            }
            else
            {
                return Properties.Resources.WarningInvalidConverterInputValueIndex;
            }
        }

        /// <summary>
        /// Convert data model field values.
        /// </summary>
        /// <returns>True on success.</returns>
        public override bool Convert()
        {
            if (this.IsInputValueAssigned(0) == false)
            {
                Logger.Log(LogLevels.Warning, "CombinerShaderAnimationTableCountConverter.Convert : There is no input value assigned.");
                return false;
            }

            int? shaderType = this.GetInputValue(0) as int?;
            if (shaderType == null)
            {
                Logger.Log(LogLevels.Warning, "CombinerShaderAnimationTableCountConverter.Convert : There is no input value assigned.");
                return false;
            }

            if (this.IsInputValueAssigned(1) == false)
            {
                Logger.Log(LogLevels.Warning, "CombinerShaderAnimationTableCountConverter.Convert : There is no input value assigned.");
                return false;
            }

            var combinerPath = this.GetInputValue(1) as String;
            if (combinerPath == null)
            {
                Logger.Log(LogLevels.Warning, "CombinerShaderAnimationTableCountConverter.Convert : There is no input value assigned.");
                return false;
            }

            if (this.IsInputValueAssigned(2) == false || this.IsInputValueAssigned(3) == false)
            {
                Logger.Log(LogLevels.Warning, "CombinerShaderAnimationTableCountConverter.Convert : There is no input value assigned.");
                return false;
            }

            AnimationTableData table;
            if (combinerPath == String.Empty && shaderType != 0)
            {
                table = this.GetInputValue(2) as AnimationTableData;
                if (table == null)
                {
                    Logger.Log(LogLevels.Warning, "CombinerShaderAnimationTableCountConverter.Convert : The assigned input value is not an animation table.");
                    return false;
                }
            }
            else
            {
                table = this.GetInputValue(3) as AnimationTableData;
                if (table == null)
                {
                    Logger.Log(LogLevels.Warning, "CombinerShaderAnimationTableCountConverter.Convert : The assigned input value is not an animation table.");
                    return false;
                }
            }

            this.SetOutputValue(table.Count);

            return true;
        }
    }
}
