﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections;

using EffectMaker.DataModel.Attributes;
using EffectMaker.Foundation.Utility;

namespace EffectMaker.DataModelLogic.BinaryConverters
{
    /// <summary>
    /// コレクションの各値をコンバートするクラスです.
    /// </summary>
    public class CollectionValueConverter : ConverterBase
    {
        /// <summary>Constant for element count parameter name.</summary>
        private const string ElementCountParamName = "ElementCount";

        /// <summary>Constant for element type parameter name.</summary>
        private const string ElementTypeParamName = "ElementType";

        /// <summary>The converter parameter holder.</summary>
        private static ConverterParamHolder paramHolder = new ConverterParamHolder();

        /// <summary>
        /// Static constructor.
        /// </summary>
        static CollectionValueConverter()
        {
            paramHolder.RegisterParameter(
                ElementCountParamName,
                "The output element count.",
                ValidateElementCountParamter);

            paramHolder.RegisterParameter(
                ElementTypeParamName,
                "The type of the output elements.",
                ValidateElementTypeParamter);
        }

        /// <summary>
        /// Constructor.
        /// </summary>
        public CollectionValueConverter() :
            base(paramHolder)
        {
            /* DO_NOTHING */
        }

        /// <summary>
        /// Get the maximum number of input values.
        /// </summary>
        public override int MaxInputValueCount
        {
            get { return 1; }
        }

        /// <summary>
        /// Get the description of the converter.
        /// </summary>
        public override string Description
        {
            get { return Properties.Resources.CollectionValueConverterDesc; }
        }

        /// <summary>
        /// Get the output value description.
        /// </summary>
        public override string OutputValueDescription
        {
            get { return Properties.Resources.CollectionValueConverterOutputValueDesc; }
        }

        /// <summary>
        /// Get the description of the input value.
        /// </summary>
        /// <param name="index">The index to the input value.</param>
        /// <returns>The description.</returns>
        public override string GetInputValueDescription(int index)
        {
            if (index == 0)
            {
                return Properties.Resources.CollectionValueConverterInputValueDesc;
            }
            else
            {
                return Properties.Resources.WarningInvalidConverterInputValueIndex;
            }
        }

        /// <summary>
        /// Convert data model field values.
        /// </summary>
        /// <returns>True on success.</returns>
        public override bool Convert()
        {
            // 入力値を取得.
            var input = this.GetInputValue(0) as IEnumerable;
            if (input == null)
            {
                return false;
            }

            object param;

            // 配列サイズを取得.
            int elementCount = 1;
            if (this.GetParameter(ElementCountParamName, out param) == true)
            {
                elementCount = (int)param;
            }

            // 配列要素の型を取得.
            Type elementType = typeof(float);
            if (this.GetParameter(ElementTypeParamName, out param) == true)
            {
                elementType = (Type)param;
            }

            // 固定長サイズの配列を作成.
            Array output = Array.CreateInstance(elementType, elementCount);

            // デフォルト値を取得する.
            object defaultValue = null;
            if (elementType.IsValueType == true)
            {
                defaultValue = Activator.CreateInstance(elementType);
            }

            // 入力要素がIEnumerableである場合.
            if (input is IEnumerable)
            {
                int count = 0;
                IEnumerable collection = input as IEnumerable;

                // 各要素をコンバートしながら格納.
                foreach (var item in collection)
                {
                    // カウントが格納サイズ未満である場合のみ処理.
                    if (count < elementCount)
                    {
                        // 入力値.
                        object convertedInputValue = item;

                        // 型変換.
                        if (TypeConversionUtility.TryConvert(
                            item.GetType(),
                            elementType,
                            ref convertedInputValue) == false)
                        {
                            // 失敗した場合は，デフォルト値で埋める.
                            output.SetValue(defaultValue, count);
                            count++;
                            continue;
                        }

                        // 変換したデータを格納.
                        output.SetValue(convertedInputValue, count);
                        count++;
                    }
                }

                // 残り分をデフォルト値で埋める.
                if (count < elementCount)
                {
                    for (int i = count; i < elementCount; ++i)
                    {
                        output.SetValue(defaultValue, i);
                    }
                }
            }
            else
            {
                // デフォルト値で埋める.
                for (int i = 0; i < elementCount; ++i)
                {
                    output.SetValue(defaultValue, i);
                }
            }

            // 出力値設定.
            this.SetOutputValue(output);

            // 正常終了.
            return true;
        }

        /// <summary>
        /// Validate element count parameter.
        /// </summary>
        /// <param name="value">The parameter value.</param>
        /// <returns>True if the value is valid.</returns>
        private static bool ValidateElementCountParamter(object value)
        {
            int count;
            if (value is int)
            {
                count = (int)value;
            }
            else if (value is string)
            {
                if (int.TryParse((string)value, out count) == false)
                {
                    return false;
                }
            }
            else
            {
                return false;
            }

            if (count <= 0 || count > int.MaxValue)
            {
                return false;
            }

            return true;
        }

        /// <summary>
        /// Validate element type parameter.
        /// </summary>
        /// <param name="value">The parameter value.</param>
        /// <returns>True if the value is valid.</returns>
        private static bool ValidateElementTypeParamter(object value)
        {
            Type type;
            if (value is Type)
            {
                type = (Type)value;
            }
            else if (value is string)
            {
                string[] matches =
                    RegexUtility.ExtractStrings("^typeof\\(([^\\)]+)\\)$", (string)value);
                if (matches == null || matches.Length <= 0 || matches[0] == null)
                {
                    return false;
                }

                type = Type.GetType(matches[0], false);
                if (type == null)
                {
                    return false;
                }
            }
            else
            {
                return false;
            }

            return true;
        }
    }
}
