﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System.Collections;
using System.Linq;

using EffectMaker.DataModel.AnimationTable;
using EffectMaker.DataModel.Attributes;

using EffectMaker.Foundation.Log;
using EffectMaker.Foundation.Primitives;
using EffectMaker.Foundation.Utility;

namespace EffectMaker.DataModelLogic.BinaryConverters
{
    /// <summary>
    /// Convert alpha animation table.
    /// </summary>
    public class AlphaAnimationTableConverter : ConverterBase
    {
        /// <summary>Constant for maximum key frame count parameter name.</summary>
        private const string MaxKeyFrameCountParamName = "MaxKeyFrameCount";

        /// <summary>The converter parameter holder.</summary>
        private static ConverterParamHolder paramHolder = new ConverterParamHolder();

        /// <summary>
        /// Static constructor.
        /// </summary>
        static AlphaAnimationTableConverter()
        {
            // Register parameters.
            paramHolder.RegisterParameter(
                MaxKeyFrameCountParamName,
                "The maximum key frame count to output.",
                ValidateMaxKeyFrameCountParameter);
        }

        /// <summary>
        /// Constructor.
        /// </summary>
        public AlphaAnimationTableConverter() :
            base(paramHolder)
        {
        }

        /// <summary>
        /// Get the maximum number of input values.
        /// </summary>
        public override int MaxInputValueCount
        {
            get { return 1; }
        }

        /// <summary>
        /// Get the description of the converter.
        /// </summary>
        public override string Description
        {
            get { return Properties.Resources.AnimationTableConverterDesc; }
        }

        /// <summary>
        /// Get the output value description.
        /// </summary>
        public override string OutputValueDescription
        {
            get { return Properties.Resources.AnimationTableConverterOutputValueDesc; }
        }

        /// <summary>
        /// Get the description of the input value.
        /// </summary>
        /// <param name="index">The index to the input value.</param>
        /// <returns>The description.</returns>
        public override string GetInputValueDescription(int index)
        {
            if (index == 0)
            {
                return Properties.Resources.AnimationTableConverterInputValue0Desc;
            }
            else
            {
                return Properties.Resources.WarningInvalidConverterInputValueIndex;
            }
        }

        /// <summary>
        /// Convert data model field values.
        /// </summary>
        /// <returns>True on success.</returns>
        public override bool Convert()
        {
            object param;

            // Get the maximum key frame count from parameter.
            int maxKeyFrameCount = 8;
            if (this.GetParameter(MaxKeyFrameCountParamName, out param) == true)
            {
                object convertedValue = param;
                TypeConversionUtility.TryConvert(param.GetType(), typeof(int), ref convertedValue);

                maxKeyFrameCount = (int)convertedValue;
            }

            // Create the array of vector4 for output.
            var output = new Vector4f[maxKeyFrameCount];

            if (this.IsInputValueAssigned(0) == false)
            {
                Logger.Log(LogLevels.Warning, "AnimationTableConverter.Convert : There is no input value assigned.");
                return false;
            }

            var table = this.GetInputValue(0) as AnimationTableData;
            if (table == null)
            {
                Logger.Log(LogLevels.Warning, "AnimationTableConverter.Convert : The assigned input value is not an animation table.");
                return false;
            }

            // Convert the animation table.
            for (int i = 0; i < maxKeyFrameCount; ++i)
            {
                if (i < table.Count)
                {
                    KeyFrameData data = table[i];
                    output[i] = new Vector4f(
                        data.Value.W,
                        data.Value.W,
                        data.Value.W,
                        (float)data.Frame / 100.0f);
                }
                else
                {
                    output[i] = new Vector4f(0.0f, 0.0f, 0.0f, 0.0f);
                }
            }

            this.SetOutputValue(output);

            return true;
        }

        /// <summary>
        /// Validate maximum key frame count parameter.
        /// </summary>
        /// <param name="value">The parameter value.</param>
        /// <returns>True if the value is valid.</returns>
        private static bool ValidateMaxKeyFrameCountParameter(object value)
        {
            int count;
            if (value is string)
            {
                if (int.TryParse((string)value, out count) == false)
                {
                    return false;
                }
            }
            else if (value != null)
            {
                object convertedValue = value;

                bool result = TypeConversionUtility.TryConvert(
                    value.GetType(),
                    typeof(int),
                    ref convertedValue);

                if (result == false)
                {
                    return false;
                }

                count = (int)convertedValue;
            }
            else
            {
                return false;
            }

            return true;
        }
    }
}
