﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading;
using System.Threading.Tasks;
using EffectMaker.BusinessLogic.IO;
using EffectMaker.BusinessLogic.Synchronization;
using EffectMaker.Foundation.Log;
using EffectMaker.Foundation.Utility;

namespace EffectMaker.BusinessLogic.ViewerMessages
{
    /// <summary>
    /// Viewer message class that sends binary data to the viewer.
    /// </summary>
    public class SendBinaryMessage : MessageBase
    {
        /// <summary>The type of the asset the binary data is.</summary>
        private AssetTypes assetType;

        /// <summary>The Guid of the asset.</summary>
        private Guid assetGuid;

        /// <summary>The byte array that contains the message data.</summary>
        private byte[] messageData = null;

        /// <summary>The actual size of the message data in bytes.</summary>
        private int messageDataSize = 0;

        /// <summary>
        /// The stream that stores (or will store) the binary data.
        /// </summary>
        /// <remarks>
        /// The data might be written to the stream asynchronously, which means
        /// more data might be written after the construction of the message.
        /// One should not assume the message is ready without checking the Status
        /// property in the MessageBase class.
        /// </remarks>
        private MemoryStream binaryDataStream = null;

        /// <summary>
        /// Constructor.
        /// </summary>
        /// <remarks>
        /// This constructor can be used when the binary data is written asynchronously,
        /// thus the default status of this message is set to MessageStatus.Awaiting, and
        /// the user must explicitly set the message ready with MessageBase.SetReady() method.
        /// </remarks>
        /// <param name="assetType">The type of the asset.</param>
        /// <param name="assetGuid">The Guid of the asset.</param>
        /// <param name="stream">The stream that contains or will contain the binary data.</param>
        public SendBinaryMessage(
            AssetTypes assetType,
            Guid assetGuid,
            MemoryStream stream)
        {
            if (stream == null || stream.CanRead == false)
            {
                throw new ArgumentException("The stream is either empty or closed.");
            }

            this.assetType = assetType;
            this.assetGuid = assetGuid;
            this.binaryDataStream = stream;

            // The binary data might not be ready yet, we have to wait for
            // the user to explicitly set this message to ready.
            this.Status = MessageStatus.Awaiting;
        }

        /// <summary>
        /// ViewerMessageを送信する時専用のコンストラクタです。
        /// ファイル書き込みを介さず、直接データを転送します。
        /// </summary>
        /// <param name="assetType">The type of the asset.</param>
        /// <param name="assetGuid">The Guid of the asset.</param>
        /// <param name="binaryData">The byte array contains the binary data of the asset.</param>
        /// <param name="binaryDataSize">The actual size of the binary data.</param>
        /// <remarks>
        /// The byte array can be larger than the binary data, as long as the binary data size
        /// is correctly set.
        /// </remarks>
        public SendBinaryMessage(
            AssetTypes assetType,
            Guid assetGuid,
            byte[] binaryData,
            int binaryDataSize) : base()
        {
            Debug.Assert(assetType == AssetTypes.ViewerMessage, "このコンストラクタはViewerMessage専用。");

            this.assetType = assetType;
            this.assetGuid = assetGuid;

            this.messageData = binaryData;
            this.messageDataSize = binaryDataSize;
            ////this.WriteBinaryDataToFile(binaryData, binaryDataSize);
        }

        /// <summary>
        /// Get the type of the message.
        /// </summary>
        public override MessageTypes MessageType
        {
            get { return MessageTypes.SendBinary; }
        }

        /// <summary>
        /// Get the size of the message.
        /// </summary>
        public override int MessageSize
        {
            get
            {
                // (sizeof(assetType) = 4) + (sizeof(assetGuid) = 16) = 20
                return 20 + this.messageDataSize;
            }
        }

        /// <summary>
        /// Get the asset type of the binary data.
        /// </summary>
        public AssetTypes AssetType
        {
            get { return this.assetType; }
        }

        /// <summary>
        /// Get the Guid of the asset.
        /// </summary>
        public Guid AssetGuid
        {
            get { return this.assetGuid; }
        }

        /// <summary>
        /// Set the status of the message to ready.
        /// Note that if Send() is called, the message will still be sent
        /// either the status is ready or not.
        /// The status only affect the message when it is queuing up in SendMessageQueue.
        /// </summary>
        public override void SetReady()
        {
            if (this.binaryDataStream != null &&
                this.binaryDataStream.CanRead == true)
            {
                this.WriteBinaryDataToFile(
                    this.binaryDataStream.GetBuffer(),
                    (int)this.binaryDataStream.Length);
            }

            // Clear the stream reference to prevent accessing it accidentally.
            this.binaryDataStream = null;

            base.SetReady();
        }

        /// <summary>
        /// Cancel this message so the message will not be sent.
        /// Note that if Send() is called, the message will still be sent
        /// no matter what the status is.
        /// The status only affect the message when it is queuing up in SendMessageQueue.
        /// </summary>
        public override void CancelMessage()
        {
            // Clear the stream reference to prevent accessing it accidentally.
            this.binaryDataStream = null;

            base.CancelMessage();
        }

        /// <summary>
        /// Write message data to a stream with the given writer.
        /// </summary>
        /// <param name="writer">The message writer.</param>
        /// <returns>True on success.</returns>
        protected override bool WriteMessage(MessageWriter writer)
        {
            return true;
        }

        /// <summary>
        /// Write the rest of the message data before the message is actually sent.
        /// </summary>
        /// <param name="writer">The message writer.</param>
        /// <returns>True on success.</returns>
        protected override bool WriteMessageBeforeSent(MessageWriter writer)
        {
            if (base.WriteMessage(writer) == false)
            {
                return false;
            }

            writer.Write(BinaryConversionUtility.ForProtocol.Convert((int)this.assetType));
            writer.Write(BinaryConversionUtility.ForProtocol.Convert(this.assetGuid));

            if (base.WriteMessageBeforeSent(writer) == false)
            {
                return false;
            }

            if (this.messageData != null && this.messageDataSize > 0)
            {
                writer.Write(this.messageData, 0, this.messageDataSize);
            }

            return true;
        }

        /// <summary>
        /// Write the binary data to file.
        /// This message only sends the binary file path, the viewer will open the binary
        /// file by itself.
        /// </summary>
        /// <param name="binaryData">The byte array contains the binary data of the asset.</param>
        /// <param name="binaryDataSize">The actual size of the binary data.</param>
        private void WriteBinaryDataToFile(byte[] binaryData, int binaryDataSize)
        {
            string globalTempDirectoryPath = Path.Combine(IOConstants.AppDataWorkPath, "ViewerBinaries");
            string localTempDirectoryPath = Path.Combine(globalTempDirectoryPath, IOConstants.ProcessId);

            // テンポラリフォルダを作成
            if (Directory.Exists(localTempDirectoryPath) == false)
            {
                using (new GlobalSyncSection())
                {
                    IOUtility.SafeCreateTemporaryDirectory(globalTempDirectoryPath, IOUtility.TemporaryDirectoryUsage.ForGlobal);
                    IOUtility.SafeCreateTemporaryDirectory(localTempDirectoryPath, IOUtility.TemporaryDirectoryUsage.ForMyProcess);
                }
            }

            // バイナリファイルパスを設定
            string fileName = string.Format("{0}_{1}.bin", this.assetType.ToString(), this.assetGuid.ToString("D"));
            string filePath = Path.Combine(localTempDirectoryPath, fileName);

            // Write the binary data to temporary file.
            int count = 0;
            FileStream stream = null;
            while (true)
            {
                try
                {
                    stream = new FileStream(filePath, FileMode.Create, FileAccess.Write);
                    break;
                }
                catch (Exception ex)
                {
                    // The viewer might be reading the file, wait a little bit then try again.
                    Thread.Sleep(10);
                    if (++count >= 100)
                    {
                        // Too many retries, something went wrong.
                        Logger.Log(LogLevels.Fatal, "SendBinaryMessage.WriteBinaryDataToFile : Failed writing binary data to file : {0}", filePath);
                        throw ex;
                    }
                }
            }

            // Save binary data to temporary file.
            stream.Write(binaryData, 0, binaryDataSize);
            stream.Close();

            // Convert file path to byte array.
            byte[] pathBuffer = Encoding.GetEncoding("Shift_JIS").GetBytes(filePath);

            // Create another buffer to append the null-termination byte.
            byte[] nullTerminatedBuffer = new byte[pathBuffer.Length + 1];
            Array.Copy(pathBuffer, nullTerminatedBuffer, pathBuffer.Length);
            nullTerminatedBuffer[pathBuffer.Length] = 0;

            // Use the file path as the binary data, instead of the actual binary data.
            this.messageData = nullTerminatedBuffer;
            this.messageDataSize = nullTerminatedBuffer.Length;
        }
    }
}
