﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;

using EffectMaker.BusinessLogic.IO;

using EffectMaker.Foundation.Log;
using EffectMaker.Foundation.Serialization;
using EffectMaker.Foundation.Utility;

namespace EffectMaker.BusinessLogic.UserData
{
    /// <summary>
    /// UserDataDefinitionLoader.
    /// </summary>
    public class UserDataDefinitionLoader
    {
        /// <summary>
        /// Constructor.
        /// </summary>
        public UserDataDefinitionLoader()
        {
        }

        /// <summary>
        /// Load the user data definitions from the specified file.
        /// </summary>
        /// <remarks>
        /// This method ONLY loads the definitions.
        /// It DOES NOT compile the data model and binary conversion
        /// info source files, and it DOES NOT load the XAML files.
        /// </remarks>
        /// <param name="filePath">The file path.</param>
        /// <returns>The loaded definitions.</returns>
        public UserDataDefinitionList LoadDefinition(string filePath)
        {
            if (string.IsNullOrEmpty(filePath) == true || File.Exists(filePath) == false)
            {
                return null;
            }

            UserDataDefinitionList defList = null;
            using (var stream = new FileStream(filePath, FileMode.Open, FileAccess.Read))
            {
                // Load the definitions.
                defList =
                    SerializationHelper.LoadXmlDocSerializable<UserDataDefinitionList>(stream);
                if (defList == null)
                {
                    return null;
                }
            }

            // Set up the base folder path, because the paths in the definitions
            // are usually relative paths or even just file names.
            defList.BaseFolderPath = Path.GetDirectoryName(filePath);

            return defList;
        }

        /// <summary>
        /// Set up the user data.
        /// Compiles the data model and binary conversion info, and saves
        /// the XAML file path for later inquiry.
        /// </summary>
        /// <param name="defList">The user data definitions.</param>
        /// <returns>True on success.</returns>
        public virtual bool SetupUserData(UserDataDefinitionList defList)
        {
            if (defList == null)
            {
                return true;
            }

            foreach (UserDataDefinition def in defList.UserDataDefinitions)
            {
                // Load the user data.
                // The data model and binary conversion info will be compiled to assembly
                // and the XAML file path is saved in the user data manager.
                bool result = UserDataManager.LoadUserData(
                    def.DataModelName,
                    def.TargetTabName,
                    def.DataModelFileName,
                    def.ConversionInfoFileName,
                    def.XamlFileName);

                if (result == false)
                {
                    Logger.Log(
                        LogLevels.Warning,
                        "Failed setting up user data '{0}'.",
                        def.DataModelName);
                    return false;
                }
            }

            return true;
        }
    }
}
