﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Reflection;
using System.Text;
using System.Threading.Tasks;
using System.Xml.Serialization;

using EffectMaker.Foundation.Log;
using EffectMaker.Foundation.Serialization;
using EffectMaker.Foundation.Utility;

namespace EffectMaker.BusinessLogic.UserData
{
    /// <summary>
    /// The base class of the list of user data definitions.
    /// </summary>
    public class UserDataDefinitionList : IXmlDocSerializable
    {
        /// <summary>The base folder path.</summary>
        private string baseFolderPath = string.Empty;

        /// <summary>
        /// Default constructor.
        /// </summary>
        public UserDataDefinitionList()
        {
        }

        /// <summary>
        /// Get or set the base folder path.
        /// </summary>
        [XmlIgnore]
        public string BaseFolderPath
        {
            get
            {
                return this.baseFolderPath;
            }

            set
            {
                this.baseFolderPath = PathUtility.LoadEnvironments(value);
                this.UserDataDefinitions.ForEach(s => s.BaseFolderPath = this.baseFolderPath);
            }
        }

        /// <summary>
        /// Get or set the user data definitions.
        /// </summary>
        public List<UserDataDefinition> UserDataDefinitions { get; set; }

        /// <summary>
        /// Deserializes from the given XML node.
        /// </summary>
        /// <param name="context">The data context needed for the deserialization.</param>
        /// <returns>True on success.</returns>
        public bool ReadXml(XmlDocSerializationContext context)
        {
            var definitions =
                this.ReadElementsByTagName<UserDataDefinition>(context, "UserDataModelInfo");

            this.UserDataDefinitions = new List<UserDataDefinition>();
            this.UserDataDefinitions.AddRange(definitions);

            return true;
        }

        /// <summary>
        /// Serializes this object to a XML node.
        /// </summary>
        /// <param name="context">The data context needed for the serialization.</param>
        /// <returns>True on success.</returns>
        public bool WriteXml(XmlDocSerializationContext context)
        {
            return false;
        }
    }

    /// <summary>
    /// The user data definitions.
    /// </summary>
    public class UserDataDefinition : IXmlDocSerializable
    {
        /// <summary>Data model source file relative path.</summary>
        private string dataModelRelativePath = string.Empty;

        /// <summary>Xaml file relative path.</summary>
        private string xamlRelativePath = string.Empty;

        /// <summary>Binary conversion info source file relative path.</summary>
        private string conversionInfoRelativePath = string.Empty;

        /// <summary>
        /// コンストラクタです.
        /// </summary>
        public UserDataDefinition()
        {
        }

        /// <summary>
        /// The base folder path.
        /// </summary>
        public string BaseFolderPath { get; set; }

        /// <summary>
        /// The name of the property page to attach the user data to.
        /// </summary>
        [XmlAttribute("TabName")]
        public string TargetTabName { get; set; }

        /// <summary>
        /// The class name of the user page data model.
        /// </summary>
        [XmlAttribute("PageDataModelName")]
        public string DataModelName { get; set; }

        /// <summary>
        /// Data model source file name.
        /// </summary>
        [XmlAttribute("DataModel")]
        public string DataModelFileName
        {
            get { return PathUtility.GetRootedPath(this.BaseFolderPath, this.dataModelRelativePath); }
            set { this.dataModelRelativePath = PathUtility.LoadEnvironments(value); }
        }

        /// <summary>
        /// Xaml file name.
        /// </summary>
        [XmlAttribute("UI")]
        public string XamlFileName
        {
            get { return PathUtility.GetRootedPath(this.BaseFolderPath, this.xamlRelativePath); }
            set { this.xamlRelativePath = PathUtility.LoadEnvironments(value); }
        }

        /// <summary>
        /// Binary conversion info source file name.
        /// </summary>
        [XmlAttribute("ConversionInfo")]
        public string ConversionInfoFileName
        {
            get { return PathUtility.GetRootedPath(this.BaseFolderPath, this.conversionInfoRelativePath); }
            set { this.conversionInfoRelativePath = PathUtility.LoadEnvironments(value); }
        }

        /// <summary>
        /// Deserializes from the given XML node.
        /// </summary>
        /// <param name="context">The data context needed for the deserialization.</param>
        /// <returns>True on success.</returns>
        public bool ReadXml(XmlDocSerializationContext context)
        {
            this.TargetTabName = this.ReadAttribute(context, "TabName", this.TargetTabName);
            this.DataModelName = this.ReadAttribute(context, "PageDataModelName", this.DataModelName);
            this.DataModelFileName = this.ReadAttribute(context, "DataModel", this.DataModelFileName);
            this.XamlFileName = this.ReadAttribute(context, "UI", this.XamlFileName);
            this.ConversionInfoFileName = this.ReadAttribute(context, "ConversionInfo", this.ConversionInfoFileName);
            return true;
        }

        /// <summary>
        /// Serializes this object to a XML node.
        /// </summary>
        /// <param name="context">The data context needed for the serialization.</param>
        /// <returns>True on success.</returns>
        public bool WriteXml(XmlDocSerializationContext context)
        {
            // This class cannot be serialized.
            return false;
        }
    }
}
