﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;

using EffectMaker.BusinessLogic.IO;

using EffectMaker.Foundation.Log;
using EffectMaker.Foundation.Serialization;
using EffectMaker.Foundation.Utility;

namespace EffectMaker.BusinessLogic.UserData
{
    /// <summary>
    /// Helper class to load the reserved shader definitions.
    /// </summary>
    public class ReservedShaderDefinitionLoader
    {
        /// <summary>The owner name for the custom shader.</summary>
        public const string ReservedShaderOwnerName = "ReservedShader";

        /// <summary>The string to format the reserved shader names.</summary>
        public const string ReservedShaderNamePattern = "ReservedShader_{0}";

        /// <summary>
        /// Constructor.
        /// </summary>
        public ReservedShaderDefinitionLoader()
        {
        }

        /// <summary>
        /// Load the reserved shader definitions from the specified file.
        /// </summary>
        /// <remarks>
        /// This method ONLY loads the definitions.
        /// It DOES NOT compile the data model and binary conversion
        /// info source files, and it DOES NOT load the XAML files.
        /// </remarks>
        /// <param name="filePath">The file path.</param>
        /// <returns>The loaded definitions.</returns>
        public ReservedShaderDefinitionList LoadDefinition(string filePath)
        {
            if (string.IsNullOrEmpty(filePath) == true || File.Exists(filePath) == false)
            {
                return null;
            }

            // Load the definitions.
            ReservedShaderDefinitionList defList =
                SerializationHelper.LoadXmlDocSerializable<ReservedShaderDefinitionList>(filePath);
            if (defList == null)
            {
                return null;
            }

            // Set up the file path, because the paths in the definitions
            // are usually relative paths or even just file names.
            defList.FilePath = filePath;

            // Load reserved shader definitions.
            defList.LoadReservedShaderDefinitionFiles();

            return defList;
        }

        /// <summary>
        /// Set up the reserved shaders.
        /// Compiles the data model and binary conversion info, and saves
        /// the XAML file path for later inquiry.
        /// </summary>
        /// <param name="defList">The custom action definitions.</param>
        /// <returns>True on success.</returns>
        public bool SetupUserData(ReservedShaderDefinitionList defList)
        {
            UserDataManager.ClearUserDataByOwner(ReservedShaderOwnerName);

            if (defList == null)
            {
                ReservedShaderUserDataManager.Definition = null;
                return true;
            }

            int index = 0;
            foreach (ReservedShaderDefinition def in defList.ReservedShaderDefinitions)
            {
                string userDataName = string.Format(ReservedShaderNamePattern, index);

                // Load the user data.
                // The data model and binary conversion info will be compiled to assembly
                // and the XAML file path is saved in the user data manager.
                bool result = UserDataManager.LoadUserData(
                    userDataName,
                    ReservedShaderOwnerName,
                    def.DataModelFilePath,
                    def.BinaryConversionInfoPath,
                    def.XamlFilePath);

                if (result == false)
                {
                    Logger.Log(LogLevels.Warning, "Failed setting up reserved shader {0}.", index);
                    return false;
                }

                // Save the user data info to the definition.
                def.UserDataInfo = UserDataManager.FindUserDataByName(userDataName);

                ++index;
            }

            ReservedShaderUserDataManager.Definition = defList;

            return true;
        }
    }
}
