﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

using EffectMaker.BusinessLogic.IO;
using EffectMaker.DataModel;
using EffectMaker.Foundation.Extensions;
using EffectMaker.Foundation.Log;
using EffectMaker.Foundation.Utility;

namespace EffectMaker.BusinessLogic.UserData
{
    /// <summary>
    /// Manages user data for custom shader.
    /// </summary>
    public static class CustomShaderUserDataManager
    {
        /// <summary>The constant number of custom shader definitions.</summary>
        public const int CustomShaderCount = 8;

        /// <summary>
        /// The default shader types.
        /// </summary>
        private static readonly HashSet<ShaderTypes> DefaultShaderTypes = new HashSet<ShaderTypes>
        {
            ShaderTypes.SpecDeclarationGenericShader,
            ShaderTypes.SpecDeclarationTargetShader,
            ShaderTypes.ParticleDeclarationShader,
            ShaderTypes.ParticleDeclarationVertexShader,
            ShaderTypes.ParticleDeclarationFragmentShader,
            ShaderTypes.StreamOutDeclarationVertexShader,
            ShaderTypes.ParticleVertexShader,
            ShaderTypes.ParticleFragmentShader,
            ShaderTypes.StreamOutVertexShader,
        };

        /// <summary>The shader source files.</summary>
        private static Dictionary<ShaderTypes, ShaderData[]> shaderSourceFiles = null;

        /// <summary>The custom shader definitions.</summary>
        private static CustomShaderDefinitionList definition = null;

        /// <summary>
        /// Static constructor.
        /// </summary>
        static CustomShaderUserDataManager()
        {
            shaderSourceFiles = new Dictionary<ShaderTypes, ShaderData[]>();

            Action setDefaultShaders = () =>
            {
                SetShader(ShaderTypes.SpecDeclarationGenericShader, SpecDefinitions.SpecManager.CurrentSpec.ShaderConversionOption.SpecDeclarationGeneric);
                SetShader(ShaderTypes.SpecDeclarationTargetShader, SpecDefinitions.SpecManager.CurrentSpec.ShaderConversionOption.SpecDeclarationTarget);
                SetShader(ShaderTypes.ParticleDeclarationShader, "eft_ParticleDeclaration.glsl");
                SetShader(ShaderTypes.ParticleDeclarationVertexShader, "eft_ParticleDeclaration.vsh");
                SetShader(ShaderTypes.ParticleDeclarationFragmentShader, "eft_ParticleDeclaration.fsh");
                SetShader(ShaderTypes.StreamOutDeclarationVertexShader, "eft_StreamOutDeclaration.vsh");
                SetShader(ShaderTypes.ParticleVertexShader, "eft_Particle.vsh");
                SetShader(ShaderTypes.ParticleFragmentShader, "eft_Particle.fsh");
                SetShader(ShaderTypes.StreamOutVertexShader, "eft_StreamOut.vsh");
            };

            SpecDefinitions.SpecManager.CurrentSpecChanged += (sender, args) => setDefaultShaders();

            setDefaultShaders();
        }

        /// <summary>
        /// Get or set the custom shader definition list.
        /// </summary>
        public static CustomShaderDefinitionList Definition
        {
            get { return definition; }
            set { SetupDefinition(value); }
        }

        /// <summary>
        /// Get the source code of the specified shader type.
        /// </summary>
        /// <param name="shaderType">The shader type.</param>
        /// <returns>The shader source code.</returns>
        public static string GetShaderSourceCode(ShaderTypes shaderType)
        {
            ShaderData[] dataArray;
            if (shaderSourceFiles.TryGetValue(shaderType, out dataArray) == false ||
                dataArray == null ||
                dataArray.Length <= 0 ||
                dataArray[0] == null)
            {
                return string.Empty;
            }

            return dataArray[0].SourceCode;
        }

        /// <summary>
        /// Get the source code of the specified shader type and index.
        /// </summary>
        /// <param name="shaderType">The shader type.</param>
        /// <param name="index">The index of the shader in the specified shader type.</param>
        /// <returns>The shader source code.</returns>
        public static string GetShaderSourceCode(ShaderTypes shaderType, int index)
        {
            ShaderData[] dataArray;
            if (shaderSourceFiles.TryGetValue(shaderType, out dataArray) == false ||
                dataArray == null ||
                dataArray.Length <= 0)
            {
                return string.Empty;
            }

            if (index < 0 || index >= dataArray.Length)
            {
                return string.Empty;
            }

            ShaderData data = dataArray[index];
            if (data == null)
            {
                return string.Empty;
            }

            return data.SourceCode;
        }

        /// <summary>
        /// Get the all the source code of the specified custom shader type.
        /// </summary>
        /// <param name="shaderType">The shader type.</param>
        /// <returns>The shader source code.</returns>
        public static string[] GetCustomShaderSourceCodeArray(ShaderTypes shaderType)
        {
            if (shaderType != ShaderTypes.CustomVertexShader &&
                shaderType != ShaderTypes.CustomFragmentShader)
            {
                throw new ArgumentException("The shaderType must be either CustomVertexShader or CustomFragmentShader.");
            }

            ShaderData[] dataArray;
            if (shaderSourceFiles.TryGetValue(shaderType, out dataArray) == false)
            {
                return null;
            }

            var array = new string[CustomShaderCount];

            var shaders = dataArray.Select(d => d.SourceCode).ToArray();

            for (int i = 0; i < CustomShaderCount; ++i)
            {
                // Because the shader code for :
                // callback ID 1 => 0
                // callback ID 2 => 1
                // ...
                // callback ID 8 => 7
                // So we have to subtract 1 from i.
                int index = FindCustomShaderIndexByCallbackID(i + 1);
                if (index >= 0)
                {
                    array[i] = shaders[index];
                }
                else
                {
                    array[i] = string.Empty;
                }
            }

            return array;
        }

        /// <summary>
        /// Get the all the source code of the specified reserved shader type.
        /// </summary>
        /// <param name="shaderType">The shader type.</param>
        /// <returns>The shader source code.</returns>
        public static string[] GetReservedShaderSourceCodeArray(ShaderTypes shaderType)
        {
            if (shaderType != ShaderTypes.ReservedVertexShader &&
                shaderType != ShaderTypes.ReservedFragmentShader)
            {
                throw new ArgumentException("The shaderType must be either ReservedVertexShader or ReservedFragmentShader.");
            }

            ShaderData[] dataArray;
            if (shaderSourceFiles.TryGetValue(shaderType, out dataArray) == false)
            {
                return null;
            }

            var array = new string[ReservedShaderUserDataManager.ReservedShaderCount];

            var shaders = dataArray.Select(d => d.SourceCode).ToArray();

            for (int i = 0; i < ReservedShaderUserDataManager.ReservedShaderCount; ++i)
            {
                if (i < shaders.Length)
                {
                    array[i] = shaders[i];
                }
                else
                {
                    array[i] = string.Empty;
                }
            }

            return array;
        }

        /// <summary>
        /// Get the custom shader definition at the specified index.
        /// </summary>
        /// <param name="index">The index to the custom shader definition.</param>
        /// <returns>The custom shader definition or null if doesn't exist.</returns>
        public static CustomShaderDefinition GetCustomShaderDefinition(int index)
        {
            if (Definition == null)
            {
                return null;
            }

            if (index < 0 || index >= Definition.CustomShaderDefinitions.Count)
            {
                return null;
            }

            return Definition.CustomShaderDefinitions[index];
        }

        /// <summary>
        /// Find the index of the custom shader definition with their callback ID.
        /// </summary>
        /// <param name="callbackId">The callback ID of the custom shader definition.</param>
        /// <returns>The custom shader definition index or -1 if not found.</returns>
        public static int FindCustomShaderIndexByCallbackID(int callbackId)
        {
            if (Definition == null)
            {
                return -1;
            }

            for (int i = 0; i < Definition.CustomShaderDefinitions.Count; ++i)
            {
                var def = Definition.CustomShaderDefinitions[i];
                if (def != null && def.Id == callbackId)
                {
                    return i;
                }
            }

            return -1;
        }

        /// <summary>
        /// Find the user data info for the custom shader at the specified index.
        /// </summary>
        /// <param name="index">The custom shader index.</param>
        /// <returns>The user data info or null if the custom shader doesn't exist.</returns>
        public static UserDataInfo FindCustomShaderUserDataInfo(int index)
        {
            var name = string.Format(CustomShaderDefinitionLoader.CustomShaderNamePattern, index);
            return UserDataManager.FindUserDataByName(name);
        }

        /// <summary>
        /// Enumerate all the custom shader user data informations.
        /// </summary>
        /// <returns>all the custom shader user data informations.</returns>
        public static IEnumerable<UserDataInfo> EnumerateCustomShaderUserDataInfo()
        {
            IEnumerable<UserDataInfo> info = UserDataManager.EnumerateUserDataInfoForOwner(CustomShaderDefinitionLoader.CustomShaderOwnerName);

            return info.Concat(Enumerable.Repeat<UserDataInfo>(null, 8)).Take(8);
        }

        /// <summary>
        /// Remove shaders of the specified type.
        /// </summary>
        /// <param name="shaderType">The type of shader to delete.</param>
        internal static void RemoveShader(ShaderTypes shaderType)
        {
            var shaderTypesToDelete = shaderSourceFiles.Keys.Where(
                k => k == shaderType).ToArray();

            // Now remove these shader types.
            foreach (ShaderTypes currShaderType in shaderTypesToDelete)
            {
                shaderSourceFiles.Remove(currShaderType);
            }
        }

        /// <summary>
        /// Helper method to set shader source file to the specified shader type.
        /// </summary>
        /// <param name="shaderType">The shader type.</param>
        /// <param name="shaderFileName">The file name of the shader.</param>
        /// <param name="baseFolderPath">The base folder path.</param>
        internal static void SetShader(
            ShaderTypes shaderType,
            string shaderFileName,
            string baseFolderPath = null)
        {
            if (string.IsNullOrEmpty(baseFolderPath) == true)
            {
                baseFolderPath = IOConstants.DefaultShaderFolderPath;
            }

            // Create one ShaderData with the shader source file.
            var shaderPath = PathUtility.GetRootedPath(baseFolderPath, shaderFileName);
            var shaderData = new ShaderData(shaderPath);

            shaderSourceFiles[shaderType] = new ShaderData[] { shaderData };
        }

        /// <summary>
        /// Helper method to set shader source file to the specified shader type.
        /// </summary>
        /// <param name="shaderType">The shader type.</param>
        /// <param name="shaderFileNames">The file name of the shader files.</param>
        /// <param name="baseFolderPath">The base folder path.</param>
        internal static void SetShader(
            ShaderTypes shaderType,
            IEnumerable<string> shaderFileNames,
            string baseFolderPath = null)
        {
            if (string.IsNullOrEmpty(baseFolderPath) == true)
            {
                baseFolderPath = IOConstants.DefaultShaderFolderPath;
            }

            // Create ONE ShaderData contains all the shader source files.
            var shaderData = new ShaderData(shaderFileNames.Select(
                n => PathUtility.GetRootedPath(baseFolderPath, n)));

            shaderSourceFiles[shaderType] = new ShaderData[] { shaderData };
        }

        /// <summary>
        /// Helper method to set shader source file to the specified shader type.
        /// </summary>
        /// <param name="shaderType">The shader type.</param>
        /// <param name="shaderFileNameGroups">Groups of shader file names.</param>
        /// <param name="baseFolderPath">The base folder path.</param>
        internal static void SetShader(
            ShaderTypes shaderType,
            IEnumerable<IEnumerable<string>> shaderFileNameGroups,
            string baseFolderPath = null)
        {
            if (string.IsNullOrEmpty(baseFolderPath) == true)
            {
                baseFolderPath = IOConstants.DefaultShaderFolderPath;
            }

            // Create one shader data for each group of shader file names.
            // E.g.
            // Consider shaderFileNameGroups = { {shaderA, shaderB, shaderC}, {shaderD, shaderE} }
            // will create 2 ShaderData instances.
            // The first one contains {shaderA, shaderB, shaderC},
            // the second contains {shaderD, shaderE}
            var shaderPathGroups = shaderFileNameGroups.Select(
                g => new ShaderData(g.Select(n => PathUtility.GetRootedPath(baseFolderPath, n))));

            shaderSourceFiles[shaderType] = shaderPathGroups.ToArray();
        }

        /// <summary>
        /// Helper method for setting up the custom shader definition.
        /// </summary>
        /// <param name="def">The custom shader definitions.</param>
        private static void SetupDefinition(CustomShaderDefinitionList def)
        {
            definition = def;

            if (def == null)
            {
                // We have to select these shader types and save them to another array first,
                // otherwise InvalidOperationException would occur due to modifying the source
                // while enumerating it.
                var shaderTypesToDelete = shaderSourceFiles.Keys.Where(
                    k => DefaultShaderTypes.Contains(k) == false).ToArray();

                // Now remove these shader types.
                foreach (ShaderTypes shaderType in shaderTypesToDelete)
                {
                    shaderSourceFiles.Remove(shaderType);
                }
            }
            else
            {
                // General vertex shaders.
                SetShader(
                    ShaderTypes.GeneralVertexShader,
                    def.GeneralShaderDefinition.VertexShaderFullPaths);

                // General fragment shaders.
                SetShader(
                    ShaderTypes.GeneralFragmentShader,
                    def.GeneralShaderDefinition.FragmentShaderFullPaths);

                // General compute shaders.
                SetShader(
                    ShaderTypes.GeneralComputeShader,
                    def.GeneralShaderDefinition.ComputeShaderFullPaths);

                // Custom vertex shaders.
                SetShader(
                    ShaderTypes.CustomVertexShader,
                    def.CustomShaderDefinitions.Select(d => Enumerable.Concat(
                        def.GeneralShaderDefinition.VertexShaderFullPaths, d.VertexShaderFullPaths)));

                // Custom fragment shaders.
                SetShader(
                    ShaderTypes.CustomFragmentShader,
                    def.CustomShaderDefinitions.Select(d => Enumerable.Concat(
                        def.GeneralShaderDefinition.FragmentShaderFullPaths, d.FragmentShaderFullPaths)));
            }

            ReservedShaderUserDataManager.Definition = ReservedShaderUserDataManager.Definition;
        }

        /// <summary>
        /// Data class that stores the data of a shader, which may be composed by
        /// more than one shader source files.
        /// </summary>
        private class ShaderData
        {
            /// <summary>The information of all the shader source files.</summary>
            private List<ShaderFileInfo> sourceFileInfoList = null;

            /// <summary>The composed shader source code.</summary>
            private string composedSourceCode = string.Empty;

            /// <summary>
            /// Default constructor.
            /// </summary>
            public ShaderData()
            {
            }

            /// <summary>
            /// Constructor.
            /// </summary>
            /// <param name="sourceFilePath">The shader source file path.</param>
            public ShaderData(string sourceFilePath)
            {
                this.SetSourceFilePaths(new string[] { sourceFilePath });
            }

            /// <summary>
            /// Constructor.
            /// </summary>
            /// <param name="sourceFilePaths">The paths of the shader source files.</param>
            public ShaderData(IEnumerable<string> sourceFilePaths)
            {
                this.SetSourceFilePaths(sourceFilePaths);
            }

            /// <summary>
            /// Get or set the composed source code of all the source files.
            /// </summary>
            public string SourceCode
            {
                get
                {
                    this.UpdateSourceCode();
                    return this.composedSourceCode;
                }

                set
                {
                    // If the source code is set from outside the class,
                    // clear the source file paths.
                    this.sourceFileInfoList = null;
                }
            }

            /// <summary>
            /// Set the source file paths for this shader.
            /// </summary>
            /// <param name="filePaths">The file paths.</param>
            public void SetSourceFilePaths(IEnumerable<string> filePaths)
            {
                this.sourceFileInfoList = filePaths.Select(p =>
                    {
                        return new ShaderFileInfo()
                        {
                            FilePath = p,
                            LastModifiedTime = DateTime.MinValue,
                            SourceCode = string.Empty,
                        };
                    }).ToList();
            }

            /// <summary>
            /// Update source code from shader files.
            /// </summary>
            /// <returns>True on success.</returns>
            private bool UpdateSourceCode()
            {
                if (this.sourceFileInfoList == null ||
                    this.sourceFileInfoList.Count <= 0)
                {
                    return true;
                }

                bool isAnyFileModified = false;
                foreach (var info in this.sourceFileInfoList)
                {
                    // Does the shader file exist?
                    if (File.Exists(info.FilePath) == false)
                    {
                        this.composedSourceCode = string.Empty;

                        info.SourceCode = string.Empty;
                        info.LastModifiedTime = DateTime.MinValue;

                        Logger.Log(
                            LogLevels.Warning,
                            "ShaderData.UpdateSourceCode : Unable to find shader source file {0}",
                            info.FilePath);

                        return false;
                    }

                    // Check if the source file is modified since loaded.
                    var lastModifiedTime = File.GetLastWriteTime(info.FilePath);
                    if (DateTime.Compare(lastModifiedTime, info.LastModifiedTime) != 0)
                    {
                        // Set the flag to update the composed source code
                        // after we have checked all the source files.
                        isAnyFileModified = true;

                        try
                        {
                            // Load the source code in the shader file.
                            info.SourceCode = File.ReadAllText(info.FilePath);
                            info.LastModifiedTime = lastModifiedTime;
                        }
                        catch (Exception ex)
                        {
                            // Failed loading the shader file.
                            this.composedSourceCode = string.Empty;

                            info.SourceCode = string.Empty;
                            info.LastModifiedTime = DateTime.MinValue;

                            Logger.Log(
                                LogLevels.Warning,
                                "ShaderData.UpdateSourceCode : Failed loading shader source file {0}",
                                info.FilePath);

                            Logger.Log(LogLevels.Warning, "Exception : {0}", ex.Message);

                            return false;
                        }
                    }
                }

                if (isAnyFileModified == true)
                {
                    var builder = new StringBuilder();
                    foreach (var info in this.sourceFileInfoList)
                    {
                        builder.Append(info.SourceCode);
                        builder.Append(Environment.NewLine);
                    }

                    this.composedSourceCode = builder.ToString();
                }

                return true;
            }

            /// <summary>
            /// Data class that holds information about a shader source file.
            /// </summary>
            private class ShaderFileInfo
            {
                /// <summary>
                /// Get or set the shader source file path.
                /// </summary>
                public string FilePath { get; set; }

                /// <summary>
                /// Get or set the last modified time of the shader source file.
                /// </summary>
                public DateTime LastModifiedTime { get; set; }

                /// <summary>
                /// Get or set the contents of the shader source file.
                /// </summary>
                public string SourceCode { get; set; }
            }
        }
    }
}
