﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;

using EffectMaker.BusinessLogic.IO;

using EffectMaker.Foundation.Log;
using EffectMaker.Foundation.Serialization;
using EffectMaker.Foundation.Utility;

namespace EffectMaker.BusinessLogic.UserData
{
    /// <summary>
    /// Helper class to load the custom action definitions.
    /// </summary>
    public class CustomActionDefinitionLoader
    {
        /// <summary>The owner name for the custom actions.</summary>
        public const string CustomActionOwnerName = "CustomAction";

        /// <summary>The string to format the custom action names.</summary>
        public const string CustomActionNamePattern = "CustomAction_{0}";

        /// <summary>
        /// Constructor.
        /// </summary>
        public CustomActionDefinitionLoader()
        {
        }

        /// <summary>
        /// Load the custom action definitions from the specified file.
        /// </summary>
        /// <remarks>
        /// This method ONLY loads the definitions.
        /// It DOES NOT compile the data model and binary conversion
        /// info source files, and it DOES NOT load the XAML files.
        /// </remarks>
        /// <param name="filePath">The file path.</param>
        /// <returns>The loaded definitions.</returns>
        public CustomActionDefinitionList LoadDefinition(string filePath)
        {
            if (string.IsNullOrEmpty(filePath) == true || File.Exists(filePath) == false)
            {
                return null;
            }

            // Load the definitions.
            CustomActionDefinitionList defList =
                SerializationHelper.LoadXmlDocSerializable<CustomActionDefinitionList>(filePath);
            if (defList == null)
            {
                return null;
            }

            // Set up the base folder path, because the paths in the definitions
            // are usually relative paths or even just file names.
            defList.BaseFolderPath = Path.GetDirectoryName(filePath);

            return defList;
        }

        /// <summary>
        /// Set up the custom actions.
        /// Compiles the data model and binary conversion info, and saves
        /// the XAML file path for later inquiry.
        /// </summary>
        /// <param name="defList">The custom action definitions.</param>
        /// <returns>True on success.</returns>
        public bool SetupUserData(CustomActionDefinitionList defList)
        {
            UserDataManager.ClearUserDataByOwner(CustomActionOwnerName);

            if (defList == null)
            {
                CustomActionUserDataManager.Definition = null;
                return true;
            }

            int index = 0;
            foreach (CustomActionDefinition def in defList.CustomActionDefinitions)
            {
                if (def == null)
                {
                    UserDataManager.AddEmptyUserDataToOwner(CustomActionOwnerName);
                }
                else if (def.UsingUdd2)
                {
                    // Load the user data.
                    bool result = UserDataManager.LoadUdd2(
                        string.Format(CustomActionNamePattern, index),
                        CustomActionOwnerName,
                        def.Udd2FullFilePath);

                    if (result == false)
                    {
                        Logger.Log(LogLevels.Warning, "Failed setting up custom action {0}.", index);
                        return false;
                    }
                }
                else
                {
                    // Load the user data.
                    // The data model and binary conversion info will be compiled to assembly
                    // and the XAML file path is saved in the user data manager.
                    bool result = UserDataManager.LoadUserData(
                        string.Format(CustomActionNamePattern, index),
                        CustomActionOwnerName,
                        def.DataModelFullFilePath,
                        def.BinaryConversionInfoFullPath,
                        def.XamlFullFilePath);

                    if (result == false)
                    {
                        Logger.Log(LogLevels.Warning, "Failed setting up custom action {0}.", index);
                        return false;
                    }
                }

                ++index;
                if (index >= 8)
                {
                    // There can only be 8 custom action definitions at most.
                    break;
                }
            }

            CustomActionUserDataManager.Definition = defList;

            return true;
        }

        /// <summary>
        /// Collect obsolete user data (e.g. .udd definitions) for conversion.
        /// </summary>
        /// <param name="defList">The custom action definitions.</param>
        /// <returns>The collected information.</returns>
        public List<ObsoleteUserDataConversionInfo> CollectObsoleteUserDataToConvert(CustomActionDefinitionList defList)
        {
            var infos = new List<ObsoleteUserDataConversionInfo>();

            if (defList == null)
            {
                return infos;
            }

            string language = "jp";

            var culture = System.Threading.Thread.CurrentThread.CurrentUICulture;
            var cultureEnUs = System.Globalization.CultureInfo.CreateSpecificCulture("en-US");
            if (culture.Equals(cultureEnUs) == true)
            {
                language = "en";
            }

            foreach (CustomActionUddDefinition info in defList.UddDefinitions)
            {
                var convertionInfo = new ObsoleteUserDataConversionInfo()
                {
                    FilePath = info.UddFilePath,
                    DataModelName = info.DataModelName,
                    ConversionCommandName = "udd",
                    Language = language,
                };

                infos.Add(convertionInfo);
            }

            return infos;
        }

        /// <summary>
        /// Prepare the converted .udd file definitions for use.
        /// </summary>
        /// <param name="defList">The custom action definitions.</param>
        public void PrepareConvertedObsoleteUserData(CustomActionDefinitionList defList)
        {
            if (defList == null)
            {
                return;
            }

            foreach (CustomActionUddDefinition info in defList.UddDefinitions)
            {
                string outputPath = UserDataManager.GetConvertedObsoleteUserDataOutputFolder(
                    info.UddFilePath,
                    info.DataModelName);

                info.DataModelFileName = Path.Combine(outputPath, info.DataModelName + ".cs");

                info.XamlFileName = Path.Combine(outputPath, info.DataModelName + ".xaml");

                info.BinaryConversionInfoPath = Path.Combine(
                    outputPath,
                    info.DataModelName + "BinaryConversionInfo.cs");
            }
        }
    }
}
