﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Xml.Serialization;

using EffectMaker.Foundation.Serialization;

namespace EffectMaker.BusinessLogic.Options
{
    /// <summary>
    /// Viewer options.
    /// </summary>
    public class ViewerOptions : OptionsBase
    {
        /// <summary>
        /// ウィンドウサイズを表す文字列をインデックス化する辞書
        /// </summary>
        public static readonly Dictionary<string, int> WindowSizeDictionary = new Dictionary<string, int>();

        /// <summary>
        /// FPSを表す文字列をインデックス化する辞書
        /// </summary>
        public static readonly Dictionary<string, int> FpsDictionary = new Dictionary<string, int>();

        /// <summary>
        /// Initializes the ViewerOptions instance.
        /// </summary>
        public ViewerOptions()
        {
            WindowSizeDictionary["1280*720"] = 0;
            WindowSizeDictionary["640*480"] = 1;
            WindowSizeDictionary["960*540"] = 2;

            FpsDictionary["60fps"] = 0;
            FpsDictionary["30fps"] = 1;
            FpsDictionary["20fps"] = 2;
            FpsDictionary["15fps"] = 3;

            this.AutoConnect = true;

            this.WindowSize = "960*540";
            this.FrameRate = "60fps";

            this.SpecTarget = string.Empty;
            this.ConnectionTarget = 0;
            this.LoopMode = false;
            this.LoopFrame = 100;
        }

        /// <summary>
        /// Copy the ViewerOptions instance.
        /// </summary>
        /// <param name="src">The source optins to copy from.</param>
        public ViewerOptions(ViewerOptions src)
            : this()
        {
            this.Set(src);
        }

        /// <summary>
        /// Gets or sets the flag telling wether EffectMaker
        /// automatically connects to the viewer or not.
        /// </summary>
        [XmlAttribute]
        public bool AutoConnect { get; set; }

        /// <summary>
        /// Gets or sets the frame rate of the viewer.
        /// </summary>
        [XmlAttribute]
        public string FrameRate { get; set; }

        /// <summary>
        /// Gets or sets the viewer window size.
        /// </summary>
        [XmlAttribute]
        public string WindowSize { get; set; }

        /// <summary>
        /// スペック(プラットフォーム)
        /// </summary>
        [XmlAttribute]
        public string SpecTarget { get; set; }

        /// <summary>
        /// 接続先(0:PC/1:実機)
        /// </summary>
        [XmlAttribute]
        public uint ConnectionTarget { get; set; }

        /// <summary>
        /// ループモード
        /// </summary>
        [XmlAttribute]
        public bool LoopMode { get; set; }

        /// <summary>
        /// ループフレーム
        /// </summary>
        [XmlAttribute]
        public int LoopFrame { get; set; }

        /// <summary>
        /// Set data from the source options.
        /// </summary>
        /// <param name="src">The source options.</param>
        /// <returns>True on success.</returns>
        public override bool Set(object src)
        {
            var options = src as ViewerOptions;
            if (options == null)
            {
                return false;
            }

            this.AutoConnect = options.AutoConnect;
            this.FrameRate = options.FrameRate;
            this.WindowSize = options.WindowSize;
            this.SpecTarget = options.SpecTarget;
            this.ConnectionTarget = options.ConnectionTarget;
            this.LoopMode = options.LoopMode;
            this.LoopFrame = options.LoopFrame;

            return true;
        }

        /// <summary>
        /// Deserializes from the given XML node.
        /// </summary>
        /// <param name="context">The data context needed for the deserialization.</param>
        /// <returns>True on success.</returns>
        public override bool ReadXml(XmlDocSerializationContext context)
        {
            this.AutoConnect = this.ReadAttribute(context, "AutoConnect", this.AutoConnect);
            this.FrameRate = this.ReadAttribute(context, "FrameRate", this.FrameRate);
            this.WindowSize = this.ReadAttribute(context, "WindowSize", this.WindowSize);
            this.SpecTarget = this.ReadAttribute(context, "SpecTarget", this.SpecTarget);
            this.ConnectionTarget = this.ReadAttribute(context, "ConnectionTarget", this.ConnectionTarget);
            this.LoopMode = this.ReadAttribute(context, "LoopMode", this.LoopMode);
            this.LoopFrame = this.ReadAttribute(context, "LoopFrame", this.LoopFrame);

            return true;
        }

        /// <summary>
        /// Serializes this object to a XML node.
        /// </summary>
        /// <param name="context">The data context needed for the serialization.</param>
        /// <returns>True on success.</returns>
        public override bool WriteXml(XmlDocSerializationContext context)
        {
            this.WriteAttribute(context, "AutoConnect", this.AutoConnect);
            this.WriteAttribute(context, "FrameRate", this.FrameRate);
            this.WriteAttribute(context, "WindowSize", this.WindowSize);
            this.WriteAttribute(context, "SpecTarget", this.SpecTarget);
            this.WriteAttribute(context, "ConnectionTarget", this.ConnectionTarget);
            this.WriteAttribute(context, "LoopMode", this.LoopMode);
            this.WriteAttribute(context, "LoopFrame", this.LoopFrame);

            return true;
        }
    }
}
