﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Xml.Serialization;

using EffectMaker.Foundation.Serialization;

namespace EffectMaker.BusinessLogic.Options
{
    /// <summary>
    /// Root options.
    /// </summary>
    [XmlRoot(ElementName = "Options")]
    public class RootOptions : OptionsBase
    {
        /// <summary>
        /// Initializes the RootOptions instance.
        /// </summary>
        public RootOptions()
        {
            this.Basic = new BasicOptions();
            this.Interface = new InterfaceOptions();
            this.FileEvent = new FileEventOptions();
            this.WorldScale = new WorldScaleOptions();
            this.LinearWorkflow = new LinearWorkflowOptions();
            this.Viewer = new ViewerOptions();
            this.EffectBrowser = new EffectBrowserOptions();
            this.Details = new DetailsOptions();
        }

        /// <summary>
        /// Copy the RootOptions instance.
        /// </summary>
        /// <param name="src">The source optins to copy from.</param>
        public RootOptions(RootOptions src)
            : this()
        {
            this.Set(src);
        }

        /// <summary>
        /// Gets or sets the Basic options.
        /// </summary>
        public BasicOptions Basic { get; set; }

        /// <summary>
        /// Gets or sets the interface options.
        /// </summary>
        public InterfaceOptions Interface { get; set; }

        /// <summary>
        /// Gets or sets the file paths options.
        /// </summary>
        public FileEventOptions FileEvent { get; set; }

        /// <summary>
        /// Gets or sets the world scale options.
        /// </summary>
        public WorldScaleOptions WorldScale { get; set; }

        /// <summary>
        /// Gets or sets the linear workflow options.
        /// </summary>
        public LinearWorkflowOptions LinearWorkflow { get; set; }

        /// <summary>
        /// Gets or sets the viewer options.
        /// </summary>
        public ViewerOptions Viewer { get; set; }

        /// <summary>
        /// Gets or sets the Effect Browser options.
        /// </summary>
        public EffectBrowserOptions EffectBrowser { get; set; }

        /// <summary>
        /// Gets or sets the details options.
        /// </summary>
        public DetailsOptions Details { get; set; }

        /// <summary>
        /// Set data from the source options.
        /// </summary>
        /// <param name="src">The source options.</param>
        /// <returns>True on success.</returns>
        public override bool Set(object src)
        {
            var options = src as RootOptions;
            if (options == null)
            {
                return false;
            }

            this.Basic.Set(options.Basic);
            this.Interface.Set(options.Interface);
            this.FileEvent.Set(options.FileEvent);
            this.WorldScale.Set(options.WorldScale);
            this.LinearWorkflow.Set(options.LinearWorkflow);
            this.Viewer.Set(options.Viewer);
            this.EffectBrowser.Set(options.EffectBrowser);
            this.Details.Set(options.Details);

            return true;
        }

        /// <summary>
        /// Deserializes from the given XML node.
        /// </summary>
        /// <param name="context">The data context needed for the deserialization.</param>
        /// <returns>True on success.</returns>
        public override bool ReadXml(XmlDocSerializationContext context)
        {
            this.Basic = this.ReadElement(context, "Basic", this.Basic);
            this.Interface = this.ReadElement(context, "Interface", this.Interface);
            this.FileEvent = this.ReadElement(context, "FilePaths", this.FileEvent);
            this.WorldScale = this.ReadElement(context, "WorldScale", this.WorldScale);
            this.LinearWorkflow = this.ReadElement(context, "LinearWorkflow", this.LinearWorkflow);
            this.Viewer = this.ReadElement(context, "Viewer", this.Viewer);
            this.EffectBrowser = this.ReadElement(context, "EffectBrowser", this.EffectBrowser);
            this.Details = this.ReadElement(context, "Details", this.Details);

            return true;
        }

        /// <summary>
        /// Serializes this object to a XML node.
        /// </summary>
        /// <param name="context">The data context needed for the serialization.</param>
        /// <returns>True on success.</returns>
        public override bool WriteXml(XmlDocSerializationContext context)
        {
            this.WriteElement(context, "Basic", this.Basic);
            this.WriteElement(context, "Interface", this.Interface);
            this.WriteElement(context, "FilePaths", this.FileEvent);
            this.WriteElement(context, "WorldScale", this.WorldScale);
            this.WriteElement(context, "LinearWorkflow", this.LinearWorkflow);
            this.WriteElement(context, "Viewer", this.Viewer);
            this.WriteElement(context, "EffectBrowser", this.EffectBrowser);
            this.WriteElement(context, "Details", this.Details);

            return true;
        }
    }
}
