﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Xml.Serialization;

using EffectMaker.Foundation.Serialization;

namespace EffectMaker.BusinessLogic.Options
{
    /// <summary>
    /// Details options.
    /// </summary>
    public class DetailsOptions : OptionsBase
    {
        /// <summary>
        /// Initializes the DetailsOptions instance.
        /// </summary>
        public DetailsOptions()
        {
            this.MaximumTextureCacheSize = 256;

            this.IsAutoBackupActive = true;
            this.BackupInterval = 5;
            this.DeleteOldBackupFilesAtStartup = true;
            this.DeleteDaysLater = 3;
        }

        /// <summary>
        /// Copy the DetailsOptions instance.
        /// </summary>
        /// <param name="src">The source optins to copy from.</param>
        public DetailsOptions(DetailsOptions src)
            : this()
        {
            this.Set(src);
        }

        /// <summary>
        /// Gets or sets the maximum texture cache size, in MB.
        /// </summary>
        [XmlAttribute]
        public int MaximumTextureCacheSize { get; set; }

        /// <summary>
        /// Gets or sets a flag telling whether the auto backup is on or off.
        /// </summary>
        [XmlAttribute]
        public bool IsAutoBackupActive { get; set; }

        /// <summary>
        /// Gets or sets the auto backup interval, in minutes.
        /// </summary>
        [XmlAttribute]
        public int BackupInterval { get; set; }

        /// <summary>
        /// Gets or sets the flag telling whether to delete old backup file at startup.
        /// </summary>
        [XmlAttribute]
        public bool DeleteOldBackupFilesAtStartup { get; set; }

        /// <summary>
        /// Gets or sets the number of days to delay old backup files deletion.
        /// </summary>
        [XmlAttribute]
        public int DeleteDaysLater { get; set; }

        /// <summary>
        /// Set data from the source options.
        /// </summary>
        /// <param name="src">The source options.</param>
        /// <returns>True on success.</returns>
        public override bool Set(object src)
        {
            var options = src as DetailsOptions;
            if (options == null)
            {
                return false;
            }

            this.MaximumTextureCacheSize = options.MaximumTextureCacheSize;
            this.IsAutoBackupActive = options.IsAutoBackupActive;
            this.BackupInterval = options.BackupInterval;
            this.DeleteOldBackupFilesAtStartup = options.DeleteOldBackupFilesAtStartup;
            this.DeleteDaysLater = options.DeleteDaysLater;

            return true;
        }

        /// <summary>
        /// Deserializes from the given XML node.
        /// </summary>
        /// <param name="context">The data context needed for the deserialization.</param>
        /// <returns>True on success.</returns>
        public override bool ReadXml(XmlDocSerializationContext context)
        {
            this.MaximumTextureCacheSize = this.ReadAttribute(
                context,
                "MaximumTextureCacheSize",
                this.MaximumTextureCacheSize);

            this.IsAutoBackupActive = this.ReadAttribute(
                context,
                "IsAutoBackupActive",
                this.IsAutoBackupActive);

            this.BackupInterval = this.ReadAttribute(
                context,
                "BackupInterval",
                this.BackupInterval);

            this.DeleteOldBackupFilesAtStartup = this.ReadAttribute(
                context,
                "DeleteOldBackupFilesAtStartup",
                this.DeleteOldBackupFilesAtStartup);

            this.DeleteDaysLater = this.ReadAttribute(
                context,
                "DeleteDaysLater",
                this.DeleteDaysLater);

            return true;
        }

        /// <summary>
        /// Serializes this object to a XML node.
        /// </summary>
        /// <param name="context">The data context needed for the serialization.</param>
        /// <returns>True on success.</returns>
        public override bool WriteXml(XmlDocSerializationContext context)
        {
            this.WriteAttribute(context, "MaximumTextureCacheSize", this.MaximumTextureCacheSize);
            this.WriteAttribute(context, "IsAutoBackupActive", this.IsAutoBackupActive);
            this.WriteAttribute(context, "BackupInterval", this.BackupInterval);
            this.WriteAttribute(context, "DeleteOldBackupFilesAtStartup", this.DeleteOldBackupFilesAtStartup);
            this.WriteAttribute(context, "DeleteDaysLater", this.DeleteDaysLater);

            return true;
        }
    }
}
