﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows.Forms;
using EffectMaker.BusinessLogic.Properties;
using EffectMaker.Foundation.Interfaces;

namespace EffectMaker.BusinessLogic.HookCommand
{
    /// <summary>
    /// The event command.
    /// </summary>
    public enum HookCommand
    {
        /// <summary>
        /// The pre save.
        /// </summary>
        PreSave,

        /// <summary>
        /// The post save.
        /// </summary>
        PostSave,

        /// <summary>
        /// The pre delete.
        /// </summary>
        PreDelete,

        /// <summary>
        /// The post delete.
        /// </summary>
        PostDelete,

        /// <summary>
        /// The pre open.
        /// </summary>
        PreOpen,
    }

    /// <summary>
    /// The hook command executer.
    /// </summary>
    public static class HookCommandExecuter
    {
        /// <summary>
        /// The execute.
        /// </summary>
        /// <param name="eventType">
        /// The event Type.
        /// </param>
        /// <param name="targetList">
        /// The target List.
        /// </param>
        /// <param name="retMsgs">
        /// エラーが発生した場合はダイアログに表示するためのエラーメッセージを返します。
        /// </param>
        /// <returns>
        /// The <see cref="bool"/>.
        /// </returns>
        public static int Execute(HookCommand eventType, List<string> targetList, out string[] retMsgs)
        {
            retMsgs = null;

            if (targetList.Count == 0)
            {
                return 0;
            }

            var filePaths = Options.OptionStore.RuntimeOptions.FileEventMerge;
            var commandPath = string.Empty;
            switch (eventType)
            {
                case HookCommand.PreSave:
                    commandPath = filePaths.PreSaveCommandFilePath;
                    break;
                case HookCommand.PostSave:
                    commandPath = filePaths.PostSaveCommandFilePath;
                    break;
                case HookCommand.PreDelete:
                    commandPath = filePaths.PreDeleteCommandFilePath;
                    break;
                case HookCommand.PostDelete:
                    commandPath = filePaths.PostDeleteCommandFilePath;
                    break;
                case HookCommand.PreOpen:
                    commandPath = filePaths.PreOpenCommandFilePath;
                    break;
                default:
                    Debug.Assert(false, "Undefined event command.");
                    break;
            }

            if (string.IsNullOrEmpty(commandPath))
            {
                return 0;
            }

            if (commandPath.Contains("%"))
            {
                commandPath = Environment.ExpandEnvironmentVariables(commandPath);
            }

            // 指定されたスクリプトが存在しない。
            if (!File.Exists(commandPath))
            {
                // 通知して中断
                retMsgs = new string[3];
                retMsgs[0] = string.Format(Resources.ErrorHookCommandNotFound, commandPath);
                retMsgs[1] = Resources.ErrorHookCommandCaption;
                retMsgs[2] = string.Format(Resources.ErrorHookCommandNotFoundLog, commandPath);

                return 1;
            }

            var listFilePath = MakeTempList(targetList);

            var startInfo = new ProcessStartInfo
            {
                FileName = System.Environment.GetEnvironmentVariable("ComSpec"),
                CreateNoWindow = true,
                UseShellExecute = false,
                Arguments = string.Format(@"/c " + commandPath + " {0}", listFilePath),
                RedirectStandardError = true,
                RedirectStandardOutput = true
            };

            Process process;
            string errMsg;
            try
            {
                process = Process.Start(startInfo);
                errMsg = process.StandardError.ReadToEnd();
                process.WaitForExit();
            }
            catch (Exception e)
            {
                // 通知して中断
                retMsgs = new string[3];
                retMsgs[0] = Resources.ErrorHookCommandException + "\r\n" + e.ToString();
                retMsgs[1] = Resources.ErrorHookCommandCaption;
                retMsgs[2] = Resources.ErrorHookCommandException;

                return 1;
            }

            File.Delete(listFilePath);

            if (process.ExitCode != 0)
            {
                // 通知して中断
                retMsgs = new string[3];
                retMsgs[0] = Resources.ErrorHookCommandFailed + "\r\n" + errMsg;
                retMsgs[1] = Resources.ErrorHookCommandCaption;
                retMsgs[2] = Resources.ErrorHookCommandFailed;
            }

            return process.ExitCode;
        }

        /// <summary>
        /// The make temp list.
        /// </summary>
        /// <param name="targetList">
        /// The target list.
        /// </param>
        /// <returns>
        /// The <see cref="string"/>.
        /// </returns>
        private static string MakeTempList(IEnumerable<string> targetList)
        {
            var tempFile = System.IO.Path.GetTempFileName();

            var sw = new System.IO.StreamWriter(
                tempFile,
                false,
                new System.Text.UTF8Encoding(false));

            foreach (var path in targetList)
            {
                sw.WriteLine(path);
            }

            sw.Close();

            return tempFile;
        }
    }
}
