﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Text;
using EffectMaker.BusinessLogic.IO;
using EffectMaker.BusinessLogic.Options;
using EffectMaker.Foundation.Log;
using EffectMaker.Foundation.Utility;

namespace EffectMaker.BusinessLogic.GfxToolsUtility
{
    /// <summary>
    /// Gfx関連のツールを実行するためのユーティリティです。
    /// </summary>
    public static class GfxToolExecutionUtility
    {
        /// <summary>
        /// exeファイルを実行します。
        /// </summary>
        /// <param name="exeFilePath">exeファイルのパス</param>
        /// <param name="workingDirectory">作業ディレクトリ</param>
        /// <param name="arguments">exeファイルに渡す引数</param>
        /// <returns>実行結果を返します。</returns>
        public static ExecutionResult ExecuteExeFile(string exeFilePath, string workingDirectory, string arguments)
        {
            // 実行オプションをいろいろ設定
            ProcessStartInfo startInfo = new ProcessStartInfo()
            {
                FileName               = exeFilePath,
                WorkingDirectory       = workingDirectory,
                Arguments              = arguments,
                CreateNoWindow         = true,
                UseShellExecute        = false,
                RedirectStandardOutput = true,
                RedirectStandardError  = true
            };

            StringBuilder standardOutputLog = new StringBuilder();
            StringBuilder standardErrorLog  = new StringBuilder();

            Process process = new Process();

            process.StartInfo          = startInfo;
            process.OutputDataReceived += (s, e) => standardOutputLog.AppendLine(e.Data);
            process.ErrorDataReceived  += (s, e) => standardErrorLog.AppendLine(e.Data);

            // 実行！
            process.Start();

            process.BeginOutputReadLine();
            process.BeginErrorReadLine();

            process.WaitForExit();

            // 結果を取得
            ExecutionResult result = new ExecutionResult()
            {
                ResultCode        = process.ExitCode,
                StandardOutputLog = standardOutputLog.ToString(),
                StandardErrorLog  = standardErrorLog.ToString()
            };

            return result;
        }

        /// <summary>
        /// エラーログをファイルに保存してテキストエディタで開きます。
        /// </summary>
        /// <param name="path">ファイルパス</param>
        /// <param name="errorLog">エラーログ</param>
        public static bool SaveAndShowErrorMessage(string path, string errorLog)
        {
            // エラーログをファイルに保存
            IOUtility.WriteAllTextEx(path, errorLog, Encoding.UTF8, 10, 30);

            // ファイルをテキストエディタで開く
            if (OptionStore.RuntimeOptions.IsCommandLineMode == false)
            {
                ApplicationIOManager.OpenTextFileWithExternalTextEditor(path);
            }

            return true;
        }

        /// <summary>
        /// exeファイルの実行結果です。
        /// </summary>
        public class ExecutionResult
        {
            /// <summary>
            /// リザルトコードを取得または設定します。
            /// </summary>
            public int ResultCode { get; set; }

            /// <summary>
            /// 標準出力のテキストを取得または設定します。
            /// </summary>
            public string StandardOutputLog { get; set; }

            /// <summary>
            /// 標準エラーのログを取得または設定します。
            /// </summary>
            public string StandardErrorLog { get; set; }
        }
    }
}
