﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Text;
using EffectMaker.BusinessLogic.BinaryHeaders;
using EffectMaker.BusinessLogic.GfxToolsUtility;
using EffectMaker.DataModel.DataModels;
using EffectMaker.DataModel.Specific.DataModels;
using EffectMaker.Foundation.Log;
using EffectMaker.Foundation.Texture;
using EffectMaker.Foundation.Utility;
using EffectMaker.BusinessLogic.IO;
using EffectMaker.BusinessLogic.Options;
using EffectMaker.BusinessLogic.SpecDefinitions;

namespace EffectMaker.BusinessLogic.BinaryResourceWriters.Texture
{
    /// <summary>
    /// GFX版のテクスチャテーブルライタです。
    /// </summary>
    public class GfxTextureTableWriter : IBinaryResourceWriter
    {
        /// <summary>
        /// Constructor.
        /// </summary>
        /// <param name="emitterSets">The emitterSet</param>
        public GfxTextureTableWriter(GfxTextureArrayData textureArray)
        {
            this.DataModel = textureArray.TableData;
            this.Position = -1;
            this.Size = 0;
            this.Offset = 0;
        }

        /// <summary>
        /// Get the data model the writer is writing.
        /// </summary>
        public DataModelBase DataModel { get; private set; }

        /// <summary>
        /// Get the start position of the texture resource in the stream.
        /// </summary>
        public long Position { get; private set; }

        /// <summary>
        /// Get the size of the written data.
        /// </summary>
        public long Size { get; private set; }

        /// <summary>
        /// Get the offset between the binary header and
        /// the beginning of the binary resource data.
        /// </summary>
        public long Offset { get; private set; }

        /// <summary>
        /// Get the number of textures that is to be written.
        /// </summary>
        public int TextureCount { get; private set; }

        /// <summary>
        /// Write data to the stream in the given context.
        /// </summary>
        /// <param name="context">The binary resource writer context.</param>
        /// <returns>True on success.</returns>
        public bool Write(BinaryResourceWriterContext context)
        {
            Stream stream = context.Stream;
            if (stream == null)
            {
                return false;
            }

            var textureArray = this.DataModel as GfxTextureTableData;
            if (textureArray == null)
            {
                return false;
            }

            // Save the start position in the stream.
            this.Position = stream.Position;

            // Write an empty binary header for place holder,
            // we will come back and fill in the correct values later.
            BinaryStructHeader.Empty.Write(stream);

            if (textureArray.TexturePaths.Any())
            {
                // u64     guid;           //!< ハッシュコード
                // u32     nextOffset;     //!< 次へのオフセット
                // u32     nameLength;     //!< 名前の長さ
                // char    name[8];        //!< 名前

                var texturePaths = textureArray.TexturePaths.Select((value, index) => new { value, index });

                foreach (var texturePath in texturePaths)
                {
                    string key = Path.GetFileNameWithoutExtension(texturePath.value);
                    int lengthWithNull = key.Length + 1;
                    int paddingSize = 8 - (lengthWithNull % 8);
                    int recordSize = 16 + lengthWithNull + paddingSize;
                    int nextOffset = texturePath.index + 1 == texturePaths.Count() ? 0 : recordSize;

                    BinaryConversionUtility.ForResource.WriteStream(
                        stream,
                        Manager.TextureManager.Instance.GetGuid(texturePath.value.ToLower()));
                    BinaryConversionUtility.ForResource.WriteStream(stream, nextOffset);
                    BinaryConversionUtility.ForResource.WriteStream(stream, lengthWithNull);
                    BinaryConversionUtility.ForResource.WriteStream(stream, key.ToArray());
                    BinaryConversionUtility.ForResource.WriteStream(stream, Enumerable.Repeat((byte)0, 1));
                    if (paddingSize != 0)
                    {
                        BinaryConversionUtility.ForResource.WriteStream(stream, Enumerable.Repeat((byte)0, paddingSize));
                    }
                }
            }

            // Save the size of the binary data.
            this.Size = stream.Position - this.Position;

            // Add this writer to the context, the context will write the binary header
            // for the added binary writers.
            context.AddBinaryWriter(this);

            return true;
        }
    }
}
