﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections;
using System.Collections.Generic;
using System.IO;
using System.Linq;

using EffectMaker.DataModel.DataModels;
using EffectMaker.DataModel.Specific.DataModels;
using EffectMaker.Foundation.Log;

namespace EffectMaker.BusinessLogic.BinaryHeaders.Helpers
{
    /// <summary>
    /// Binary header helper for field data binary structures.
    /// </summary>
    public class EmitterAnimationBinaryHeaderHelper : IBinaryHeaderHelper
    {
        /// <summary>
        /// Constructor.
        /// </summary>
        public EmitterAnimationBinaryHeaderHelper()
        {
        }

        /// <summary>
        /// Get the flag indicating whether should set the next pointer
        /// to the end of the binary data.
        /// </summary>
        public bool ShouldSetNextToEnd
        {
            get { return false; }
        }

        /// <summary>
        /// Get the alignment inserts before the next element.
        /// </summary>
        public uint AlignmentToNext
        {
            get { return 0; }
        }

        /// <summary>
        /// 有効なエミッタ時間アニメを走査してその最初のDataModelを返します.
        /// </summary>
        /// <param name="emitter">エミッタデータモデル</param>
        /// <returns>最初の有効なDataModel, 有効なものがなかったら null.</returns>
        public static DataModelBase GetFirstEnableAnime(EmitterData emitter)
        {
            if (emitter.EmitterScaleData.EmitterScaleBasicData.BasicScaleAnimation.EnableAnimation)
            {
                return emitter.EmitterScaleData.EmitterScaleBasicData.BasicScaleAnimation;
            }

            var emitterTransform = emitter.EmitterEmitterData.EmitterEmitterTransformData;

            if (emitterTransform.EmitterScaleAnimation.EnableAnimation)
            {
                return emitterTransform.EmitterScaleAnimation;
            }

            if (emitterTransform.EmitterRotationAnimation.EnableAnimation)
            {
                return emitterTransform.EmitterRotationAnimation;
            }

            if (emitterTransform.EmitterPositionAnimation.EnableAnimation)
            {
                return emitterTransform.EmitterPositionAnimation;
            }

            var emitterEmissionTiming = emitter.EmitterEmissionData.EmitterEmissionTimingData;

            if (emitterEmissionTiming.EmitTimeRateAnimation.EnableAnimation)
            {
                return emitterEmissionTiming.EmitTimeRateAnimation;
            }

            var emitterEmissionVelocity = emitter.EmitterEmissionData.EmitterEmissionVelocityData;

            if (emitterEmissionVelocity.OmnidirectionalVelocityAnimation.EnableAnimation)
            {
                return emitterEmissionVelocity.OmnidirectionalVelocityAnimation;
            }

            if (emitterEmissionVelocity.OrientedDirectionVelocityAnimation.EnableAnimation)
            {
                return emitterEmissionVelocity.OrientedDirectionVelocityAnimation;
            }

            var emitterEmissionGravity = emitter.EmitterEmissionData.EmitterEmissionGravityData;

            if (emitterEmissionGravity.GravityAnimation.EnableAnimation)
            {
                return emitterEmissionGravity.GravityAnimation;
            }

            var emitterParticleLife = emitter.EmitterParticleData.EmitterParticleLifeData;

            if (emitterParticleLife.LifeAnimation.EnableAnimation)
            {
                return emitterParticleLife.LifeAnimation;
            }

            var emitterColor = emitter.EmitterColorData.EmitterColor;

            if (emitterColor.Color0BehaviorType == 2)
            {
                return emitterColor.Color0Animation;
            }

            if (emitterColor.Color1BehaviorType == 2)
            {
                return emitterColor.Color1Animation;
            }

            if (emitterColor.Alpha0BehaviorType == 2)
            {
                return emitterColor.Alpha0Animation;
            }

            if (emitterColor.Alpha1BehaviorType == 2)
            {
                return emitterColor.Alpha1Animation;
            }

            var emitterShape = emitter.EmitterEmitterData.EmitterEmitterShapeData;

            if (emitterShape.ShapeScaleAnimation.EnableAnimation)
            {
                return emitterShape.ShapeScaleAnimation;
            }

            return null;
        }

        /// <summary>
        /// Check if the given data model is accepted for this binary header helper.
        /// </summary>
        /// <param name="dataModel">The data model.</param>
        /// <returns>True if accepted.</returns>
        public bool IsAccepted(DataModelBase dataModel)
        {
            var flag = dataModel is EmitterAnimationCommonData;

            return flag;
        }

        /// <summary>
        /// Get tag string for the binary header.
        /// </summary>
        /// <param name="dataModel">The data model.</param>
        /// <returns>The tag string.</returns>
        public string GetTag(DataModelBase dataModel)
        {
            if (dataModel is EmitterAnimationCommonData)
            {
                if (!((EmitterAnimationCommonData)dataModel).EnableAnimation)
                {
                    return string.Empty;
                }

                var esb = dataModel.Parent as EmitterScaleBasicData;
                if (esb != null)
                {
                    if (esb.BasicScaleAnimation == dataModel)
                    {
                        return "EASL";
                    }
                }

                var etd = dataModel.Parent as EmitterEmitterTransformData;
                if (etd != null)
                {
                    if (etd.EmitterScaleAnimation == dataModel)
                    {
                        return "EAES";
                    }

                    if (etd.EmitterRotationAnimation == dataModel)
                    {
                        return "EAER";
                    }

                    if (etd.EmitterPositionAnimation == dataModel)
                    {
                        return "EAET";
                    }
                }

                var eet = dataModel.Parent as EmitterEmissionTimingData;
                if (eet != null)
                {
                    if (eet.EmitTimeRateAnimation == dataModel)
                    {
                        return "EATR";
                    }
                }

                var eev = dataModel.Parent as EmitterEmissionVelocityData;
                if (eev != null)
                {
                    if (eev.OmnidirectionalVelocityAnimation == dataModel)
                    {
                        return "EAOV";
                    }

                    if (eev.OrientedDirectionVelocityAnimation == dataModel)
                    {
                        return "EADV";
                    }
                }

                var eeg = dataModel.Parent as EmitterEmissionGravityData;
                if (eeg != null)
                {
                    if (eeg.GravityAnimation == dataModel)
                    {
                        return "EAGV";
                    }
                }

                var epl = dataModel.Parent as EmitterParticleLifeData;
                if (epl != null)
                {
                    if (epl.LifeAnimation == dataModel)
                    {
                        return "EAPL";
                    }
                }

                var eec = dataModel.Parent as EmitterEmitterColorData;
                if (eec != null)
                {
                    if (eec.Color0Animation == dataModel)
                    {
                        return "EAC0";
                    }

                    if (eec.Color1Animation == dataModel)
                    {
                        return "EAC1";
                    }

                    if (eec.Alpha0Animation == dataModel)
                    {
                        return "EAA0";
                    }

                    if (eec.Alpha1Animation == dataModel)
                    {
                        return "EAA1";
                    }
                }

                var ees = dataModel.Parent as EmitterEmitterShapeData;
                if (ees != null)
                {
                    if (ees.ShapeScaleAnimation == dataModel)
                    {
                        return "EASS";
                    }
                }
            }

            return string.Empty;
        }

        /// <summary>
        /// Getting the next of the given data model.
        /// </summary>
        /// <param name="current">The current data model.</param>
        /// <returns>The next data model or null if the given is the last.</returns>
        public DataModelBase GetNext(DataModelBase current)
        {
            string tag = this.GetTag(current);
            bool next = false;
            EmitterEmitterTransformData eetd = null;
            EmitterEmissionTimingData eetr = null;
            EmitterEmissionVelocityData eevd = null;
            EmitterEmissionGravityData eegd = null;
            EmitterParticleLifeData epld = null;
            EmitterEmitterColorData eecd = null;
            EmitterEmitterShapeData eesd = null;

            if (tag == "EASL")
            {
                var emitter = GetEmitterData(current);
                if (emitter != null)
                {
                    eetd = emitter.EmitterEmitterData.EmitterEmitterTransformData;

                    if (eetd.EmitterScaleAnimation.EnableAnimation)
                    {
                        return eetd.EmitterScaleAnimation;
                    }
                }

                next = true;
            }

            if (tag == "EAES" || next)
            {
                if (eetd == null)
                {
                    eetd = current.Parent as EmitterEmitterTransformData;
                }

                if (eetd == null)
                {
                    // Logger.Log(LogLevels.Error, "DataModelの構造が想定外です。");
                    return null;
                }

                if (eetd.EmitterRotationAnimation.EnableAnimation)
                {
                    return eetd.EmitterRotationAnimation;
                }

                next = true;
            }

            if (tag == "EAER" || next)
            {
                if (eetd == null)
                {
                    eetd = current.Parent as EmitterEmitterTransformData;
                }

                if (eetd == null)
                {
                    // Logger.Log(LogLevels.Error, "DataModelの構造が想定外です。");
                    return null;
                }

                if (eetd.EmitterPositionAnimation.EnableAnimation)
                {
                    return eetd.EmitterPositionAnimation;
                }

                next = true;
            }

            if (tag == "EAET" || next)
            {
                var emitter = GetEmitterData(current);
                if (emitter != null)
                {
                    eetr = emitter.EmitterEmissionData.EmitterEmissionTimingData;

                    if (eetr.EmitTimeRateAnimation.EnableAnimation)
                    {
                        return eetr.EmitTimeRateAnimation;
                    }
                }

                next = true;
            }

            if (tag == "EATR" || next)
            {
                var emitter = GetEmitterData(current);
                if (emitter != null)
                {
                    eevd = emitter.EmitterEmissionData.EmitterEmissionVelocityData;

                    if (eevd.OmnidirectionalVelocityAnimation.EnableAnimation)
                    {
                        return eevd.OmnidirectionalVelocityAnimation;
                    }
                }

                next = true;
            }

            if (tag == "EAOV" || next)
            {
                var emitter = GetEmitterData(current);
                if (emitter != null)
                {
                    eevd = emitter.EmitterEmissionData.EmitterEmissionVelocityData;

                    if (eevd.OrientedDirectionVelocityAnimation.EnableAnimation)
                    {
                        return eevd.OrientedDirectionVelocityAnimation;
                    }
                }

                next = true;
            }

            if (tag == "EADV" || next)
            {
                var emitter = GetEmitterData(current);
                if (emitter != null)
                {
                    eegd = emitter.EmitterEmissionData.EmitterEmissionGravityData;

                    if (eegd.GravityAnimation.EnableAnimation)
                    {
                        return eegd.GravityAnimation;
                    }
                }

                next = true;
            }

            if (tag == "EAGV" || next)
            {
                var emitter = GetEmitterData(current);
                if (emitter != null)
                {
                    epld = emitter.EmitterParticleData.EmitterParticleLifeData;

                    if (epld.LifeAnimation.EnableAnimation)
                    {
                        return epld.LifeAnimation;
                    }
                }

                next = true;
            }

            if (tag == "EAPL" || next)
            {
                var emitter = GetEmitterData(current);
                if (emitter != null)
                {
                    eecd = emitter.EmitterColorData.EmitterColor;

                    if (eecd.Color0BehaviorType == 2)
                    {
                        return eecd.Color0Animation;
                    }
                }

                next = true;
            }

            if (tag == "EAC0" || next)
            {
                var emitter = GetEmitterData(current);
                if (emitter != null)
                {
                    eecd = emitter.EmitterColorData.EmitterColor;

                    if (eecd.Color1BehaviorType == 2)
                    {
                        return eecd.Color1Animation;
                    }
                }

                next = true;
            }

            if (tag == "EAC1" || next)
            {
                var emitter = GetEmitterData(current);
                if (emitter != null)
                {
                    eecd = emitter.EmitterColorData.EmitterColor;

                    if (eecd.Alpha0BehaviorType == 2)
                    {
                        return eecd.Alpha0Animation;
                    }
                }

                next = true;
            }

            if (tag == "EAA0" || next)
            {
                var emitter = GetEmitterData(current);
                if (emitter != null)
                {
                    eecd = emitter.EmitterColorData.EmitterColor;

                    if (eecd.Alpha1BehaviorType == 2)
                    {
                        return eecd.Alpha1Animation;
                    }
                }

                next = true;
            }

            if (tag == "EAA1" || next)
            {
                var emitter = GetEmitterData(current);
                if (emitter != null)
                {
                    eesd = emitter.EmitterEmitterData.EmitterEmitterShapeData;

                    if (eesd.ShapeScaleAnimation.EnableAnimation)
                    {
                        return eesd.ShapeScaleAnimation;
                    }
                }
            }

            return null;
        }

        /// <summary>
        /// Getting the first child of the given data model.
        /// </summary>
        /// <param name="current">The current data model.</param>
        /// <returns>The first child data model or null if the child does not exist.</returns>
        public DataModelBase GetChild(DataModelBase current)
        {
            return null;
        }

        /// <summary>
        /// Getting the first data model of the given data model's sub tree.
        /// </summary>
        /// <param name="current">The current data model.</param>
        /// <returns>The first data model of the sub tree or null there is no sub tree.</returns>
        public DataModelBase GetSub(DataModelBase current)
        {
            return null;
        }

        /// <summary>
        /// Getting the offset from the binary header to the binary data.
        /// </summary>
        /// <param name="current">The current data model.</param>
        /// <param name="headerPos">The position of the binary header.</param>
        /// <returns>The offset from the binary header to the binary data.</returns>
        public uint GetOffset(DataModelBase current, uint headerPos)
        {
            return BinaryStructHeader.Size;
        }

        /// <summary>
        /// データモデルの親を再帰的に辿り、エミッタのルートデータモデルを返します。
        /// </summary>
        /// <param name="dataModel">データモデル</param>
        /// <returns>エミッタデータモデル</returns>
        private static EmitterData GetEmitterData(DataModelBase dataModel)
        {
            var result = dataModel;
            while (!(result is EmitterData))
            {
                result = result.Parent;
                if (result == null)
                {
                    // Logger.Log(LogLevels.Error, "DataModelの構造が想定外です。");
                    return null;
                }
            }

            return result as EmitterData;
        }
    }
}
