﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
#region File Description
#endregion

using EffectMaker.Foundation.Serialization;

namespace EffectMaker.Communicator
{
    #region Using Statements
    using System;
    using System.Net;
    using System.Text;
    #endregion

    /// <summary>
    /// コネクションコンテキスト インタフェースです.
    /// </summary>
    [Serializable]
    public class ConnectionContext : IXmlDocSerializable
    {
        #region Public Methods

        /// <summary>
        /// コンストラクタです.
        /// </summary>
        public ConnectionContext()
        {
            // リセットします.
            this.Reset();
        }

        /// <summary>
        /// コピーコンストラクタです.
        /// </summary>
        /// <param name="src">コピー元の値です.</param>
        public ConnectionContext(ConnectionContext src)
        {
            // コピーを行います.
            this.Copy(src);
        }

        #region Public Properties

        /// <summary>
        /// 接続名を設定/取得します.
        /// </summary>
        public string Name { get; private set; }

        /// <summary>
        /// 識別名を設定/取得します.
        /// </summary>
        public string Identifier { get; private set; }

        /// <summary>
        /// エンコーディングタイプを設定/取得します.
        /// </summary>
        public Encoding Encoding { get; private set; }

        /// <summary>
        /// プラットフォームタイプを設定/取得します.
        /// </summary>
        public int Platform { get; private set; }

        /// <summary>
        /// IGE通信時のプロトコルを設定/取得します.
        /// </summary>
        public string Protocol { get; private set; }

        /// <summary>
        /// HTCS通信時のピア名を設定/取得します.
        /// </summary>
        public string PeerType { get; private set; }

        /// <summary>
        /// IPアドレスを設定/取得します.
        /// </summary>
        public IPAddress IPAddress { get; set; }

        /// <summary>
        /// ポート番号を設定/取得します.
        /// </summary>
        public int Port { get; set; }

        /// <summary>
        /// ビッグエンディアンかどうかを設定/取得します.
        /// </summary>
        public bool IsBigEndian { get; private set; }

        /// <summary>
        /// 情報を使用可能かどうかのフラグを取得します.
        /// </summary>
        public bool IsReady { get; private set; }

        #endregion

        /// <summary>
        /// 同じターゲットかどうかチェックします.
        /// </summary>
        /// <param name="context">コネクションコンテキストです</param>
        /// <returns>同じターゲットであればtrue, そうでなければfalseを返却します</returns>
        public bool IsSameTarget(ConnectionContext context)
        {
            if (this.Name != context.Name)
            {
                return false;
            }

            if (this.Identifier != context.Identifier)
            {
                return false;
            }

            if (!Equals(this.Encoding, context.Encoding))
            {
                return false;
            }

            if (this.Platform != context.Platform)
            {
                return false;
            }

            if (this.Protocol != context.Protocol)
            {
                return false;
            }

            if (this.PeerType != context.PeerType)
            {
                return false;
            }

            if (!Equals(this.IPAddress, context.IPAddress))
            {
                return false;
            }

            if (this.Port != context.Port)
            {
                return false;
            }

            if (this.IsBigEndian != context.IsBigEndian)
            {
                return false;
            }

            if (this.IsReady != context.IsReady)
            {
                return false;
            }

            return true;
        }

        /// <summary>
        /// コピーを行います.
        /// </summary>
        /// <param name="src">コピー元の値.</param>
        /// <returns>コピーに成功した場合はtrue, 失敗した場合falseを返却します.</returns>
        public bool Copy(ConnectionContext src)
        {
            if (src == null)
            {
                return false;
            }

            this.Name        = src.Name;
            this.Identifier  = src.Identifier;
            this.Platform    = src.Platform;
            this.Protocol    = src.Protocol;
            this.PeerType    = src.PeerType;
            this.Encoding    = src.Encoding;
            this.IPAddress   = src.IPAddress;
            this.Port        = src.Port;
            this.IsBigEndian = src.IsBigEndian;
            this.IsReady     = src.IsReady;

            return true;
        }

        /// <summary>
        /// リセットします.
        /// </summary>
        public void Reset()
        {
            this.Name        = string.Empty;
            this.Identifier  = string.Empty;
            this.Platform    = -1;
            this.Protocol    = string.Empty;
            this.PeerType    = string.Empty;
            this.Encoding    = Encoding.UTF8;
            this.IPAddress   = null;
            this.Port        = -1;
            this.IsBigEndian = false;
            this.IsReady     = true;
        }

        #endregion

        /// <summary>
        /// Deserializes from the given XML node.
        /// </summary>
        /// <param name="context">The data context needed for the deserialization.</param>
        /// <returns>True on success.</returns>
        public bool ReadXml(XmlDocSerializationContext context)
        {
            this.Name = this.ReadElement(context, "Name", this.Name);
            this.Identifier = this.ReadElement(context, "Identifier", this.Identifier);
            this.Platform = this.ReadElement(context, "Platform", this.Platform);
            this.Protocol = this.ReadElement(context, "Protocol", this.Protocol);
            this.PeerType = this.ReadElement(context, "PeerType", this.PeerType);
            this.Encoding = this.ReadElement(context, "Encoding", this.Encoding);
            this.IPAddress = this.ReadElement(context, "IPAddress", this.IPAddress);
            this.Port = this.ReadElement(context, "Port", this.Port);
            this.IsBigEndian = this.ReadElement(context, "IsBigEndian", this.IsBigEndian);
            this.IsReady = this.ReadElement(context, "IsReady", this.IsReady);
            return true;
        }

        /// <summary>
        /// Serializes this object to a XML node.
        /// </summary>
        /// <param name="context">The data context needed for the serialization.</param>
        /// <returns>True on success.</returns>
        public bool WriteXml(XmlDocSerializationContext context)
        {
            this.WriteElement(context, "Name", this.Name);
            this.WriteElement(context, "Identifier", this.Identifier);
            this.WriteElement(context, "Platform", this.Platform);
            this.WriteElement(context, "Protocol", this.Protocol);
            this.WriteElement(context, "PeerType", this.PeerType);
            this.WriteElement(context, "Encoding", this.Encoding);
            this.WriteElement(context, "IPAddress", this.IPAddress);
            this.WriteElement(context, "Port", this.Port);
            this.WriteElement(context, "IsBigEndian", this.IsBigEndian);
            this.WriteElement(context, "IsReady", this.IsReady);
            return true;
        }
    }
}
