﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "pcgx_hio.h"

#define PCGX_HIO_OK        (0)
#define PCGX_HIO_ERR    (-1)
#define PCGX_ASSERT(x)    ((void)0)

////////////////////////////////////////////////////////////////
//    PCGX_HostFile
////////////////////////////////////////////////////////////////
PCGX_HostFile::PCGX_HostFile()
{
    fd = -1;
    fp = NULL;
}

PCGX_HostFile::~PCGX_HostFile()
{
    Release();
}

void PCGX_HostFile::Release()
{
    if(fp) fclose(fp);
    fp = NULL;
    fd = -1;
}

////////////////////////////////////////////////////////////////
//    PCGX_SIOMemory
////////////////////////////////////////////////////////////////
PCGX_SIOMemory::PCGX_SIOMemory()
{
    hSharedMemory = NULL;
    pAllMem = NULL;
    Release();
}

PCGX_SIOMemory::~PCGX_SIOMemory()
{
    Release();
}

HRESULT PCGX_SIOMemory::Init(int _ch)
{
    Release();

    ch = _ch;
    TCHAR name[64];
    _stprintf_s(name, 64, PCGX_SIOMEM_NAME_FMT, (UINT)ch);

    hSharedMemory = OpenFileMapping(FILE_MAP_ALL_ACCESS, FALSE, name);
    if(hSharedMemory){
        HRESULT hr = Map();
        if(FAILED(hr)){
            PCGX_ASSERT(0);
            Release();
            return PCGX_HIO_ERR;
        }
        (*pRefCount)++;
        Unmap();
    }
    else{
        hSharedMemory = CreateFileMapping((HANDLE)(-1), NULL, PAGE_READWRITE, 0, PCGX_SIOMEM_SIZE, name);
        if(!hSharedMemory){
            PCGX_ASSERT(0);
            Release();
            return PCGX_HIO_ERR;
        }

        HRESULT hr = Map();
        if(FAILED(hr)){
            PCGX_ASSERT(0);
            Release();
            return PCGX_HIO_ERR;
        }
        (*pRefCount)    = 1;
        (*pState)       = 0;
        (*pReadCur[0])  = 0;
        (*pWriteCur[0]) = 0;
        (*pReadCur[1])  = 0;
        (*pWriteCur[1]) = 0;
        Unmap();
    }
    return PCGX_HIO_OK;
}

void PCGX_SIOMemory::Release()
{
    if(hSharedMemory){
        PCGX_ASSERT(!pAllMem);
        HRESULT hr = Map();
        if(SUCCEEDED(hr)){
            (*pRefCount)--;
            (*pState)++;
        }
        Unmap();

        CloseHandle(hSharedMemory);
        hSharedMemory = NULL;
    }
    ch = -1;
}

HRESULT PCGX_SIOMemory::Map()
{
    PCGX_ASSERT(!pAllMem);
    pAllMem = (BYTE *)MapViewOfFile(hSharedMemory, FILE_MAP_ALL_ACCESS, 0, 0, 0);
    if(!pAllMem) return PCGX_HIO_ERR;
    pRefCount    = (UINT *)(pAllMem + (sizeof(UINT)*0));
    pState       = (UINT *)(pAllMem + (sizeof(UINT)*1));
#ifdef NN_PCGX_HIO_DEV
    //    DEV (PCViewer) 側
    pReadCur[0]  = (UINT *)(pAllMem + ((sizeof(UINT)*2)+(sizeof(UINT)*0)));
    pWriteCur[0] = (UINT *)(pAllMem + ((sizeof(UINT)*2)+(sizeof(UINT)*1)));
    pBuf[0]      = (BYTE *)(pAllMem + ((sizeof(UINT)*2)+(sizeof(UINT)*2)));
    pReadCur[1]  = (UINT *)(pAllMem + ((sizeof(UINT)*2)+(sizeof(UINT)*2)+PCGX_SIOBUF_SIZE+(sizeof(UINT)*0)));
    pWriteCur[1] = (UINT *)(pAllMem + ((sizeof(UINT)*2)+(sizeof(UINT)*2)+PCGX_SIOBUF_SIZE+(sizeof(UINT)*1)));
    pBuf[1]      = (BYTE *)(pAllMem + ((sizeof(UINT)*2)+(sizeof(UINT)*2)+PCGX_SIOBUF_SIZE+(sizeof(UINT)*2)));
#else
    //    HOST (PC) 側
    pReadCur[1]  = (UINT *)(pAllMem + ((sizeof(UINT)*2)+(sizeof(UINT)*0)));
    pWriteCur[1] = (UINT *)(pAllMem + ((sizeof(UINT)*2)+(sizeof(UINT)*1)));
    pBuf[1]      = (BYTE *)(pAllMem + ((sizeof(UINT)*2)+(sizeof(UINT)*2)));
    pReadCur[0]  = (UINT *)(pAllMem + ((sizeof(UINT)*2)+(sizeof(UINT)*2)+PCGX_SIOBUF_SIZE+(sizeof(UINT)*0)));
    pWriteCur[0] = (UINT *)(pAllMem + ((sizeof(UINT)*2)+(sizeof(UINT)*2)+PCGX_SIOBUF_SIZE+(sizeof(UINT)*1)));
    pBuf[0]      = (BYTE *)(pAllMem + ((sizeof(UINT)*2)+(sizeof(UINT)*2)+PCGX_SIOBUF_SIZE+(sizeof(UINT)*2)));
#endif
    return PCGX_HIO_OK;
}

void PCGX_SIOMemory::Unmap()
{
    if(!pAllMem) return;
    UnmapViewOfFile(pAllMem);
    pAllMem      = NULL;
    pRefCount    = NULL;
    pState       = NULL;
    pBuf[0]      = NULL;
    pReadCur[0]  = NULL;
    pWriteCur[0] = NULL;
    pBuf[1]      = NULL;
    pReadCur[1]  = NULL;
    pWriteCur[1] = NULL;
}

size_t PCGX_SIOMemory::GetReadDataSize()
{
    PCGX_ASSERT(pAllMem);
    if(!pAllMem) return 0;
    if((*pReadCur[0]) > (*pWriteCur[0])) return (*pWriteCur[0]) + (PCGX_SIOBUF_SIZE - (*pReadCur[0]));
    return (*pWriteCur[0]) - (*pReadCur[0]);
}

size_t PCGX_SIOMemory::GetWriteDataSize()
{
    PCGX_ASSERT(pAllMem);
    if(!pAllMem) return 0;
    if((*pReadCur[1]) > (*pWriteCur[1])) return (*pWriteCur[1]) + (PCGX_SIOBUF_SIZE - (*pReadCur[1]));
    return (*pWriteCur[1]) - (*pReadCur[1]);
}

size_t PCGX_SIOMemory::GetWritableSize()
{
    PCGX_ASSERT(pAllMem);
    if(!pAllMem) return 0;
    int w_size = (PCGX_SIOBUF_SIZE - 1) - (int)GetWriteDataSize();
    if(w_size > 0)  return (size_t)w_size;
    else            return 0;
}

void PCGX_SIOMemory::Write(const void *buf, size_t size)
{
    PCGX_ASSERT(pAllMem);
    PCGX_ASSERT(size <= GetWritableSize());
    BYTE *in = (BYTE *)buf;

    if((*pReadCur[1]) <= (*pWriteCur[1])){
        size_t s2 = (PCGX_SIOBUF_SIZE - (*pWriteCur[1]));
        if(size > s2){
            memcpy_s((pBuf[1] + (*pWriteCur[1])), s2, in, s2);
            (*pWriteCur[1]) = 0;
            in += s2;

            s2 = size - s2;
            memcpy_s((pBuf[1] /*+ (*pWriteCur[1])*/), s2, in, s2);
            (*pWriteCur[1]) += s2;
        }
        else{
            memcpy_s((pBuf[1] + (*pWriteCur[1])), size, in, size);
            (*pWriteCur[1]) += size;

            if((*pWriteCur[1]) == PCGX_SIOBUF_SIZE){
                (*pWriteCur[1]) = 0;
            }
        }
    }
    else{
        memcpy_s((pBuf[1] + (*pWriteCur[1])), size, in, size);
        (*pWriteCur[1]) += size;
    }
}

void PCGX_SIOMemory::Read(void *buf, size_t size)
{
    PCGX_ASSERT(pAllMem);
    PCGX_ASSERT(size <= GetReadDataSize());
    BYTE *out = (BYTE *)buf;

    if((*pReadCur[0]) > (*pWriteCur[0])){
        size_t s2 = (PCGX_SIOBUF_SIZE - (*pReadCur[0]));
        if(size > s2){
            memcpy_s(out, s2, (pBuf[0] + (*pReadCur[0])), s2);
            (*pReadCur[0]) = 0;
            out += s2;

            s2 = size - s2;
            memcpy_s(out, s2, (pBuf[0] /*+ (*pReadCur[0])*/), s2);
            (*pReadCur[0]) += s2;
        }
        else{
            memcpy_s(out, size, (pBuf[0] + (*pReadCur[0])), size);
            (*pReadCur[0]) += size;

            if((*pReadCur[0]) == PCGX_SIOBUF_SIZE){
                (*pReadCur[0]) = 0;
            }
        }
    }
    else{
        memcpy_s(out, size, (pBuf[0] + (*pReadCur[0])), size);
        (*pReadCur[0]) += size;
    }

    if((*pReadCur[0]) == (*pWriteCur[0])){
        (*pReadCur[0]) = 0;
        (*pWriteCur[0]) = 0;
    }
}

////////////////////////////////////////////////////////////////
//    PCGX_HIO
////////////////////////////////////////////////////////////////
PCGX_HIO::PCGX_HIO()
{
    hioMutex = OpenMutex(MUTEX_ALL_ACCESS, FALSE, PCGX_HIO_MUTEX_NAME);
    if(!hioMutex){
        hioMutex = CreateMutex(NULL, FALSE, PCGX_HIO_MUTEX_NAME);
    }
}

PCGX_HIO::~PCGX_HIO()
{
    for(UINT i=0; i<_fileList.size(); i++){
        delete _fileList[i];
        _fileList[i] = NULL;
    }
    for(UINT i=0; i<_sioList.size(); i++){
        delete _sioList[i];
        _sioList[i] = NULL;
    }
    _fileList.clear();
    _sioList.clear();

    if(hioMutex){
        CloseHandle(hioMutex);
        hioMutex = NULL;
    }
}

bool PCGX_HIO::IsHIODevRunning()
{
    HANDLE mutex = OpenMutex(MUTEX_ALL_ACCESS, FALSE, PCGX_HIODEV_MUTEX_NAME);
    if(!mutex) return false;
    CloseHandle(mutex);
    return true;
}

bool PCGX_HIO::IsHIODevDisable()
{
    HANDLE mutex = OpenMutex(MUTEX_ALL_ACCESS, FALSE, PCGX_HIODEV_DISABLE_MUTEX_NAME);
    if(!mutex) return false;
    CloseHandle(mutex);
    return true;
}

void PCGX_HIO::MUTEX_LOCK()
{
    WaitForSingleObject(hioMutex, INFINITE);
}

void PCGX_HIO::MUTEX_UNLOCK()
{
    ReleaseMutex(hioMutex);
}

PCGX_HostFile *PCGX_HIO::SearchFile(int fd)
{
    for(UINT i=0; i<_fileList.size(); i++){
        if(_fileList[i]->fd == fd) return _fileList[i];
    }
    return NULL;
}

PCGX_SIOMemory *PCGX_HIO::SearchSIO(int ch)
{
    for(UINT i=0; i<_sioList.size(); i++){
        if(_sioList[i]->ch == ch) return _sioList[i];
    }
    return NULL;
}

int PCGX_HIO::SeekHostFile(int fd, INT64* pPosition, INT64 offset, int type)
{
    PCGX_HostFile *file = SearchFile(fd);
    if(!file || !file->fp){
        return PCGX_HIO_ERR;
    }

    int ret = -1;
    switch(type){
    case 0:    ret = fseek(file->fp, (long)offset, SEEK_SET);    break;
    case 1:    ret = fseek(file->fp, (long)offset, SEEK_CUR);    break;
    case 2:    ret = fseek(file->fp, (long)offset, SEEK_END);    break;
    }
    *pPosition = ftell(file->fp);
    return (ret==0) ? PCGX_HIO_OK : PCGX_HIO_ERR;
}

int PCGX_HIO::OpenHostFile(int fd, const char* path, int accessMode)
{
    CloseHostFile(fd);

    PCGX_HostFile *file = new PCGX_HostFile;
    file->fd = fd;
    if(accessMode==PCGX_HIO_ACCESS_MODE_READ)
        fopen_s(&file->fp, path, "rb");
    else if(accessMode==PCGX_HIO_ACCESS_MODE_WRITE)
        fopen_s(&file->fp, path, "wb");
    else if(accessMode==PCGX_HIO_ACCESS_MODE_READ_WRITE)
        fopen_s(&file->fp, path, "rb+");
    if(!file->fp){
        delete file;
        return PCGX_HIO_ERR;
    }
    _fileList.push_back(file);

    return PCGX_HIO_OK;
}

int PCGX_HIO::OpenHostFile(int fd, const wchar_t* path, int accessMode)
{
    CloseHostFile(fd);

    PCGX_HostFile *file = new PCGX_HostFile;
    file->fd = fd;
    if(accessMode==PCGX_HIO_ACCESS_MODE_READ)
        _wfopen_s(&file->fp, path, L"rb");
    else if(accessMode==PCGX_HIO_ACCESS_MODE_READ_WRITE)
        _wfopen_s(&file->fp, path, L"wb");
    if(!file->fp){
        delete file;
        return PCGX_HIO_ERR;
    }
    _fileList.push_back(file);

    return PCGX_HIO_OK;
}

int PCGX_HIO::CloseHostFile(int fd)
{
    PCGX_HostFile *del;
    for(std::vector<PCGX_HostFile *>::iterator i=_fileList.begin(); i!=_fileList.end(); i++){
        del = (*i);
        if(del->fd == fd){
            _fileList.erase(i);
            delete del;
            break;
        }
    }
    return PCGX_HIO_OK;
}

int PCGX_HIO::ReadHostFile(int fd, size_t* pRead, void* buf, size_t size)
{
    PCGX_HostFile *file = SearchFile(fd);
    if(!file || !file->fp){
        return PCGX_HIO_ERR;
    }

    size_t ret = fread_s(buf, size, size, 1, file->fp);
    *pRead = (ret ? size : 0);
    if(ret)    return PCGX_HIO_OK;
    else    return PCGX_HIO_ERR;
}

int PCGX_HIO::WriteHostFile(int fd, size_t* pWritten, const void* buf, size_t size)
{
    PCGX_HostFile *file = SearchFile(fd);
    if(!file || !file->fp){
        return PCGX_HIO_ERR;
    }

    size_t ret = fwrite(buf, size, 1, file->fp);
    *pWritten = (ret ? size : 0);
    if(ret)    return PCGX_HIO_OK;
    else    return PCGX_HIO_ERR;
}

////////////////////////////////
//    DEVHIO WRAPPER
////////////////////////////////
int PCGX_HIO::Find(unsigned long *ids, int size)
{
    //MessageBox(NULL, _T("HIO::Find Called"), _T("pcgx_hio"), 0);
    if(IsHIODevRunning()){
        ids[0] = PCGX_HIODEV_ID;
        return 1;
    }
    else{
        return 0;
    }
}

int PCGX_HIO::Connect(unsigned long id)
{
    if(id != PCGX_HIODEV_ID) return PCGX_HIO_ERR;
    if(!IsHIODevRunning()) return PCGX_HIO_ERR;
    return PCGX_HIO_OK;
}

int PCGX_HIO::Disconnect()
{
    return PCGX_HIO_OK;
}

int PCGX_HIO::OpenSio(int ch)
{
    //MessageBox(NULL, _T("HIO::OpenSio Called"), _T("pcgx_hio"), 0);
    MUTEX_LOCK();    ////////////////
    PCGX_SIOMemory *siomem;
    siomem = SearchSIO(ch);
    PCGX_ASSERT(siomem==NULL);

    siomem = new PCGX_SIOMemory;
    HRESULT hr = siomem->Init(ch);
    if(SUCCEEDED(hr)){
        _sioList.push_back(siomem);
    }
    else{
        delete siomem;
    }
    MUTEX_UNLOCK();    ////////////////

    if(SUCCEEDED(hr))    return PCGX_HIO_OK;
    else                return PCGX_HIO_ERR;
}

int PCGX_HIO::CloseSio(int ch)
{
    //MessageBox(NULL, _T("HIO::CloseSio Called"), _T("pcgx_hio"), 0);
    MUTEX_LOCK();    ////////////////
    PCGX_SIOMemory *del;
    for(std::vector<PCGX_SIOMemory *>::iterator i=_sioList.begin(); i!=_sioList.end(); i++){
        del = (*i);
        if(del->ch == ch){
            _sioList.erase(i);
            delete del;
            break;
        }
    }
    MUTEX_UNLOCK();    ////////////////

    return PCGX_HIO_OK;
}

int PCGX_HIO::ConnectSio(int ch)
{
    //MessageBox(NULL, _T("HIO::ConnectSio Called"), _T("pcgx_hio"), 0);
    bool connect = false;

    while(!connect){
        if(!IsHIODevRunning()){
            break;
        }

        MUTEX_LOCK();    ////////////////
        PCGX_SIOMemory *siomem = SearchSIO(ch);
        if(siomem){
            HRESULT hr = siomem->Map();
            if(SUCCEEDED(hr)){
                if(*siomem->pRefCount >= 2) connect = true;
                (*siomem->pState)       = 0;
                (*siomem->pReadCur[0])  = 0;
                (*siomem->pWriteCur[0]) = 0;
                (*siomem->pReadCur[1])  = 0;
                (*siomem->pWriteCur[1]) = 0;
                siomem->Unmap();
            }
            else{
                MUTEX_UNLOCK();
                break;
            }
        }
        MUTEX_UNLOCK();    ////////////////

        //    ウェイト
        if(!connect){
            Sleep(PCGX_SIO_WAIT_INTERVAL);
        }
    }
    return connect ? PCGX_HIO_OK : PCGX_HIO_ERR;
}

int PCGX_HIO::WaitSio(int ch, int attr)
{
    bool connect = false;
    bool err = false;

    while(!connect){
        if(!IsHIODevRunning()){
            err = true;
            break;
        }

        MUTEX_LOCK();    ////////////////
        PCGX_SIOMemory *siomem = SearchSIO(ch);
        if(siomem){
            HRESULT hr = siomem->Map();
            if(SUCCEEDED(hr)){
                if(*siomem->pState > 0){
                    //    この SIO インスタンスは無効です
                }
                else{
                    if(*siomem->pRefCount >= 2) connect = true;
                }
                siomem->Unmap();
            }
            else{
                err = true;
                MUTEX_UNLOCK();
                break;
            }
        }
        MUTEX_UNLOCK();    ////////////////

        //    ウェイト
        if(attr == PCGX_SIO_ATTR_NO_WAIT) break;
        if(!connect){
            Sleep(PCGX_SIO_WAIT_INTERVAL);
        }
    }

    if(connect) return 1;
    return err ? PCGX_HIO_ERR : 0;
}

int PCGX_HIO::DisconnectSio(int ch)
{
    return PCGX_HIO_OK;
}

int PCGX_HIO::FlushSio(int ch)
{
    return PCGX_HIO_OK;
}

int PCGX_HIO::ReadSio(int ch, void* buf, int length, int attr)
{
    int read_size = 0;
    int rest = length;

    while(read_size < length){
        MUTEX_LOCK();    ////////////////
        PCGX_SIOMemory *siomem = SearchSIO(ch);
        if(!siomem){
            MUTEX_UNLOCK();
            return -1;
        }

        HRESULT hr = siomem->Map();
        if(SUCCEEDED(hr)){
            if(*siomem->pRefCount < 2){
                siomem->Unmap();
                MUTEX_UNLOCK();
                return -1;
            }
            if(*siomem->pState > 0){
                siomem->Unmap();
                MUTEX_UNLOCK();
                return -1;
            }

            int s2 = rest;
            if((int)siomem->GetReadDataSize() < s2){
                s2 = siomem->GetReadDataSize();
            }
            siomem->Read(buf, s2);
            read_size += s2;
            rest -= s2;
            siomem->Unmap();
        }
        else{
            MUTEX_UNLOCK();
            return -1;
        }
        MUTEX_UNLOCK();    ////////////////

        //    ウェイト
        if(attr == PCGX_SIO_ATTR_NO_WAIT) break;
        if(read_size < length){
            Sleep(PCGX_SIO_WAIT_INTERVAL);
        }
    }

    return read_size;
}

int PCGX_HIO::WriteSio(int ch, const void* buf, int length, int attr)
{
    int write_size = 0;
    int rest = length;

    while(write_size < length){
        MUTEX_LOCK();    ////////////////
        PCGX_SIOMemory *siomem = SearchSIO(ch);
        if(!siomem){
            MUTEX_UNLOCK();
            return -1;
        }

        HRESULT hr = siomem->Map();
        if(SUCCEEDED(hr)){
            if(*siomem->pRefCount < 2){
                siomem->Unmap();
                MUTEX_UNLOCK();
                return -1;
            }
            if(*siomem->pState > 0){
                siomem->Unmap();
                MUTEX_UNLOCK();
                return -1;
            }

            int s2 = rest;
            if((int)siomem->GetWritableSize() < s2){
                s2 = siomem->GetWritableSize();
            }
            siomem->Write(buf, s2);
            write_size += s2;
            rest -= s2;
            siomem->Unmap();
        }
        else{
            MUTEX_UNLOCK();
            return -1;
        }
        MUTEX_UNLOCK();    ////////////////

        //    ウェイト
        if(attr == PCGX_SIO_ATTR_NO_WAIT) break;
        if(write_size < length){
            Sleep(PCGX_SIO_WAIT_INTERVAL);
        }
    }

    return write_size;
}

int PCGX_HIO::GetReadableBytesSio(int ch, int attr)
{
    int read_size = 0;

    MUTEX_LOCK();    ////////////////
    PCGX_SIOMemory *siomem = SearchSIO(ch);
    if(!siomem){
        MUTEX_UNLOCK();
        return -1;
    }

    HRESULT hr = siomem->Map();
    if(SUCCEEDED(hr)){
        read_size = siomem->GetReadDataSize();
        siomem->Unmap();
    }
    else{
        MUTEX_UNLOCK();
        return -1;
    }
    MUTEX_UNLOCK();    ////////////////

    return read_size;
}

int PCGX_HIO::GetWritableBytesSio(int ch, int attr)
{
    int write_size = 0;

    MUTEX_LOCK();    ////////////////
    PCGX_SIOMemory *siomem = SearchSIO(ch);
    if(!siomem){
        MUTEX_UNLOCK();
        return -1;
    }

    HRESULT hr = siomem->Map();
    if(SUCCEEDED(hr)){
        write_size = siomem->GetWritableSize();
        siomem->Unmap();
    }
    else{
        MUTEX_UNLOCK();
        return -1;
    }
    MUTEX_UNLOCK();    ////////////////

    return write_size;
}

#ifndef NN_PCGX_HIO_DEV
////////////////////////////////////////////////////////////////
//    PCGX_HIODevice
////////////////////////////////////////////////////////////////
int PCGX_HIODevice::Find(unsigned long *ids, int size)
{
    int ret = mpDevhio->Find(ids, size);
    if(!ret) mPCGXDev.Find(ids, size);
    return ret;
}

int PCGX_HIODevice::Connect(unsigned long id)
{
    mConnectID = id;
    if(id == PCGX_HIODEV_ID){
        return mPCGXDev.Connect(id);
    }
    else{
        return mpDevhio->Connect(id);
    }
}

int PCGX_HIODevice::Disconnect()
{
    unsigned long id = mConnectID;
    mConnectID = 0;
    if(id == PCGX_HIODEV_ID){
        return mPCGXDev.Disconnect();
    }
    else{
        return mpDevhio->Disconnect();
    }
}

int PCGX_HIODevice::OpenSio(int ch)
{
    if(mPCGXDev.IsHIODevRunning() && (!mPCGXDev.IsHIODevDisable())){
        return mPCGXDev.OpenSio(ch);
    }
    else{
        return mpDevhio->OpenSio(ch);
    }
}

int PCGX_HIODevice::CloseSio(int ch)
{
    if(mPCGXDev.SearchSIO(ch)){
        return mPCGXDev.CloseSio(ch);
    }
    else{
        return mpDevhio->CloseSio(ch);
    }
}

int PCGX_HIODevice::ConnectSio(int ch)
{
    if(mPCGXDev.SearchSIO(ch)){
        return mPCGXDev.ConnectSio(ch);
    }
    else{
        return mpDevhio->ConnectSio(ch);
    }
}

int PCGX_HIODevice::WaitSio(int ch, int attr)
{
    if(mPCGXDev.SearchSIO(ch)){
        return mPCGXDev.WaitSio(ch, attr);
    }
    else{
        return mpDevhio->WaitSio(ch, attr);
    }
}

int PCGX_HIODevice::DisconnectSio(int ch)
{
    if(mPCGXDev.SearchSIO(ch)){
        return mPCGXDev.DisconnectSio(ch);
    }
    else{
        return mpDevhio->DisconnectSio(ch);
    }
}

int PCGX_HIODevice::FlushSio(int ch)
{
    if(mPCGXDev.SearchSIO(ch)){
        return mPCGXDev.FlushSio(ch);
    }
    else{
        return mpDevhio->FlushSio(ch);
    }
}

int PCGX_HIODevice::ReadSio(int ch, void* buf, int length, int attr)
{
    if(mPCGXDev.SearchSIO(ch)){
        return mPCGXDev.ReadSio(ch, buf, length, attr);
    }
    else{
        return mpDevhio->ReadSio(ch, buf, length, attr);
    }
}

int PCGX_HIODevice::WriteSio(int ch, const void* buf, int length, int attr)
{
    if(mPCGXDev.SearchSIO(ch)){
        return mPCGXDev.WriteSio(ch, buf, length, attr);
    }
    else{
        return mpDevhio->WriteSio(ch, buf, length, attr);
    }
}

int PCGX_HIODevice::GetReadableBytesSio(int ch, int attr)
{
    if(mPCGXDev.SearchSIO(ch)){
        return mPCGXDev.GetReadableBytesSio(ch, attr);
    }
    else{
        return mpDevhio->GetReadableBytesSio(ch, attr);
    }
}

int PCGX_HIODevice::GetWritableBytesSio(int ch, int attr)
{
    if(mPCGXDev.SearchSIO(ch)){
        return mPCGXDev.GetWritableBytesSio(ch, attr);
    }
    else{
        return mpDevhio->GetWritableBytesSio(ch, attr);
    }
}
#endif
