﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections;
using System.Collections.Generic;
using System.IO;

namespace EffectMaker.Foundation.Utility
{
    /// <summary>
    /// Convert data to byte array.
    /// </summary>
    public class BinaryConversionUtility
    {
        /// <summary>
        /// プロトコル生成用のインスタンス
        /// </summary>
        private static BinaryConversionUtility forProtocol = new BinaryConversionUtility();

        /// <summary>
        /// リソースバイナリ生成用のインスタンス
        /// </summary>
        private static BinaryConversionUtility forResource = new BinaryConversionUtility();

        /// <summary>
        /// Static constructor.
        /// </summary>
        private BinaryConversionUtility()
        {
            IsLittleEndian = false;
        }

        /// <summary>
        /// プロトコル生成用のインスタンスを取得します。
        /// </summary>
        public static BinaryConversionUtility ForProtocol
        {
            get { return forProtocol; }
        }

        /// <summary>
        /// リソースバイナリ生成用のインスタンスを取得します。
        /// </summary>
        public static BinaryConversionUtility ForResource
        {
            get { return forResource; }
        }

        /// <summary>
        /// Get or set the flag indicating whether to use little-endian or big-endian
        /// for binary conversion.
        /// </summary>
        public bool IsLittleEndian { get; set; }

        /// <summary>
        /// Convert a value to byte array and write it to the stream.
        /// </summary>
        /// <typeparam name="T">The type to cast the value to.</typeparam>
        /// <param name="stream">The stream to write to.</param>
        /// <param name="value">The value.</param>
        public void WriteStream<T>(Stream stream, object value)
        {
            WriteStream(stream, (T)value);
        }

        /// <summary>
        /// Convert a value to byte array and write it to the stream.
        /// </summary>
        /// <param name="stream">The stream to write to.</param>
        /// <param name="value">The value.</param>
        public void WriteStream(Stream stream, object value)
        {
            if (value is byte)
            {
                stream.WriteByte((byte)value);
            }
            else if (value is byte[])
            {
                var byteArray = (byte[])value;
                stream.Write(byteArray, 0, byteArray.Length);
            }
            else
            {
                byte[] result = Convert(value);
                stream.Write(result, 0, result.Length);
            }
        }

        /// <summary>
        /// Convert a value to byte array.
        /// </summary>
        /// <typeparam name="T">The type to cast the value to.</typeparam>
        /// <param name="value">The value.</param>
        /// <returns>The byte array.</returns>
        public byte[] Convert<T>(object value)
        {
            return Convert((T)value);
        }

        /// <summary>
        /// Convert a value to byte array.
        /// </summary>
        /// <param name="value">The value.</param>
        /// <returns>The byte array.</returns>
        public byte[] Convert(object value)
        {
            bool flipEndian = IsLittleEndian == false;

            if (value is bool)
            {
                return FromBoolean(flipEndian, (bool)value);
            }
            else if (value is int)
            {
                return FromInt32(flipEndian, (int)value);
            }
            else if (value is uint)
            {
                return FromUInt32(flipEndian, (uint)value);
            }
            else if (value is float)
            {
                return FromSingle(flipEndian, (float)value);
            }
            else if (value is double)
            {
                return FromDouble(flipEndian, (double)value);
            }
            else if (value is long)
            {
                return FromInt64(flipEndian, (long)value);
            }
            else if (value is ulong)
            {
                return FromUInt64(flipEndian, (ulong)value);
            }
            else if (value is short)
            {
                return FromInt16(flipEndian, (short)value);
            }
            else if (value is ushort)
            {
                return FromUInt16(flipEndian, (ushort)value);
            }
            else if (value is byte)
            {
                return FromByte(flipEndian, (byte)value);
            }
            else if (value is byte[])
            {
                return FromByteArray(flipEndian, (byte[])value);
            }
            else if (value is Guid)
            {
                return FromGuid(flipEndian, (Guid)value);
            }
            else if (value is char)
            {
                return FromChar(flipEndian, (char)value);
            }
            else if (value is char[])
            {
                return FromCharArray(flipEndian, (char[])value);
            }
            else if (value is string)
            {
                // Not supported for now.
                flipEndian = false;
                throw new NotSupportedException();
            }
            else if (value is IEnumerable)
            {
                flipEndian = false;
                List<byte> tmpList = new List<byte>();
                foreach (object element in (IEnumerable)value)
                {
                    byte[] elementBytes = Convert(element);
                    if (elementBytes == null)
                    {
                        // Failed, just return null.
                        return null;
                    }

                    tmpList.AddRange(elementBytes);
                }

                var result = tmpList.ToArray();

                // Flip byte order.
                FlipEndian(flipEndian, ref result);

                return result;
            }
            else
            {
                throw new NotSupportedException();
            }
        }

        /// <summary>
        /// Read a value from the binary stream.
        /// </summary>
        /// <typeparam name="T">The type of the value.</typeparam>
        /// <param name="stream">The stream to read from.</param>
        /// <returns>The value.</returns>
        public T ReadStream<T>(Stream stream)
        {
            bool flipEndian = IsLittleEndian == false;

            var buffer = new byte[16];
            if (typeof(T) == typeof(bool))
            {
                stream.Read(buffer, 0, 1);
                return (T)(object)ToBoolean(flipEndian, buffer);
            }
            else if (typeof(T) == typeof(int))
            {
                stream.Read(buffer, 0, 4);
                return (T)(object)ToInt32(flipEndian, buffer);
            }
            else if (typeof(T) == typeof(uint))
            {
                stream.Read(buffer, 0, 4);
                return (T)(object)ToUInt32(flipEndian, buffer);
            }
            else if (typeof(T) == typeof(float))
            {
                stream.Read(buffer, 0, 4);
                return (T)(object)ToSingle(flipEndian, buffer);
            }
            else if (typeof(T) == typeof(double))
            {
                stream.Read(buffer, 0, 8);
                return (T)(object)ToDouble(flipEndian, buffer);
            }
            else if (typeof(T) == typeof(long))
            {
                stream.Read(buffer, 0, 8);
                return (T)(object)ToInt64(flipEndian, buffer);
            }
            else if (typeof(T) == typeof(ulong))
            {
                stream.Read(buffer, 0, 8);
                return (T)(object)ToUInt64(flipEndian, buffer);
            }
            else if (typeof(T) == typeof(short))
            {
                stream.Read(buffer, 0, 2);
                return (T)(object)ToInt16(flipEndian, buffer);
            }
            else if (typeof(T) == typeof(ushort))
            {
                stream.Read(buffer, 0, 2);
                return (T)(object)ToUInt16(flipEndian, buffer);
            }
            else if (typeof(T) == typeof(byte))
            {
                stream.Read(buffer, 0, 1);
                return (T)(object)ToByte(flipEndian, buffer);
            }
            else if (typeof(T) == typeof(Guid))
            {
                stream.Read(buffer, 0, 16);
                return (T)(object)ToGuid(flipEndian, buffer);
            }
            else if (typeof(T) == typeof(char))
            {
                stream.Read(buffer, 0, 1);
                return (T)(object)ToChar(flipEndian, buffer);
            }
            else
            {
                throw new NotSupportedException();
            }
        }

        /// <summary>
        /// Convert the binary data in the buffer back to typed value.
        /// </summary>
        /// <param name="buffer">The binary data.</param>
        /// <param name="startIndex">The start index in the buffer.</param>
        /// <typeparam name="T">The type of the value.</typeparam>
        /// <returns>The typed value.</returns>
        public T ConvertBack<T>(byte[] buffer, int startIndex = 0)
        {
            bool flipEndian = IsLittleEndian == false;

            if (typeof(T) == typeof(bool))
            {
                return (T)(object)ToBoolean(flipEndian, buffer, startIndex);
            }
            else if (typeof(T) == typeof(int))
            {
                return (T)(object)ToInt32(flipEndian, buffer, startIndex);
            }
            else if (typeof(T) == typeof(uint))
            {
                return (T)(object)ToUInt32(flipEndian, buffer, startIndex);
            }
            else if (typeof(T) == typeof(float))
            {
                return (T)(object)ToSingle(flipEndian, buffer, startIndex);
            }
            else if (typeof(T) == typeof(double))
            {
                return (T)(object)ToDouble(flipEndian, buffer, startIndex);
            }
            else if (typeof(T) == typeof(long))
            {
                return (T)(object)ToInt64(flipEndian, buffer, startIndex);
            }
            else if (typeof(T) == typeof(ulong))
            {
                return (T)(object)ToUInt64(flipEndian, buffer, startIndex);
            }
            else if (typeof(T) == typeof(short))
            {
                return (T)(object)ToInt16(flipEndian, buffer, startIndex);
            }
            else if (typeof(T) == typeof(ushort))
            {
                return (T)(object)ToUInt16(flipEndian, buffer, startIndex);
            }
            else if (typeof(T) == typeof(byte))
            {
                return (T)(object)ToByte(flipEndian, buffer, startIndex);
            }
            else if (typeof(T) == typeof(Guid))
            {
                return (T)(object)ToGuid(flipEndian, buffer, startIndex);
            }
            else if (typeof(T) == typeof(char))
            {
                return (T)(object)ToChar(flipEndian, buffer, startIndex);
            }
            else
            {
                throw new NotSupportedException();
            }
        }

        #region Conversion methods for specific types

        /// <summary>
        /// Convert a value to byte array.
        /// </summary>
        /// <param name="flipEndian">True to flip endian.</param>
        /// <param name="value">The value.</param>
        /// <returns>The byte array.</returns>
        public byte[] FromBoolean(bool flipEndian, bool value)
        {
            byte[] result = BitConverter.GetBytes((bool)value);
            FlipEndian(flipEndian, ref result);
            return result;
        }

        /// <summary>
        /// Convert a byte array back to typed value.
        /// </summary>
        /// <param name="buffer">The buffer containing the binary data.</param>
        /// <param name="startIndex">The start index of the data in the buffer.</param>
        /// <returns>The converted value.</returns>
        public bool ToBoolean(byte[] buffer, int startIndex = 0)
        {
            return ToBoolean(IsLittleEndian == false, buffer, startIndex);
        }

        /// <summary>
        /// Convert a byte array back to typed value.
        /// </summary>
        /// <param name="flipEndian">True to flip endian.</param>
        /// <param name="buffer">The buffer containing the binary data.</param>
        /// <param name="startIndex">The start index of the data in the buffer.</param>
        /// <returns>The converted value.</returns>
        public bool ToBoolean(bool flipEndian, byte[] buffer, int startIndex = 0)
        {
            // Boolean has only one byte, no need to flip.
            return BitConverter.ToBoolean(buffer, startIndex);
        }

        /// <summary>
        /// Convert a value to byte array.
        /// </summary>
        /// <param name="flipEndian">True to flip endian.</param>
        /// <param name="value">The value.</param>
        /// <returns>The byte array.</returns>
        public byte[] FromInt16(bool flipEndian, short value)
        {
            byte[] result = BitConverter.GetBytes((short)value);
            FlipEndian(flipEndian, ref result);
            return result;
        }

        /// <summary>
        /// Convert a byte array back to typed value.
        /// </summary>
        /// <param name="buffer">The buffer containing the binary data.</param>
        /// <param name="startIndex">The start index of the data in the buffer.</param>
        /// <returns>The converted value.</returns>
        public short ToInt16(byte[] buffer, int startIndex = 0)
        {
            return ToInt16(IsLittleEndian == false, buffer, startIndex);
        }

        /// <summary>
        /// Convert a byte array back to typed value.
        /// </summary>
        /// <param name="flipEndian">True to flip endian.</param>
        /// <param name="buffer">The buffer containing the binary data.</param>
        /// <param name="startIndex">The start index of the data in the buffer.</param>
        /// <returns>The converted value.</returns>
        public short ToInt16(bool flipEndian, byte[] buffer, int startIndex = 0)
        {
            byte[] bytes = FlipEndian(flipEndian, buffer, startIndex, 2);
            return BitConverter.ToInt16(bytes, 0);
        }

        /// <summary>
        /// Convert a value to byte array.
        /// </summary>
        /// <param name="flipEndian">True to flip endian.</param>
        /// <param name="value">The value.</param>
        /// <returns>The byte array.</returns>
        public byte[] FromUInt16(bool flipEndian, ushort value)
        {
            byte[] result = BitConverter.GetBytes((ushort)value);
            FlipEndian(flipEndian, ref result);
            return result;
        }

        /// <summary>
        /// Convert a byte array back to typed value.
        /// </summary>
        /// <param name="buffer">The buffer containing the binary data.</param>
        /// <param name="startIndex">The start index of the data in the buffer.</param>
        /// <returns>The converted value.</returns>
        public ushort ToUInt16(byte[] buffer, int startIndex = 0)
        {
            return ToUInt16(IsLittleEndian == false, buffer, startIndex);
        }

        /// <summary>
        /// Convert a byte array back to typed value.
        /// </summary>
        /// <param name="flipEndian">True to flip endian.</param>
        /// <param name="buffer">The buffer containing the binary data.</param>
        /// <param name="startIndex">The start index of the data in the buffer.</param>
        /// <returns>The converted value.</returns>
        public ushort ToUInt16(bool flipEndian, byte[] buffer, int startIndex = 0)
        {
            byte[] bytes = FlipEndian(flipEndian, buffer, startIndex, 2);
            return BitConverter.ToUInt16(bytes, 0);
        }

        /// <summary>
        /// Convert a value to byte array.
        /// </summary>
        /// <param name="flipEndian">True to flip endian.</param>
        /// <param name="value">The value.</param>
        /// <returns>The byte array.</returns>
        public byte[] FromInt32(bool flipEndian, int value)
        {
            byte[] result = BitConverter.GetBytes((int)value);
            FlipEndian(flipEndian, ref result);
            return result;
        }

        /// <summary>
        /// Convert a byte array back to typed value.
        /// </summary>
        /// <param name="buffer">The buffer containing the binary data.</param>
        /// <param name="startIndex">The start index of the data in the buffer.</param>
        /// <returns>The converted value.</returns>
        public int ToInt32(byte[] buffer, int startIndex = 0)
        {
            return ToInt32(IsLittleEndian == false, buffer, startIndex);
        }

        /// <summary>
        /// Convert a byte array back to typed value.
        /// </summary>
        /// <param name="flipEndian">True to flip endian.</param>
        /// <param name="buffer">The buffer containing the binary data.</param>
        /// <param name="startIndex">The start index of the data in the buffer.</param>
        /// <returns>The converted value.</returns>
        public int ToInt32(bool flipEndian, byte[] buffer, int startIndex = 0)
        {
            byte[] bytes = FlipEndian(flipEndian, buffer, startIndex, 4);
            return BitConverter.ToInt32(bytes, 0);
        }

        /// <summary>
        /// Convert a value to byte array.
        /// </summary>
        /// <param name="flipEndian">True to flip endian.</param>
        /// <param name="value">The value.</param>
        /// <returns>The byte array.</returns>
        public byte[] FromUInt32(bool flipEndian, uint value)
        {
            byte[] result = BitConverter.GetBytes((uint)value);
            FlipEndian(flipEndian, ref result);
            return result;
        }

        /// <summary>
        /// Convert a byte array back to typed value.
        /// </summary>
        /// <param name="buffer">The buffer containing the binary data.</param>
        /// <param name="startIndex">The start index of the data in the buffer.</param>
        /// <returns>The converted value.</returns>
        public uint ToUInt32(byte[] buffer, int startIndex = 0)
        {
            return ToUInt32(IsLittleEndian == false, buffer, startIndex);
        }

        /// <summary>
        /// Convert a byte array back to typed value.
        /// </summary>
        /// <param name="flipEndian">True to flip endian.</param>
        /// <param name="buffer">The buffer containing the binary data.</param>
        /// <param name="startIndex">The start index of the data in the buffer.</param>
        /// <returns>The converted value.</returns>
        public uint ToUInt32(bool flipEndian, byte[] buffer, int startIndex = 0)
        {
            byte[] bytes = FlipEndian(flipEndian, buffer, startIndex, 4);
            return BitConverter.ToUInt32(bytes, 0);
        }

        /// <summary>
        /// Convert a value to byte array.
        /// </summary>
        /// <param name="flipEndian">True to flip endian.</param>
        /// <param name="value">The value.</param>
        /// <returns>The byte array.</returns>
        public byte[] FromInt64(bool flipEndian, long value)
        {
            byte[] result = BitConverter.GetBytes((long)value);
            FlipEndian(flipEndian, ref result);
            return result;
        }

        /// <summary>
        /// Convert a byte array back to typed value.
        /// </summary>
        /// <param name="buffer">The buffer containing the binary data.</param>
        /// <param name="startIndex">The start index of the data in the buffer.</param>
        /// <returns>The converted value.</returns>
        public long ToInt64(byte[] buffer, int startIndex = 0)
        {
            return ToInt64(IsLittleEndian == false, buffer, startIndex);
        }

        /// <summary>
        /// Convert a byte array back to typed value.
        /// </summary>
        /// <param name="flipEndian">True to flip endian.</param>
        /// <param name="buffer">The buffer containing the binary data.</param>
        /// <param name="startIndex">The start index of the data in the buffer.</param>
        /// <returns>The converted value.</returns>
        public long ToInt64(bool flipEndian, byte[] buffer, int startIndex = 0)
        {
            byte[] bytes = FlipEndian(flipEndian, buffer, startIndex, 8);
            return BitConverter.ToInt64(bytes, 0);
        }

        /// <summary>
        /// Convert a value to byte array.
        /// </summary>
        /// <param name="flipEndian">True to flip endian.</param>
        /// <param name="value">The value.</param>
        /// <returns>The byte array.</returns>
        public byte[] FromUInt64(bool flipEndian, ulong value)
        {
            byte[] result = BitConverter.GetBytes((ulong)value);
            FlipEndian(flipEndian, ref result);
            return result;
        }

        /// <summary>
        /// Convert a byte array back to typed value.
        /// </summary>
        /// <param name="buffer">The buffer containing the binary data.</param>
        /// <param name="startIndex">The start index of the data in the buffer.</param>
        /// <returns>The converted value.</returns>
        public ulong ToUInt64(byte[] buffer, int startIndex = 0)
        {
            return ToUInt64(IsLittleEndian == false, buffer, startIndex);
        }

        /// <summary>
        /// Convert a byte array back to typed value.
        /// </summary>
        /// <param name="flipEndian">True to flip endian.</param>
        /// <param name="buffer">The buffer containing the binary data.</param>
        /// <param name="startIndex">The start index of the data in the buffer.</param>
        /// <returns>The converted value.</returns>
        public ulong ToUInt64(bool flipEndian, byte[] buffer, int startIndex = 0)
        {
            byte[] bytes = FlipEndian(flipEndian, buffer, startIndex, 8);
            return BitConverter.ToUInt64(bytes, 0);
        }

        /// <summary>
        /// Convert a value to byte array.
        /// </summary>
        /// <param name="flipEndian">True to flip endian.</param>
        /// <param name="value">The value.</param>
        /// <returns>The byte array.</returns>
        public byte[] FromSingle(bool flipEndian, float value)
        {
            byte[] result = BitConverter.GetBytes((float)value);
            FlipEndian(flipEndian, ref result);
            return result;
        }

        /// <summary>
        /// Convert a byte array back to typed value.
        /// </summary>
        /// <param name="buffer">The buffer containing the binary data.</param>
        /// <param name="startIndex">The start index of the data in the buffer.</param>
        /// <returns>The converted value.</returns>
        public float ToSingle(byte[] buffer, int startIndex = 0)
        {
            return ToSingle(IsLittleEndian == false, buffer, startIndex);
        }

        /// <summary>
        /// Convert a byte array back to typed value.
        /// </summary>
        /// <param name="flipEndian">True to flip endian.</param>
        /// <param name="buffer">The buffer containing the binary data.</param>
        /// <param name="startIndex">The start index of the data in the buffer.</param>
        /// <returns>The converted value.</returns>
        public float ToSingle(bool flipEndian, byte[] buffer, int startIndex = 0)
        {
            byte[] bytes = FlipEndian(flipEndian, buffer, startIndex, 4);
            return BitConverter.ToSingle(bytes, 0);
        }

        /// <summary>
        /// Convert a value to byte array.
        /// </summary>
        /// <param name="flipEndian">True to flip endian.</param>
        /// <param name="value">The value.</param>
        /// <returns>The byte array.</returns>
        public byte[] FromDouble(bool flipEndian, double value)
        {
            byte[] result = BitConverter.GetBytes((double)value);
            FlipEndian(flipEndian, ref result);
            return result;
        }

        /// <summary>
        /// Convert a byte array back to typed value.
        /// </summary>
        /// <param name="buffer">The buffer containing the binary data.</param>
        /// <param name="startIndex">The start index of the data in the buffer.</param>
        /// <returns>The converted value.</returns>
        public double ToDouble(byte[] buffer, int startIndex = 0)
        {
            return ToDouble(IsLittleEndian == false, buffer, startIndex);
        }

        /// <summary>
        /// Convert a byte array back to typed value.
        /// </summary>
        /// <param name="flipEndian">True to flip endian.</param>
        /// <param name="buffer">The buffer containing the binary data.</param>
        /// <param name="startIndex">The start index of the data in the buffer.</param>
        /// <returns>The converted value.</returns>
        public double ToDouble(bool flipEndian, byte[] buffer, int startIndex = 0)
        {
            byte[] bytes = FlipEndian(flipEndian, buffer, startIndex, 8);
            return BitConverter.ToDouble(bytes, 0);
        }

        /// <summary>
        /// Convert a value to byte array.
        /// </summary>
        /// <param name="flipEndian">True to flip endian.</param>
        /// <param name="value">The value.</param>
        /// <returns>The byte array.</returns>
        public byte[] FromByte(bool flipEndian, byte value)
        {
            return new byte[] { value };
        }

        /// <summary>
        /// Convert a byte array back to typed value.
        /// </summary>
        /// <param name="buffer">The buffer containing the binary data.</param>
        /// <param name="startIndex">The start index of the data in the buffer.</param>
        /// <returns>The converted value.</returns>
        public byte ToByte(byte[] buffer, int startIndex = 0)
        {
            return ToByte(IsLittleEndian == false, buffer, startIndex);
        }

        /// <summary>
        /// Convert a byte array back to typed value.
        /// </summary>
        /// <param name="flipEndian">True to flip endian.</param>
        /// <param name="buffer">The buffer containing the binary data.</param>
        /// <param name="startIndex">The start index of the data in the buffer.</param>
        /// <returns>The converted value.</returns>
        public byte ToByte(bool flipEndian, byte[] buffer, int startIndex = 0)
        {
            return buffer[startIndex];
        }

        /// <summary>
        /// Convert a value to byte array.
        /// </summary>
        /// <param name="flipEndian">True to flip endian.</param>
        /// <param name="value">The value.</param>
        /// <returns>The byte array.</returns>
        public byte[] FromChar(bool flipEndian, char value)
        {
            return new byte[] { (byte)value };
        }

        /// <summary>
        /// Convert a byte array back to typed value.
        /// </summary>
        /// <param name="buffer">The buffer containing the binary data.</param>
        /// <param name="startIndex">The start index of the data in the buffer.</param>
        /// <returns>The converted value.</returns>
        public char ToChar(byte[] buffer, int startIndex = 0)
        {
            return ToChar(IsLittleEndian == false, buffer, startIndex);
        }

        /// <summary>
        /// Convert a byte array back to typed value.
        /// </summary>
        /// <param name="flipEndian">True to flip endian.</param>
        /// <param name="buffer">The buffer containing the binary data.</param>
        /// <param name="startIndex">The start index of the data in the buffer.</param>
        /// <returns>The converted value.</returns>
        public char ToChar(bool flipEndian, byte[] buffer, int startIndex = 0)
        {
            return (char)buffer[startIndex];
        }

        /// <summary>
        /// Convert a value to byte array.
        /// </summary>
        /// <param name="flipEndian">True to flip endian.</param>
        /// <param name="value">The value.</param>
        /// <returns>The byte array.</returns>
        public byte[] FromGuid(bool flipEndian, Guid value)
        {
            return value.ToByteArray();
        }

        /// <summary>
        /// Convert a byte array back to typed value.
        /// </summary>
        /// <param name="buffer">The buffer containing the binary data.</param>
        /// <param name="startIndex">The start index of the data in the buffer.</param>
        /// <returns>The converted value.</returns>
        public Guid ToGuid(byte[] buffer, int startIndex = 0)
        {
            return ToGuid(IsLittleEndian == false, buffer, startIndex);
        }

        /// <summary>
        /// Convert a byte array back to typed value.
        /// </summary>
        /// <param name="flipEndian">True to flip endian.</param>
        /// <param name="buffer">The buffer containing the binary data.</param>
        /// <param name="startIndex">The start index of the data in the buffer.</param>
        /// <returns>The converted value.</returns>
        public Guid ToGuid(bool flipEndian, byte[] buffer, int startIndex = 0)
        {
            byte[] bytes = new byte[16];
            Array.Copy(buffer, startIndex, bytes, 0, 16);
            return new Guid(bytes);
        }

        /// <summary>
        /// Convert a value to byte array.
        /// </summary>
        /// <param name="flipEndian">True to flip endian.</param>
        /// <param name="value">The value.</param>
        /// <returns>The byte array.</returns>
        public byte[] FromByteArray(bool flipEndian, byte[] value)
        {
            return value;
        }

        /// <summary>
        /// Convert a byte array back to typed value.
        /// </summary>
        /// <param name="arraySize">The array element count.</param>
        /// <param name="buffer">The buffer containing the binary data.</param>
        /// <param name="startIndex">The start index of the data in the buffer.</param>
        /// <returns>The converted value.</returns>
        public byte[] ToByteArray(int arraySize, byte[] buffer, int startIndex = 0)
        {
            return ToByteArray(IsLittleEndian == false, arraySize, buffer, startIndex);
        }

        /// <summary>
        /// Convert a byte array back to typed value.
        /// </summary>
        /// <param name="flipEndian">True to flip endian.</param>
        /// <param name="arraySize">The array element count.</param>
        /// <param name="buffer">The buffer containing the binary data.</param>
        /// <param name="startIndex">The start index of the data in the buffer.</param>
        /// <returns>The converted value.</returns>
        public byte[] ToByteArray(
            bool flipEndian,
            int arraySize,
            byte[] buffer,
            int startIndex = 0)
        {
            var data = new byte[arraySize];
            Array.Copy(buffer, startIndex, data, 0, arraySize);
            return data;
        }

        /// <summary>
        /// Convert a value to byte array.
        /// </summary>
        /// <param name="flipEndian">True to flip endian.</param>
        /// <param name="value">The value.</param>
        /// <returns>The byte array.</returns>
        public byte[] FromCharArray(bool flipEndian, char[] value)
        {
            var data = new byte[value.Length];
            for (int i = 0; i < value.Length; ++i)
            {
                data[i] = (byte)value[i];
            }

            return data;
        }

        /// <summary>
        /// Convert a byte array back to typed value.
        /// </summary>
        /// <param name="arraySize">The array element count.</param>
        /// <param name="buffer">The buffer containing the binary data.</param>
        /// <param name="startIndex">The start index of the data in the buffer.</param>
        /// <returns>The converted value.</returns>
        public char[] ToCharArray(int arraySize, byte[] buffer, int startIndex = 0)
        {
            return ToCharArray(IsLittleEndian == false, arraySize, buffer, startIndex);
        }

        /// <summary>
        /// Convert a byte array back to typed value.
        /// </summary>
        /// <param name="flipEndian">True to flip endian.</param>
        /// <param name="arraySize">The array element count.</param>
        /// <param name="buffer">The buffer containing the binary data.</param>
        /// <param name="startIndex">The start index of the data in the buffer.</param>
        /// <returns>The converted value.</returns>
        public char[] ToCharArray(
            bool flipEndian,
            int arraySize,
            byte[] buffer,
            int startIndex = 0)
        {
            var data = new char[arraySize];
            for (int i = 0; i < arraySize; ++i)
            {
                data[i] = (char)buffer[startIndex + i];
            }

            return data;
        }

        #endregion

        /// <summary>
        /// Utility method for flipping byte endian.
        /// </summary>
        /// <param name="data">The data to flip.</param>
        public void FlipEndian(ref byte[] data)
        {
            int length = data.Length;
            int helfLength = length / 2;
            for (int i = 0; i < helfLength; ++i)
            {
                byte tmp = data[i];
                data[i] = data[length - i - 1];
                data[length - i - 1] = tmp;
            }
        }

        /// <summary>
        /// Utility method for flipping byte endian.
        /// </summary>
        /// <param name="flip">True to flip endian.</param>
        /// <param name="data">The data to flip.</param>
        private void FlipEndian(bool flip, ref byte[] data)
        {
            if (flip == false)
            {
                return;
            }

            int length = data.Length;
            int helfLength = length / 2;
            for (int i = 0; i < helfLength; ++i)
            {
                byte tmp = data[i];
                data[i] = data[length - i - 1];
                data[length - i - 1] = tmp;
            }
        }

        /// <summary>
        /// Utility method for flipping byte endian.
        /// </summary>
        /// <param name="flip">True to flip endian.</param>
        /// <param name="buffer">The data to flip.</param>
        /// <param name="startIndex">The start index in the buffer.</param>
        /// <param name="size">Number of bytes to flip.</param>
        /// <returns>The flipped data.</returns>
        private byte[] FlipEndian(
            bool flip,
            byte[] buffer,
            int startIndex,
            int size)
        {
            byte[] data = new byte[size];
            Array.Copy(buffer, startIndex, data, 0, size);

            if (flip == true)
            {
                // Flip the byte order.
                int length = data.Length;
                int helfLength = length / 2;
                for (int i = 0; i < helfLength; ++i)
                {
                    byte tmp = data[i];
                    data[i] = data[length - i - 1];
                    data[length - i - 1] = tmp;
                }
            }

            return data;
        }
    }
}
