﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Xml;

using EffectMaker.Foundation.Extensions;

namespace EffectMaker.Foundation.Serialization
{
    /// <summary>
    /// Holds data needed for XML document serialization.
    /// </summary>
    public class XmlDocSerializationContext
    {
        /// <summary>A stack storing the XML node that is being serialized/deserialized.</summary>
        private Stack<XmlNode> currNodeStack = new Stack<XmlNode>();

        /// <summary>A dictionary that maps the type name to the types.</summary>
        private Dictionary<string, Type> extraTypeMap = null;

        /// <summary>The IXmlDocSerializable pre-serialization event handlers.</summary>
        private Dictionary<Type, List<Action<IXmlDocSerializable>>> preSerializeHandlers =
            new Dictionary<Type, List<Action<IXmlDocSerializable>>>();

        /// <summary>The IXmlDocSerializable post-serialization event handlers.</summary>
        private Dictionary<Type, List<Action<IXmlDocSerializable>>> postSerializeHandlers =
            new Dictionary<Type, List<Action<IXmlDocSerializable>>>();

        /// <summary>The IXmlDocSerializable post-deserialization event handlers.</summary>
        private Dictionary<Type, List<Action<IXmlDocSerializable>>> postDeserializeHandlers =
            new Dictionary<Type, List<Action<IXmlDocSerializable>>>();

        /// <summary>
        /// Constructor.
        /// </summary>
        /// <param name="doc">The XML document for serialization/deserialization.</param>
        public XmlDocSerializationContext(XmlDocument doc)
        {
            this.XmlDocument = doc;
            this.PushCurrentNode(doc.DocumentElement);
        }

        /// <summary>
        /// Constructor.
        /// </summary>
        /// <param name="doc">The XML document for serialization/deserialization.</param>
        /// <param name="extraTypes">The extra types for serialization/deserialization.</param>
        public XmlDocSerializationContext(
            XmlDocument doc,
            IEnumerable<Type> extraTypes) : this(doc)
        {
            if (extraTypes.IsNullOrEmpty() == false)
            {
                this.extraTypeMap = new Dictionary<string, Type>();
                extraTypes.ForEach(t => this.extraTypeMap.Add(t.Name, t));
            }
        }

        /// <summary>
        /// Constructor.
        /// </summary>
        /// <param name="doc">The XML document for serialization/deserialization.</param>
        /// <param name="extraTypes">The extra types for serialization/deserialization.</param>
        public XmlDocSerializationContext(
            XmlDocument doc,
            Dictionary<string, Type> extraTypes) : this(doc)
        {
            this.extraTypeMap = extraTypes;
        }

        /// <summary>
        /// Get or set the XML document that is being serialized/deserialized.
        /// </summary>
        public XmlDocument XmlDocument { get; private set; }

        /// <summary>
        /// Get the XML node that is currently being processed.
        /// </summary>
        public XmlNode CurrentNode
        {
            get { return this.currNodeStack.Peek(); }
        }

        /// <summary>
        /// Push a XML node to process.
        /// </summary>
        /// <param name="node">The XML node.</param>
        public void PushCurrentNode(XmlNode node)
        {
            this.currNodeStack.Push(node);
        }

        /// <summary>
        /// Pop the XML node out so we can process the previous node.
        /// </summary>
        /// <returns>The XML node that is removed from the stack.</returns>
        public XmlNode PopCurrentNode()
        {
            return this.currNodeStack.Pop();
        }

        /// <summary>
        /// Find the extra type that is mapped to the given name.
        /// </summary>
        /// <param name="name">The name of the mapped type.</param>
        /// <returns>The extra type or null if not found.</returns>
        public Type FindExtraType(string name)
        {
            Type type;
            if (this.extraTypeMap == null ||
                this.extraTypeMap.TryGetValue(name, out type) == false)
            {
                return null;
            }

            return type;
        }

        /// <summary>
        /// Add pre-serialize handler for the specified IXmlDocSerializable type.
        /// The handler will be called before an instance of the specified type
        /// is being serialized.
        /// </summary>
        /// <typeparam name="T">
        /// The type which implements IXmlDocSerializable, to trigger the event.
        /// </typeparam>
        /// <param name="handler">The event handler.</param>
        public void AddPreSerializeHandler<T>(Action<IXmlDocSerializable> handler)
            where T : IXmlDocSerializable
        {
            List<Action<IXmlDocSerializable>> handlers;
            if (this.preSerializeHandlers.TryGetValue(typeof(T), out handlers) == false)
            {
                handlers = new List<Action<IXmlDocSerializable>>();
                this.preSerializeHandlers.Add(typeof(T), handlers);
            }

            handlers.Add(handler);
        }

        /// <summary>
        /// Add post-serialize handler for the specified IXmlDocSerializable type.
        /// The handler will be called after an instance of the specified type
        /// is being serialized.
        /// </summary>
        /// <typeparam name="T">
        /// The type which implements IXmlDocSerializable, to trigger the event.
        /// </typeparam>
        /// <param name="handler">The event handler.</param>
        public void AddPostSerializeHandler<T>(Action<IXmlDocSerializable> handler)
            where T : IXmlDocSerializable
        {
            List<Action<IXmlDocSerializable>> handlers;
            if (this.postSerializeHandlers.TryGetValue(typeof(T), out handlers) == false)
            {
                handlers = new List<Action<IXmlDocSerializable>>();
                this.postSerializeHandlers.Add(typeof(T), handlers);
            }

            handlers.Add(handler);
        }

        /// <summary>
        /// Add post-deserialize handler for the specified IXmlDocSerializable type.
        /// The handler will be called after an instance of the specified type
        /// is being deserialized.
        /// </summary>
        /// <typeparam name="T">
        /// The type which implements IXmlDocSerializable, to trigger the event.
        /// </typeparam>
        /// <param name="handler">The event handler.</param>
        public void AddPostDeserializeHandler<T>(Action<IXmlDocSerializable> handler)
            where T : IXmlDocSerializable
        {
            List<Action<IXmlDocSerializable>> handlers;
            if (this.postDeserializeHandlers.TryGetValue(typeof(T), out handlers) == false)
            {
                handlers = new List<Action<IXmlDocSerializable>>();
                this.postDeserializeHandlers.Add(typeof(T), handlers);
            }

            handlers.Add(handler);
        }

        /// <summary>
        /// Called right before IXmlDocSerializables being serialized.
        /// </summary>
        /// <param name="instance">The IXmlDocSerializable instance that is to be serialized.</param>
        public void ReportPreSerialize(IXmlDocSerializable instance)
        {
            if (instance == null)
            {
                return;
            }

            // Find the handlers for the instance type.
            List<Action<IXmlDocSerializable>> handlers;
            if (this.preSerializeHandlers.TryGetValue(instance.GetType(), out handlers) == false)
            {
                return;
            }

            // Call these handlers.
            handlers.ForEach(h => h(instance));
        }

        /// <summary>
        /// Called right after IXmlDocSerializables being serialized.
        /// </summary>
        /// <param name="instance">The IXmlDocSerializable instance that is serialized.</param>
        public void ReportPostSerialize(IXmlDocSerializable instance)
        {
            if (instance == null)
            {
                return;
            }

            // Find the handlers for the instance type.
            List<Action<IXmlDocSerializable>> handlers;
            if (this.postSerializeHandlers.TryGetValue(instance.GetType(), out handlers) == false)
            {
                return;
            }

            // Call these handlers.
            handlers.ForEach(h => h(instance));
        }

        /// <summary>
        /// Called right after IXmlDocSerializables being deserialized.
        /// </summary>
        /// <param name="instance">The IXmlDocSerializable instance that is deserialized.</param>
        public void ReportPostDeserialize(IXmlDocSerializable instance)
        {
            if (instance == null)
            {
                return;
            }

            // Find the handlers for the instance type.
            List<Action<IXmlDocSerializable>> handlers;
            if (this.postDeserializeHandlers.TryGetValue(instance.GetType(), out handlers) == false)
            {
                return;
            }

            // Call these handlers.
            handlers.ForEach(h => h(instance));
        }
    }
}
