﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Concurrent;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Runtime.Remoting.Messaging;
using System.Text;
using System.Xml;
using System.Xml.Serialization;
using EffectMaker.Foundation.Extensions;
using EffectMaker.Foundation.Log;

namespace EffectMaker.Foundation.Serialization
{
    /// <summary>
    /// Utility class for serialization and deserialization.
    /// </summary>
    public abstract class SerializationHelper
    {
        /// <summary>A cache for XML serializers.</summary>
        private static readonly ConcurrentDictionary<Type, XmlSerializer> SerializerCache =
            new ConcurrentDictionary<Type, XmlSerializer>();

        /// <summary>A cache for XML serializers with extra types.</summary>
        private static readonly ConcurrentDictionary<SerializerKey, XmlSerializer> SerializerIncludeExtraTypesCache =
            new ConcurrentDictionary<SerializerKey, XmlSerializer>(new SerializerKeyComparer());

        /// <summary>
        /// Event triggers when the serialization helper needs to collect extra types for
        /// deserialization.
        /// </summary>
        public static event EventHandler<RequestExtraTypesEventArgs> RequestExtraTypes;

        /// <summary>
        /// Get extra types for deserialization.
        /// </summary>
        /// <returns>The array of extra types.</returns>
        public static Type[] GetExtraTypes()
        {
            var args = new RequestExtraTypesEventArgs();

            if (RequestExtraTypes != null)
            {
                RequestExtraTypes(null, args);
            }

            return args.ExtraTypes;
        }

        /// <summary>
        /// Deserializes an instance from a given data stream.
        /// </summary>
        /// <typeparam name="T">The type of object to deserialize.</typeparam>
        /// <param name="stream">The stream that contains the serialized instance data.</param>
        /// <param name="includeExtraTypes">True to include extra types for deserialization.</param>
        /// <returns>Returns the deserialized instance, or the default type value.</returns>
        public static T Load<T>(Stream stream, bool includeExtraTypes = false)
        {
            try
            {
                var serializer = GetSerializer(typeof(T), includeExtraTypes);

                return (T)serializer.Deserialize(stream);
            }
            catch (Exception ex)
            {
                Logger.Log(LogLevels.Error, "SerializationHelper.Load : Failed loading {0} from stream.", typeof(T).Name);
                Logger.Log(LogLevels.Error, ex.Message);
                return default(T);
            }
        }

        /// <summary>
        /// Deserializes an instance from a given XML reader.
        /// </summary>
        /// <typeparam name="T">The type of object to deserialize.</typeparam>
        /// <param name="reader">The reader to read the data from.</param>
        /// <param name="includeExtraTypes">True to include extra types for deserialization.</param>
        /// <returns>Returns the deserialized instance, or the default type value.</returns>
        public static T Load<T>(XmlReader reader, bool includeExtraTypes = false)
        {
            try
            {
                var serializer = GetSerializer(typeof(T), includeExtraTypes);
                return (T)serializer.Deserialize(reader);
            }
            catch (Exception ex)
            {
                Logger.Log(LogLevels.Error, "SerializationHelper.Load : Failed loading {0} from XML reader.", typeof(T).Name);
                Logger.Log(LogLevels.Error, ex.Message);
                return default(T);
            }
        }

        /// <summary>
        /// Deserializes an instance from a given XML reader.
        /// </summary>
        /// <param name="type">The type of the object to deserialize to.</param>
        /// <param name="reader">The reader to read the data from.</param>
        /// <param name="includeExtraTypes">True to include extra types for deserialization.</param>
        /// <returns>Returns the deserialized instance, or the default type value.</returns>
        public static object Load(Type type, XmlReader reader, bool includeExtraTypes = false)
        {
            try
            {
                var serializer = GetSerializer(type, includeExtraTypes);
                return serializer.Deserialize(reader);
            }
            catch (Exception ex)
            {
                Logger.Log(LogLevels.Error, "SerializationHelper.Load : Failed loading {0} from XML reader.", type.Name);
                Logger.Log(LogLevels.Error, ex.Message);
                return null;
            }
        }

        /// <summary>
        /// Deserializes an instance from a file.
        /// </summary>
        /// <typeparam name="T">The type of object to deserialize.</typeparam>
        /// <param name="filePath">The file path.</param>
        /// <param name="includeExtraTypes">True to include extra types for deserialization.</param>
        /// <returns>Returns the deserialized instance, or the default type value.</returns>
        public static T Load<T>(string filePath, bool includeExtraTypes = false)
        {
            byte[] buffer = null;

            try
            {
                // First load all contents from the file, to speed up loading.
                using (var fs = new FileStream(filePath, FileMode.Open, FileAccess.Read))
                {
                    buffer = new byte[fs.Length];
                    fs.Read(buffer, 0, (int)fs.Length);
                }
            }
            catch (Exception ex)
            {
                Logger.Log(LogLevels.Error, "SerializationHelper.Load : Failed loading file from {0}", filePath);
                Logger.Log(LogLevels.Error, ex.Message);
                return default(T);
            }

            try
            {
                var serializer = GetSerializer(typeof(T), includeExtraTypes);

                using (var ms = new MemoryStream(buffer))
                {
                    return (T)serializer.Deserialize(ms);
                }
            }
            catch (Exception ex)
            {
                Logger.Log(LogLevels.Error, "SerializationHelper.Load : Failed deserializing the file contents from {0}", filePath);
                Logger.Log(LogLevels.Error, ex.Message);
                return default(T);
            }
        }

        /// <summary>
        /// Deserializes an instance from the specified file.
        /// </summary>
        /// <typeparam name="T">The type of object to deserialize.</typeparam>
        /// <param name="filePath">The path of the file that contains the serialized instance data.</param>
        /// <param name="includeExtraTypes">True to include extra types for deserialization.</param>
        /// <returns>Returns the deserialized instance, or the default type value.</returns>
        public static T LoadXmlDocSerializable<T>(
            string filePath,
            bool includeExtraTypes = false) where T : IXmlDocSerializable
        {
            byte[] buffer = null;

            try
            {
                // First load all contents from the file, to speed up loading.
                using (var fs = new FileStream(filePath, FileMode.Open, FileAccess.Read))
                {
                    buffer = new byte[fs.Length];
                    fs.Read(buffer, 0, (int)fs.Length);
                }
            }
            catch (Exception ex)
            {
                Logger.Log(LogLevels.Error, "SerializationHelper.Load : Failed loading file from {0}", filePath);
                Logger.Log(LogLevels.Error, ex.Message);
                return default(T);
            }

            Type[] extraTypes = null;
            if (includeExtraTypes == true)
            {
                extraTypes = GetExtraTypes();
            }

            try
            {
                using (var ms = new MemoryStream(buffer))
                {
                    var serializer = new XmlDocSerializer();
                    return serializer.Deserialize<T>(ms, extraTypes);
                }
            }
            catch (Exception ex)
            {
                Logger.Log(LogLevels.Error, "SerializationHelper.Load : Failed loading {0} from {1}.", typeof(T).Name, filePath);
                Logger.Log(LogLevels.Error, ex.Message);
                return default(T);
            }
        }

        /// <summary>
        /// Deserializes an instance from a given XML reader.
        /// </summary>
        /// <typeparam name="T">The type of object to deserialize.</typeparam>
        /// <param name="stream">The stream that contains the serialized instance data.</param>
        /// <param name="includeExtraTypes">True to include extra types for deserialization.</param>
        /// <returns>Returns the deserialized instance, or the default type value.</returns>
        public static T LoadXmlDocSerializable<T>(
            Stream stream,
            bool includeExtraTypes = false) where T : IXmlDocSerializable
        {
            Type[] extraTypes = null;
            if (includeExtraTypes == true)
            {
                extraTypes = GetExtraTypes();
            }

            try
            {
                var serializer = new XmlDocSerializer();
                return serializer.Deserialize<T>(stream, extraTypes);
            }
            catch (Exception ex)
            {
                Logger.Log(LogLevels.Error, "SerializationHelper.Load : Failed loading {0} from stream.", typeof(T).Name);
                Logger.Log(LogLevels.Error, ex.Message);
                return default(T);
            }
        }

        /// <summary>
        /// Deserializes an instance from a given XML reader.
        /// </summary>
        /// <typeparam name="T">The type of object to deserialize.</typeparam>
        /// <param name="xmlDocument">The XML document to read the data from.</param>
        /// <param name="includeExtraTypes">True to include extra types for deserialization.</param>
        /// <returns>Returns the deserialized instance, or the default type value.</returns>
        public static T LoadXmlDocSerializable<T>(
            XmlDocument xmlDocument,
            bool includeExtraTypes = false) where T : IXmlDocSerializable
        {
            Type[] extraTypes = null;
            if (includeExtraTypes == true)
            {
                extraTypes = GetExtraTypes();
            }

            try
            {
                var serializer = new XmlDocSerializer();
                return serializer.Deserialize<T>(xmlDocument, extraTypes);
            }
            catch (Exception ex)
            {
                Logger.Log(LogLevels.Error, "SerializationHelper.Load : Failed loading {0} from XML document.", typeof(T).Name);
                Logger.Log(LogLevels.Error, ex.Message);
                return default(T);
            }
        }

        /// <summary>
        /// Deserializes an instance from the specified file.
        /// </summary>
        /// <param name="type">The type of the object to deserialize to.</param>
        /// <param name="filePath">The path of the file that contains the serialized instance data.</param>
        /// <param name="includeExtraTypes">True to include extra types for deserialization.</param>
        /// <returns>Returns the deserialized instance, or the default type value.</returns>
        public static IXmlDocSerializable LoadXmlDocSerializable(
            Type type,
            string filePath,
            bool includeExtraTypes = false)
        {
            byte[] buffer = null;

            try
            {
                // First load all contents from the file, to speed up loading.
                using (var fs = new FileStream(filePath, FileMode.Open, FileAccess.Read))
                {
                    buffer = new byte[fs.Length];
                    fs.Read(buffer, 0, (int)fs.Length);
                }
            }
            catch (Exception ex)
            {
                Logger.Log(LogLevels.Error, "SerializationHelper.Load : Failed loading file from {0}", filePath);
                Logger.Log(LogLevels.Error, ex.Message);
                return null;
            }

            Type[] extraTypes = null;
            if (includeExtraTypes == true)
            {
                extraTypes = GetExtraTypes();
            }

            try
            {
                using (var ms = new MemoryStream(buffer))
                {
                    var serializer = new XmlDocSerializer();
                    return serializer.Deserialize(type, ms, extraTypes);
                }
            }
            catch (Exception ex)
            {
                Logger.Log(LogLevels.Error, "SerializationHelper.Load : Failed loading {0} from {1}.", type.Name, filePath);
                Logger.Log(LogLevels.Error, ex.Message);
                return null;
            }
        }

        /// <summary>
        /// Deserializes an instance from a given XML reader.
        /// </summary>
        /// <param name="type">The type of the object to deserialize to.</param>
        /// <param name="stream">The stream that contains the serialized instance data.</param>
        /// <param name="includeExtraTypes">True to include extra types for deserialization.</param>
        /// <returns>Returns the deserialized instance, or the default type value.</returns>
        public static IXmlDocSerializable LoadXmlDocSerializable(
            Type type,
            Stream stream,
            bool includeExtraTypes = false)
        {
            Type[] extraTypes = null;
            if (includeExtraTypes == true)
            {
                extraTypes = GetExtraTypes();
            }

            try
            {
                var serializer = new XmlDocSerializer();
                return serializer.Deserialize(type, stream, extraTypes);
            }
            catch (Exception ex)
            {
                Logger.Log(LogLevels.Error, "SerializationHelper.Load : Failed loading {0} from stream.", type.Name);
                Logger.Log(LogLevels.Error, ex.Message);
                return null;
            }
        }

        /// <summary>
        /// Deserializes an instance from a given XML reader.
        /// </summary>
        /// <param name="type">The type of the object to deserialize to.</param>
        /// <param name="xmlDocument">The XML document to read the data from.</param>
        /// <param name="includeExtraTypes">True to include extra types for deserialization.</param>
        /// <returns>Returns the deserialized instance, or the default type value.</returns>
        public static IXmlDocSerializable LoadXmlDocSerializable(
            Type type,
            XmlDocument xmlDocument,
            bool includeExtraTypes = false)
        {
            Type[] extraTypes = null;
            if (includeExtraTypes == true)
            {
                extraTypes = GetExtraTypes();
            }

            try
            {
                var serializer = new XmlDocSerializer();
                return serializer.Deserialize(type, xmlDocument, extraTypes);
            }
            catch (Exception ex)
            {
                Logger.Log(LogLevels.Error, "SerializationHelper.Load : Failed loading {0} from XML document.", type.Name);
                Logger.Log(LogLevels.Error, ex.Message);
                return null;
            }
        }

        /// <summary>
        /// Serializes an instance to a given stream.
        /// </summary>
        /// <typeparam name="T">The type of object to serialize.</typeparam>
        /// <param name="instance">The instance to serialize to stream.</param>
        /// <param name="stream">The stream to serialize the instance to.</param>
        /// <param name="includeExtraTypes">True to include extra types for deserialization.</param>
        public static void Save<T>(T instance, Stream stream, bool includeExtraTypes = false)
        {
            if (instance == null)
            {
                throw new ArgumentNullException("instance");
            }

            var ns = new XmlSerializerNamespaces();
            ns.Add(string.Empty, string.Empty);

            var serializer = GetSerializer(typeof(T), includeExtraTypes);

            if (stream is FileStream)
            {
                using (var ms = new MemoryStream())
                {
                    serializer.Serialize(ms, instance, ns);

                    stream.Write(ms.GetBuffer(), 0, (int)ms.Length);
                }
            }
            else
            {
                serializer.Serialize(stream, instance, ns);
            }
        }

        /// <summary>
        /// Serializes an instance to a given stream.
        /// </summary>
        /// <typeparam name="T">The type of object to serialize.</typeparam>
        /// <param name="instance">The instance to serialize to stream.</param>
        /// <param name="writer">The writer to serialize the instance with.</param>
        /// <param name="includeExtraTypes">True to include extra types for deserialization.</param>
        public static void Save<T>(T instance, XmlWriter writer, bool includeExtraTypes = false)
        {
            if (instance == null)
            {
                throw new ArgumentNullException("instance");
            }

            var ns = new XmlSerializerNamespaces();
            ns.Add(string.Empty, string.Empty);

            var serializer = GetSerializer(typeof(T), includeExtraTypes);

            serializer.Serialize(writer, instance, ns);
        }

        /// <summary>
        /// Serializes an instance to a given stream.
        /// </summary>
        /// <param name="type">The type of object to serialize.</param>
        /// <param name="instance">The instance to serialize to stream.</param>
        /// <param name="stream">The stream to serialize the instance to.</param>
        /// <param name="includeExtraTypes">True to include extra types for deserialization.</param>
        public static void Save(
            Type type,
            object instance,
            Stream stream,
            bool includeExtraTypes = false)
        {
            if (instance == null)
            {
                throw new ArgumentNullException("instance");
            }

            var ns = new XmlSerializerNamespaces();
            ns.Add(string.Empty, string.Empty);

            var serializer = GetSerializer(type, includeExtraTypes);

            if (stream is FileStream)
            {
                using (var ms = new MemoryStream())
                {
                    serializer.Serialize(ms, instance, ns);

                    stream.Write(ms.GetBuffer(), 0, (int)ms.Length);
                }
            }
            else
            {
                serializer.Serialize(stream, instance, ns);
            }
        }

        /// <summary>
        /// Serializes an instance to a given stream.
        /// </summary>
        /// <param name="type">The type of object to serialize.</param>
        /// <param name="instance">The instance to serialize to stream.</param>
        /// <param name="writer">The writer to serialize the instance with.</param>
        /// <param name="includeExtraTypes">True to include extra types for deserialization.</param>
        public static void Save(
            Type type,
            object instance,
            XmlWriter writer,
            bool includeExtraTypes = false)
        {
            if (instance == null)
            {
                throw new ArgumentNullException("instance");
            }

            var ns = new XmlSerializerNamespaces();
            ns.Add(string.Empty, string.Empty);

            var serializer = GetSerializer(type, includeExtraTypes);

            serializer.Serialize(writer, instance, ns);
        }

        /// <summary>
        /// Serializes an instance to a file.
        /// </summary>
        /// <typeparam name="T">The type of object to serialize.</typeparam>
        /// <param name="instance">The instance to serialize to stream.</param>
        /// <param name="filePath">The file path.</param>
        /// <param name="includeExtraTypes">True to include extra types for deserialization.</param>
        public static void Save<T>(T instance, string filePath, bool includeExtraTypes = false)
        {
            if (instance == null)
            {
                throw new ArgumentNullException("instance");
            }

            var ns = new XmlSerializerNamespaces();
            ns.Add(string.Empty, string.Empty);

            var serializer = GetSerializer(typeof(T), includeExtraTypes);

            int size = 0;
            byte[] buffer = null;
            using (var ms = new MemoryStream())
            {
                serializer.Serialize(ms, instance, ns);

                buffer = ms.GetBuffer();
                size = (int)ms.Length;
            }

            using (var fs = new FileStream(filePath, FileMode.Create, FileAccess.Write))
            {
                fs.Write(buffer, 0, size);
            }
        }

        /// <summary>
        /// Serializes an IXmlDocSerializable to a file.
        /// </summary>
        /// <param name="obj">The IXmlDocSerializable to serialize.</param>
        /// <param name="filePath">The file path to serialize to.</param>
        public static void SaveXmlDocSerializable(
            IXmlDocSerializable obj,
            string filePath)
        {
            if (obj == null)
            {
                throw new ArgumentNullException("obj");
            }

            int size = 0;
            byte[] buffer = null;
            using (var ms = new MemoryStream())
            {
                var serializer = new XmlDocSerializer();
                if (serializer.Serialize(ms, obj) == true)
                {
                    buffer = ms.GetBuffer();
                    size = (int)ms.Length;
                }
            }

            using (var fs = new FileStream(filePath, FileMode.Create, FileAccess.Write))
            {
                fs.Write(buffer, 0, size);
            }
        }

        /// <summary>
        /// Serializes an IXmlDocSerializable to a stream.
        /// </summary>
        /// <param name="obj">The IXmlDocSerializable to serialize.</param>
        /// <param name="stream">The stream that contains the serialized instance data.</param>
        public static void SaveXmlDocSerializable(
            IXmlDocSerializable obj,
            Stream stream)
        {
            if (obj == null)
            {
                throw new ArgumentNullException("obj");
            }

            if (stream is FileStream)
            {
                using (var ms = new MemoryStream())
                {
                    var serializer = new XmlDocSerializer();
                    if (serializer.Serialize(ms, obj) == true)
                    {
                        stream.Write(ms.GetBuffer(), 0, (int)ms.Length);
                    }
                }
            }
            else
            {
                var serializer = new XmlDocSerializer();
                serializer.Serialize(stream, obj);
            }
        }

        /// <summary>
        /// Serializes an IXmlDocSerializable to a XML document.
        /// </summary>
        /// <param name="obj">The IXmlDocSerializable to serialize.</param>
        /// <param name="xmlDocument">The XML document to read the data from.</param>
        public static void SaveXmlDocSerializable(
            IXmlDocSerializable obj,
            XmlDocument xmlDocument)
        {
            var serializer = new XmlDocSerializer();
            serializer.Serialize(xmlDocument, obj);
        }

        /// <summary>
        /// Clones the instance by serializing, then deserializing it.
        /// </summary>
        /// <typeparam name="T">The type of obejct to clone.</typeparam>
        /// <param name="instance">The instance to clone.</param>
        /// <returns>Returns a cloned instance of the input one.
        /// Throws an exception on failure.</returns>
        public static T Clone<T>(T instance)
        {
            var ns = new XmlSerializerNamespaces();
            ns.Add(string.Empty, string.Empty);

            using (var stream = new MemoryStream())
            {
                var serializer = new XmlSerializer(typeof(T), string.Empty);
                serializer.Serialize(stream, instance, ns);
                stream.Position = 0;
                return (T)serializer.Deserialize(stream);
            }
        }

        /// <summary>
        /// Get XML serializer for the specified type.
        /// </summary>
        /// <param name="type">The type to serialize.</param>
        /// <param name="includeExtraTypes">True to include extra types.</param>
        /// <returns>The XML serializer for the type.</returns>
        private static XmlSerializer GetSerializer(Type type, bool includeExtraTypes)
        {
            if (includeExtraTypes)
            {
                return SerializerIncludeExtraTypesCache.GetOrAdd(
                    new SerializerKey
                    {
                        TargetType = type,
                        ExtraTypes = GetExtraTypes()
                    },
                    t => new XmlSerializer(t.TargetType, t.ExtraTypes));
            }
            else
            {
                return SerializerCache.GetOrAdd(type, t => new XmlSerializer(t));
            }
        }

        /// <summary>
        /// Holds the main type and extra types for a XML serializer.
        /// </summary>
        private class SerializerKey
        {
            /// <summary>
            /// Get or set the target type of the serializer.
            /// </summary>
            public Type TargetType { get; set; }

            /// <summary>
            /// Get or set the extra types of the serializer.
            /// </summary>
            public Type[] ExtraTypes { get; set; }
        }

        /// <summary>
        /// Compares serializers.
        /// </summary>
        private class SerializerKeyComparer : IEqualityComparer<SerializerKey>
        {
            /// <summary>
            /// Compares the given serializer keys.
            /// </summary>
            /// <param name="x">The serializer key.</param>
            /// <param name="y">Another serializer key.</param>
            /// <returns>True if the keys are equal.</returns>
            public bool Equals(SerializerKey x, SerializerKey y)
            {
                return (x.TargetType == y.TargetType) &&
                       x.ExtraTypes.SequenceEqual(y.ExtraTypes);
            }

            /// <summary>
            /// Get hash code for the specified serializer key.
            /// </summary>
            /// <param name="obj">The serializer key.</param>
            /// <returns>The hash code.</returns>
            public int GetHashCode(SerializerKey obj)
            {
                var hash = obj.TargetType.GetHashCode();

                obj.ExtraTypes.ForEach(x =>
                {
                    unchecked
                    {
                        hash *= 31;
                        hash += x.GetHashCode();
                    }
                });

                return hash;
            }
        }
    }
}
