﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Drawing;
using System.Windows.Forms;

namespace EffectMaker.Foundation.Render.ScrollBar
{
    /// <summary>
    /// Class for the horizontal auto hidden scroll bars.
    /// </summary>
    internal class HScrollBar : ScrollBarBase
    {
        /// <summary>
        /// Constructor.
        /// </summary>
        public HScrollBar()
            : base()
        {
        }

        /// <summary>Get the flag indicating if the scroll bar is visible.</summary>
        public override bool IsVisible
        {
            get
            {
                if ((this.IsEnabled == false) ||
                    (this.OwnerContentSize.Width <= this.OwnerViewSize.Width))
                {
                    return false;
                }
                else
                {
                    return true;
                }
            }
        }

        /// <summary>Update layout.</summary>
        /// <param name="rectDisplay">The display rectangle of the owner view.</param>
        /// <param name="sizeShrink">How much size to shrink from the rectangle.</param>
        public override void UpdateLayout(Rectangle rectDisplay, int sizeShrink)
        {
            if (this.IsVisible == false)
            {
                return;
            }

            // Rectangle for the whole scroll bar.
            this.SetScrollBarRectangle(
                rectDisplay.Left,
                rectDisplay.Bottom - ScrollBarBase.ScrollBarSize,
                rectDisplay.Width - sizeShrink,
                ScrollBarBase.ScrollBarSize);

            // Rectangle for the left button.
            this.SetDecreaseButtonRectangle(
                this.ScrollBarBounds.Left,
                this.ScrollBarBounds.Top,
                ScrollBarBase.ArrowButtonSize,
                this.ScrollBarBounds.Height);

            // Rectangle for the right button.
            this.SetIncreaseButtonRectangle(
                this.ScrollBarBounds.Right - ScrollBarBase.ArrowButtonSize,
                this.ScrollBarBounds.Top,
                ScrollBarBase.ArrowButtonSize,
                this.ScrollBarBounds.Height);

            // Rectangle for the scroll track.
            this.SetScrollTrackRectangle(
                this.ScrollBarBounds.Left + ScrollBarBase.ArrowButtonSize,
                this.ScrollBarBounds.Top,
                this.ScrollBarBounds.Width - (ScrollBarBase.ArrowButtonSize * 2),
                this.ScrollBarBounds.Height);

            // Rectangle for the scroll bar handle.
            this.ComputeScrollHandle();

            this.TriggerRenderRequiredEvent();
        }

        /// <summary>
        /// Handle MouseMove event from the owner control.
        /// </summary>
        /// <param name="e">The event arguments.</param>
        public override void OnOwnerMouseMove(MouseEventArgs e)
        {
            base.OnOwnerMouseMove(e);

            if (this.IsMouseDragging == true)
            {
                int x = e.X + this.MouseDragOffset.X;

                this.Value = (int)(((float)x / (float)this.ScrollTrackBounds.Width) *
                                   ((float)this.OwnerContentSize.Width));
            }
        }

        /// <summary>
        /// Clamp the given scroll value to the scroll range.
        /// </summary>
        /// <param name="value">The scroll value.</param>
        /// <returns>The clamped value.</returns>
        protected override int ClampValue(int value)
        {
            int max = this.OwnerContentSize.Width - this.OwnerViewSize.Width;
            return Math.Max(0, Math.Min(max, value));
        }

        /// <summary>
        /// Render the increase button of the scroll bar.
        /// </summary>
        /// <param name="g">The graphics object for rendering.</param>
        protected override void RenderIncreaseButton(Graphics g)
        {
            using (var brush = new SolidBrush(Color.FromArgb(this.ForegroundAlpha, this.ForeColor)))
            {
                Rectangle rect = this.IncreaseButtonBounds;

                Point[] vertices = new Point[3];
                vertices[0] = new Point(rect.Right - 4, (rect.Bottom + rect.Top) / 2);
                vertices[1] = new Point(rect.Left + 2, rect.Top);
                vertices[2] = new Point(rect.Left + 2, rect.Bottom);

                g.FillPolygon(brush, vertices);
            }
        }

        /// <summary>
        /// Render the decrease button of the scroll bar.
        /// </summary>
        /// <param name="g">The graphics object for rendering.</param>
        protected override void RenderDecreaseButton(Graphics g)
        {
            using (var brush = new SolidBrush(Color.FromArgb(this.ForegroundAlpha, this.ForeColor)))
            {
                Rectangle rect = this.DecreaseButtonBounds;

                Point[] vertices = new Point[3];
                vertices[0] = new Point(rect.Left + 4, (rect.Bottom + rect.Top) / 2);
                vertices[1] = new Point(rect.Right - 2, rect.Top);
                vertices[2] = new Point(rect.Right - 2, rect.Bottom);

                g.FillPolygon(brush, vertices);
            }
        }

        /// <summary>
        /// Compute scroll bar handle position.
        /// </summary>
        protected override void ComputeScrollHandle()
        {
            if ((this.OwnerViewSize.Width <= 0) ||
                (this.OwnerContentSize.Width <= this.OwnerViewSize.Width))
            {
                this.SetScrollHandleRectangle(
                    this.ScrollTrackBounds.X,
                    this.ScrollTrackBounds.Top,
                    this.ScrollTrackBounds.Width,
                    this.ScrollTrackBounds.Height);
                return;
            }

            float viewContentRatio =
                Math.Min(1.0f, (float)this.OwnerViewSize.Width / (float)this.OwnerContentSize.Width);

            int handleWidth =
                Math.Max(MinScrollHandleLength, (int)((float)this.ScrollTrackBounds.Width * viewContentRatio));

            float t = (float)this.Value / (float)(this.OwnerContentSize.Width - this.OwnerViewSize.Width);

            int len = this.ScrollTrackBounds.Width - this.ScrollHandleBounds.Width;
            int pos = (int)Math.Round((float)len * t);

            this.SetScrollHandleRectangle(
                this.ScrollTrackBounds.Left + pos,
                this.ScrollTrackBounds.Y + ((this.ScrollTrackBounds.Height - ScrollBarBase.ScrollHandleSize) / 2),
                handleWidth,
                ScrollBarBase.ScrollHandleSize);
        }
    }
}
