﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows.Forms;

using EffectMaker.Foundation.Animation;

namespace EffectMaker.Foundation.Render.ScrollBar
{
    /// <summary>
    /// Class for the auto hidden scroll bars.
    /// </summary>
    public class AutoHiddenScrollBar
    {
        /// <summary>The owner control.</summary>
        private Control ownerControl = null;

        /// <summary>Size of the contents.</summary>
        private Size ownerContentSize = new Size(0, 0);

        /// <summary>Size of the view.</summary>
        private Size ownerViewSize = new Size(0, 0);

        /// <summary>The display rectangle for rendering the scroll bars.</summary>
        private Rectangle displayRectangle = new Rectangle(0, 0, 0, 0);

        /// <summary>The vertical scroll bar.</summary>
        private VScrollBar scrollBarV = new VScrollBar();

        /// <summary>The horizontal scroll bar.</summary>
        private HScrollBar scrollBarH = new HScrollBar();

        /// <summary>The number of times BeginLayout has been called.</summary>
        private int beginLayoutCount = 0;

        /// <summary>
        /// Constructor.
        /// </summary>
        /// <param name="owner">The owner control of the scroll bars.</param>
        public AutoHiddenScrollBar(Control owner)
        {
            this.ownerControl = owner;

            // Require rendering event.
            this.scrollBarV.RenderRequired += this.OnScrollBarRenderRequired;
            this.scrollBarH.RenderRequired += this.OnScrollBarRenderRequired;

            this.scrollBarV.Scroll += (s, e) =>
            {
                if (this.Scroll != null)
                {
                    this.Scroll(this, EventArgs.Empty);
                }
            };

            this.scrollBarH.Scroll += (s, e) =>
            {
                if (this.Scroll != null)
                {
                    this.Scroll(this, EventArgs.Empty);
                }
            };

            // Owner event handlers.
            this.ownerControl.MouseEnter += this.OnOwnerMouseEnter;
            this.ownerControl.MouseLeave += this.OnOwnerMouseLeave;
            this.ownerControl.MouseMove += this.OnOwnerMouseMove;
            this.ownerControl.MouseDown += this.OnOwnerMouseDown;
            this.ownerControl.MouseUp += this.OnOwnerMouseUp;
        }

        /// <summary>
        /// Event triggered when rendering is required for this scroll bar.
        /// </summary>
        public event EventHandler RenderRequired = null;

        /// <summary>
        /// Event triggered when the scroll position is changed.
        /// </summary>
        public event EventHandler Scroll = null;

        /// <summary>Get or set the foreground color of the scroll bars.</summary>
        public Color ForeColor
        {
            get
            {
                return this.scrollBarV.ForeColor;
            }

            set
            {
                this.scrollBarV.ForeColor = value;
                this.scrollBarH.ForeColor = value;
            }
        }

        /// <summary>Get or set the background color of the scroll bars.</summary>
        public Color BackColor
        {
            get
            {
                return this.scrollBarV.BackColor;
            }

            set
            {
                this.scrollBarV.BackColor = value;
                this.scrollBarH.BackColor = value;
            }
        }

        /// <summary>
        /// Get or set the scroll position.
        /// </summary>
        public Point ScrollPosition
        {
            get
            {
                return new Point(this.scrollBarH.Value, this.scrollBarV.Value);
            }

            set
            {
                this.SuspendLayout();
                this.scrollBarH.Value = value.X;
                this.scrollBarV.Value = value.Y;
                this.ResumeLayout();
            }
        }

        /// <summary>
        /// Get or set the increment for the vertical scroll bar
        /// when the increment/decrement button is pressed.
        /// </summary>
        public int IncrementV
        {
            get { return this.scrollBarV.Increment; }
            set { this.scrollBarV.Increment = value; }
        }

        /// <summary>
        /// Get or set the increment for the horizontal scroll bar
        /// when the increment/decrement button is pressed.
        /// </summary>
        public int IncrementH
        {
            get { return this.scrollBarH.Increment; }
            set { this.scrollBarH.Increment = value; }
        }

        /// <summary>
        /// Get the flag indicating whether the cursor is over the scroll bars.
        /// </summary>
        public bool IsMouseOverScrollBars
        {
            get
            {
                return this.scrollBarH.IsMouseOverScrollBar || this.scrollBarV.IsMouseOverScrollBar;
            }
        }

        /// <summary>
        /// Get the flag indicating whether any of the scroll handles are getting dragged by mouse cursor.
        /// </summary>
        public bool IsMouseDragging
        {
            get
            {
                return this.scrollBarH.IsMouseDragging || this.scrollBarV.IsMouseDragging;
            }
        }

        /// <summary>
        /// Suspend layout.
        /// </summary>
        public void SuspendLayout()
        {
            ++this.beginLayoutCount;
        }

        /// <summary>
        /// Resume layout.
        /// </summary>
        public void ResumeLayout()
        {
            --this.beginLayoutCount;
            if (this.beginLayoutCount == 0)
            {
                this.UpdateLayout();
                if (this.RenderRequired != null)
                {
                    this.RenderRequired(this, EventArgs.Empty);
                }
            }
            else if (this.beginLayoutCount < 0)
            {
                this.beginLayoutCount = 0;
            }
        }

        /// <summary>
        /// Set the size of the contents of the scroll bar owner.
        /// </summary>
        /// <remarks>
        /// The unit of this size can be the actual number of pixels
        /// or any other unit (e.g. line count in a text view),
        /// but the unit of the view size and content size has to
        /// be the same.
        /// </remarks>
        /// <param name="size">The content size.</param>
        public void SetContentSize(Size size)
        {
            if (this.ownerContentSize == null ||
                this.ownerContentSize.Width != size.Width ||
                this.ownerContentSize.Height != size.Height)
            {
                this.ownerContentSize = size;
                this.UpdateLayout();
            }
        }

        /// <summary>
        /// Set the size of the view of the scroll bar owner.
        /// </summary>
        /// <remarks>
        /// The unit of this size can be the actual number of pixels
        /// or any other unit (e.g. line count in a text view),
        /// but the unit of the view size and content size has to
        /// be the same.
        /// </remarks>
        /// <param name="size">The view size.</param>
        public void SetViewSize(Size size)
        {
            this.ownerViewSize = size;
            this.UpdateLayout();
        }

        /// <summary>
        /// Set the display rectangle for rendering the scroll bar.
        /// (both vertical and horizontal)
        /// </summary>
        /// <param name="rect">The display rectandle.</param>
        public void SetDisplayRectangle(Rectangle rect)
        {
            this.displayRectangle = rect;
            this.UpdateLayout();
        }

        /// <summary>
        /// Render the scroll bars.
        /// </summary>
        /// <param name="g">The graphics object for rendering.</param>
        public void Render(Graphics g)
        {
            this.scrollBarV.Render(g);
            this.scrollBarH.Render(g);
        }

        /// <summary>
        /// Update scroll bar layout.
        /// </summary>
        protected virtual void UpdateLayout()
        {
            this.scrollBarV.SetContentAndViewSize(this.ownerContentSize, this.ownerViewSize);
            this.scrollBarH.SetContentAndViewSize(this.ownerContentSize, this.ownerViewSize);

            int scrollBarVSizeOffset = 0;
            if (this.scrollBarH.IsVisible == true)
            {
                scrollBarVSizeOffset = HScrollBar.ScrollBarSize;
            }

            int scrollBarHSizeOffset = 0;
            if (this.scrollBarV.IsVisible == true)
            {
                scrollBarHSizeOffset = VScrollBar.ScrollBarSize;
            }

            this.scrollBarV.UpdateLayout(this.displayRectangle, scrollBarVSizeOffset);
            this.scrollBarH.UpdateLayout(this.displayRectangle, scrollBarHSizeOffset);
        }

        /// <summary>
        /// Handle MouseEnter event from the owner control.
        /// </summary>
        /// <param name="sender">The sender of the event.</param>
        /// <param name="e">The event arguments.</param>
        private void OnOwnerMouseEnter(object sender, EventArgs e)
        {
            this.scrollBarV.OnOwnerMouseEnter();
            this.scrollBarH.OnOwnerMouseEnter();
        }

        /// <summary>
        /// Handle MouseLeave event from the owner control.
        /// </summary>
        /// <param name="sender">The sender of the event.</param>
        /// <param name="e">The event arguments.</param>
        private void OnOwnerMouseLeave(object sender, EventArgs e)
        {
            this.scrollBarV.OnOwnerMouseLeave();
            this.scrollBarH.OnOwnerMouseLeave();
        }

        /// <summary>
        /// Handle MouseMove event from the owner control.
        /// </summary>
        /// <param name="sender">The sender of the event.</param>
        /// <param name="e">The event arguments.</param>
        private void OnOwnerMouseMove(object sender, MouseEventArgs e)
        {
            this.scrollBarV.OnOwnerMouseMove(e);
            this.scrollBarH.OnOwnerMouseMove(e);
        }

        /// <summary>
        /// Handle MouseDown event from the owner control.
        /// </summary>
        /// <param name="sender">The sender of the event.</param>
        /// <param name="e">The event arguments.</param>
        private void OnOwnerMouseDown(object sender, MouseEventArgs e)
        {
            this.scrollBarV.OnOwnerMouseDown(e);
            this.scrollBarH.OnOwnerMouseDown(e);
        }

        /// <summary>
        /// Handle MouseUp event from the owner control.
        /// </summary>
        /// <param name="sender">The sender of the event.</param>
        /// <param name="e">The event arguments.</param>
        private void OnOwnerMouseUp(object sender, MouseEventArgs e)
        {
            this.scrollBarV.OnOwnerMouseUp(e);
            this.scrollBarH.OnOwnerMouseUp(e);
        }

        /// <summary>
        /// Handle RenderRequired event from the scroll bars.
        /// </summary>
        /// <param name="sender">The sender of the event.</param>
        /// <param name="e">The event arguments.</param>
        private void OnScrollBarRenderRequired(object sender, EventArgs e)
        {
            if ((this.beginLayoutCount <= 0) &&
                (this.RenderRequired != null))
            {
                this.RenderRequired(this, EventArgs.Empty);
            }
        }
    }
}
