﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Linq;

using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Render.ObjectPicking;
using EffectMaker.Foundation.Utility;

namespace EffectMaker.Foundation.Render.Renderable
{
    /// <summary>
    /// Abstract class for renderable items.
    /// </summary>
    public abstract class RenderableBase : IDisposable, IPickableObject
    {
        /// <summary>A static counter for the renderable ID.</summary>
        private static int uniqueIdCounter = 0;

        /// <summary>The transformation for rendering.</summary>
        private Transformation transformation = new Transformation();

        /// <summary>The boundaries of the renderable.</summary>
        private RectangleF bounds = RectangleF.Empty;

        /// <summary>The fill color.</summary>
        private Color fillColor = Color.Transparent;

        /// <summary>The border color.</summary>
        private Color borderColor = Color.Transparent;

        /// <summary>The border thickness.</summary>
        private float borderThickness = 0.0f;

        /// <summary>The flag indicating whether the renderable is visible or not.</summary>
        private bool isVisible = true;

        /// <summary>The counter for suspending update and rendering.</summary>
        private int suspendUpdateAndRenderingCount = 0;

        /// <summary>The render context saved for rendering.</summary>
        private RenderContext savedRenderContext = null;

        /// <summary>The child renderable items.</summary>
        private List<RenderableBase> renderables = new List<RenderableBase>();

        /// <summary>
        /// Default constructor.
        /// </summary>
        public RenderableBase()
        {
            this.ID = uniqueIdCounter++;
            this.IsTranslationOnly = false;
            this.TransformedBounds = RectangleF.Empty;
            this.ClippedBounds = RectangleF.Empty;
            this.CanPick = true;
        }

        /// <summary>Get the ID of the renderable.</summary>
        public int ID { get; private set; }

        /// <summary>
        /// Get the flag indicating whether update and rendering is suspended.
        /// </summary>
        public bool IsUpdateAndRenderingSuspended
        {
            get { return this.suspendUpdateAndRenderingCount > 0; }
        }

        /// <summary>
        /// Get or set the transformation.
        /// </summary>
        public Transformation Transformation
        {
            get { return this.transformation; }
        }

        /// <summary>
        /// Get or set the scale.
        /// </summary>
        public PointF Scale
        {
            get { return this.transformation.Scale; }
            set { this.transformation.Scale = value; }
        }

        /// <summary>
        /// Get or set the translation.
        /// </summary>
        public PointF Translation
        {
            get { return this.transformation.Translation; }
            set { this.transformation.Translation = value; }
        }

        /// <summary>
        /// Get the transformation matrix.
        /// </summary>
        public Matrix TransformationMatrix
        {
            get { return this.transformation.TransformationMatrix; }
        }

        /// <summary>
        /// Get or set the boundary of the item.
        /// </summary>
        public virtual RectangleF Bounds
        {
            get
            {
                return this.bounds;
            }

            set
            {
                this.bounds = value;
                this.RequestRedraw();
            }
        }

        /// <summary>
        /// Get or set the location of this item.
        /// </summary>
        public virtual PointF Location
        {
            get { return this.Bounds.Location; }
            set { this.Bounds = new RectangleF(value, this.Bounds.Size); }
        }

        /// <summary>
        /// Get or set the size of this item.
        /// </summary>
        public virtual SizeF Size
        {
            get { return this.Bounds.Size; }
            set { this.Bounds = new RectangleF(this.Bounds.Location, value); }
        }

        /// <summary>
        /// Get the location X.
        /// </summary>
        public float X
        {
            get { return this.Bounds.X; }
        }

        /// <summary>
        /// Get the location Y.
        /// </summary>
        public float Y
        {
            get { return this.Bounds.Y; }
        }

        /// <summary>
        /// Get the width.
        /// </summary>
        public float Width
        {
            get { return this.Bounds.Width; }
        }

        /// <summary>
        /// Get the height.
        /// </summary>
        public float Height
        {
            get { return this.Bounds.Height; }
        }

        /// <summary>
        /// Get or set the flag indicating whether to ignore transformations
        /// other than translation.
        /// </summary>
        public virtual bool IsTranslationOnly { get; set; }

        /// <summary>
        /// Get or set the transformed boundary of the item.
        /// </summary>
        public virtual RectangleF TransformedBounds { get; set; }

        /// <summary>
        /// Get or set the view clipped boundary of the item.
        /// </summary>
        public virtual RectangleF ClippedBounds { get; set; }

        /// <summary>
        /// Get or set the border color of the item.
        /// </summary>
        public virtual Color BorderColor
        {
            get
            {
                return this.borderColor;
            }

            set
            {
                if (this.borderColor.Equals(value) == false)
                {
                    this.borderColor = value;
                    this.RequestRedraw();
                }
            }
        }

        /// <summary>
        /// Get or set the thickness of the item.
        /// </summary>
        public virtual float BorderThickness
        {
            get
            {
                return this.borderThickness;
            }

            set
            {
                if (this.borderThickness != value)
                {
                    this.borderThickness = value;
                    this.RequestRedraw();
                }
            }
        }

        /// <summary>
        /// Get or set the fill color of the item.
        /// </summary>
        public virtual Color FillColor
        {
            get
            {
                return this.fillColor;
            }

            set
            {
                if (this.fillColor.Equals(value) == false)
                {
                    this.fillColor = value;
                    this.RequestRedraw();
                }
            }
        }

        /// <summary>
        /// Get or set the flag indicating whether this renderable is visible or not.
        /// </summary>
        public virtual bool Visible
        {
            get
            {
                return this.isVisible;
            }

            set
            {
                if (this.isVisible != value)
                {
                    this.isVisible = value;
                    this.RequestRedraw();
                }
            }
        }

        /// <summary>
        /// Get or set the flag indicating whether this renderable can be picked by cursor.
        /// </summary>
        public bool CanPick { get; set; }

        /// <summary>
        /// Get or set the flag indicating whether rendering is requested while suspending.
        /// </summary>
        public bool IsRenderRequestedWhileSuspending { get; protected set; }

        /// <summary>
        /// Get the render context received on Update().
        /// DO NOT use this render context except for Draw() method.
        /// The render context could be expired if use elsewhere.
        /// </summary>
        public RenderContext SavedRenderContext
        {
            get { return this.savedRenderContext; }
            protected set { this.savedRenderContext = value; }
        }

        /// <summary>
        /// Get or set the parent renderable.
        /// </summary>
        public RenderableBase Parent { get; protected set; }

        /// <summary>
        /// Enumerate the child renderables.
        /// </summary>
        public IEnumerable<RenderableBase> Renderables
        {
            get
            {
                return this.renderables;
            }
        }

        /// <summary>
        /// Release the resources the renderable uses.
        /// </summary>
        public void Dispose()
        {
            this.Dispose(true);
        }

        /// <summary>
        /// Suspend update and rendering.
        /// </summary>
        public virtual void SuspendUpdateAndRendering()
        {
            // Reset the flag.
            if (this.suspendUpdateAndRenderingCount <= 0)
            {
                this.IsRenderRequestedWhileSuspending = false;
            }

            ++this.suspendUpdateAndRenderingCount;
        }

        /// <summary>
        /// Resume update and rendering.
        /// </summary>
        /// <returns>True if rendering is requested while suspending.</returns>
        public virtual bool ResumeUpdateAndRendering()
        {
            --this.suspendUpdateAndRenderingCount;

            return this.IsRenderRequestedWhileSuspending;
        }

        /// <summary>
        /// Add a child renderable.
        /// </summary>
        /// <param name="child">The child to add.</param>
        public virtual void AddRenderable(RenderableBase child)
        {
            this.renderables.Add(child);
            child.Parent = this;

            this.RequestRedraw();
        }

        /// <summary>
        /// Add child renderables.
        /// </summary>
        /// <param name="children">The children to add.</param>
        public virtual void AddRenderableRange(IEnumerable<RenderableBase> children)
        {
            foreach (RenderableBase child in children)
            {
                int index = this.renderables.IndexOf(child);
                if (index < 0)
                {
                    this.renderables.Add(child);
                    child.Parent = this;
                }
            }

            this.RequestRedraw();
        }

        /// <summary>
        /// Remove a child renderable.
        /// </summary>
        /// <param name="child">The child to remove.</param>
        public virtual void RemoveRenderable(RenderableBase child)
        {
            int index = this.renderables.IndexOf(child);
            if (index >= 0)
            {
                child.Parent = null;
                this.renderables.RemoveAt(index);

                this.RequestRedraw();
            }
        }

        /// <summary>
        /// Remove child renderables.
        /// </summary>
        /// <param name="children">The children to remove.</param>
        public virtual void RemoveRenderableRange(IEnumerable<RenderableBase> children)
        {
            // The argument "children" could be the renderable list itself.
            // To prevent exception when modifying the list while looping through it,
            // we have to copy "children" to a new array first.
            var childrenToRemove = children.ToArray();

            foreach (RenderableBase child in childrenToRemove)
            {
                int index = this.renderables.IndexOf(child);
                if (index >= 0)
                {
                    child.Parent = null;
                    this.renderables.RemoveAt(index);
                }
            }

            this.RequestRedraw();
        }

        /// <summary>
        /// Clear child renderables.
        /// </summary>
        public virtual void ClearRenderables()
        {
            foreach (RenderableBase child in this.renderables)
            {
                child.Parent = null;
            }

            this.renderables.Clear();

            this.RequestRedraw();
        }

        /// <summary>
        /// Test if the given point lies on this object.
        /// </summary>
        /// <param name="p">The point.</param>
        /// <returns>True if the point picks this object.</returns>
        public virtual bool Pick(PointF p)
        {
            if (this.CanPick == false || this.Visible == false)
            {
                return false;
            }

            return this.ClippedBounds.Contains(p);
        }

        /// <summary>
        /// Test if the object is picked with the conditions given by the pick context.
        /// </summary>
        /// <param name="context">The pick context.</param>
        public virtual void Pick(ObjectPickContext context)
        {
            if (this.Visible == false)
            {
                return;
            }

            // First pick the children of this renderable.
            this.PickChildren(context);

            // Pick itself.
            if (this.Pick(context.PickPoint) == true)
            {
                context.AddPickedObject(this);
            }
        }

        /// <summary>
        /// Request redraw.
        /// </summary>
        public virtual void RequestRedraw()
        {
            if (this.Parent != null &&
                this.IsUpdateAndRenderingSuspended == false &&
                this.Visible == true)
            {
                this.Parent.RequestRedraw();
            }
        }

        /// <summary>
        /// Apply transformation.
        /// The transformation is added to the original transformation values.
        /// </summary>
        /// <param name="transformation">Translation to apply.</param>
        public void ApplyTransformation(Transformation transformation)
        {
            this.transformation.ApplyTransformation(transformation);
        }

        /// <summary>
        /// Apply transformation.
        /// The transformation is added to the original transformation values.
        /// </summary>
        /// <param name="translateX">Translation on X axis.</param>
        /// <param name="translateY">Translation on Y axis.</param>
        /// <param name="scaleX">Scale on X axis.</param>
        /// <param name="scaleY">Scale on Y axis.</param>
        public void ApplyTransformation(
            float translateX,
            float translateY,
            float scaleX = 1.0f,
            float scaleY = 1.0f)
        {
            this.transformation.ApplyTransformation(translateX, translateY, scaleX, scaleY);
        }

        /// <summary>
        /// Update the item for rendering.
        /// </summary>
        /// <param name="context">Data context that contains information for rendering.</param>
        public virtual void Update(RenderContext context)
        {
            if (this.Visible == false)
            {
                return;
            }

            if (this.IsUpdateAndRenderingSuspended == false)
            {
                this.SuspendUpdateAndRendering();

                this.UpdateSelf(context);
                this.UpdateChildren(context);

                this.ResumeUpdateAndRendering();
            }
        }

        /// <summary>
        /// Render the item.
        /// </summary>
        /// <param name="g">The graphics object for rendering.</param>
        /// <returns>False when the item is clipped and need not to be rendered.</returns>
        public virtual bool Draw(Graphics g)
        {
            if (this.IsUpdateAndRenderingSuspended == true)
            {
                return false;
            }

            if (this.Visible == false)
            {
                return false;
            }

            if (this.TransformedBounds.IsEmpty == true)
            {
                // The rectangle is empty, nothing to render.
                return false;
            }

            bool isAnythingRendered = true;

            this.SuspendUpdateAndRendering();

            if (this.ClippedBounds.IsEmpty == true)
            {
                // This item is not in the view rectangle, only render the children.
                isAnythingRendered = this.DrawChildren(g);
            }
            else
            {
                this.DrawBackground(g);
                this.DrawForeground(g);
                this.DrawBorder(g);
                this.DrawChildren(g);
            }

            this.ResumeUpdateAndRendering();

            return isAnythingRendered;
        }

        /// <summary>
        /// Release the resources the renderable uses.
        /// </summary>
        /// <param name="isDisposing">True if the renderable is being disposed.</param>
        protected virtual void Dispose(bool isDisposing)
        {
            // Release the children.
            foreach (RenderableBase child in this.renderables)
            {
                child.Dispose(isDisposing);
            }

            this.renderables.Clear();
        }

        /// <summary>
        /// Update the item for rendering.
        /// </summary>
        /// <param name="context">Data context that contains information for rendering.</param>
        protected virtual void UpdateSelf(RenderContext context)
        {
            // Transform the boundary.
            this.TransformedBounds = RenderUtility.TransformRectangle(
                context.TransformationMatrix,
                this.Bounds,
                this.IsTranslationOnly);

            // Do clipping with the view rectangle.
            var clippedRect = this.TransformedBounds;
            if (context.IsViewRenctangleEmpty == false)
            {
                clippedRect.Intersect(context.ViewRectangle);
            }

            this.ClippedBounds = clippedRect;

            // Save the render context for rendering.
            this.savedRenderContext = context.Clone() as RenderContext;
        }

        /// <summary>
        /// Update the children for rendering.
        /// </summary>
        /// <param name="context">Data context that contains information for rendering.</param>
        protected virtual void UpdateChildren(RenderContext context)
        {
            // Update the children.
            using (PushPopRenderContext pushPopContext = new PushPopRenderContext(context))
            {
                // Apply transformation to the children.
                context.ApplyTransformation(this.Location.X, this.Location.Y);
                context.ApplyTransformation(this.transformation);
                foreach (RenderableBase child in this.Renderables)
                {
                    child.Update(context);
                }
            }
        }

        /// <summary>
        /// Render the item's background.
        /// </summary>
        /// <param name="g">The graphics object for rendering.</param>
        protected virtual void DrawBackground(Graphics g)
        {
            // Do nothing.
        }

        /// <summary>
        /// Render the item's foreground.
        /// </summary>
        /// <param name="g">The graphics object for rendering.</param>
        protected virtual void DrawForeground(Graphics g)
        {
            // Do nothing.
        }

        /// <summary>
        /// Render the item's border.
        /// </summary>
        /// <param name="g">The graphics object for rendering.</param>
        protected virtual void DrawBorder(Graphics g)
        {
            // Do nothing.
        }

        /// <summary>
        /// Draw children items.
        /// </summary>
        /// <param name="g">The graphics object for rendering.</param>
        /// <returns>True if any children is rendered.</returns>
        protected virtual bool DrawChildren(Graphics g)
        {
            if (this.Visible == false)
            {
                return false;
            }

            bool isAnyChildRendered = false;
            foreach (RenderableBase child in this.Renderables)
            {
                isAnyChildRendered |= child.Draw(g);
            }

            return isAnyChildRendered;
        }

        /// <summary>
        /// Do picking for the children renderables.
        /// </summary>
        /// <param name="context">The pick context.</param>
        protected void PickChildren(ObjectPickContext context)
        {
            if (this.Visible == false)
            {
                return;
            }

            foreach (RenderableBase child in this.renderables)
            {
                child.Pick(context);
            }
        }
    }
}
