﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

namespace EffectMaker.Foundation.Extensions
{
    /// <summary>
    /// Class containing extensions method related to System.Type class.
    /// </summary>
    public static class TypeExtensions
    {
        /// <summary>
        /// Determine if a class inherit from a generic type definition,
        /// and if yes, retrieves it generic arguments.
        /// </summary>
        /// <param name="type">The type to check.</param>
        /// <param name="genericTypeDefinition">The generic type definition.</param>
        /// <param name="genericArguments">The retrieved generic arguments.</param>
        /// <returns>Returns true if inheritance is confirmed, false otherwise.</returns>
        public static bool IsAssignableFromGenericTypeDefinition(
            this Type type,
            Type genericTypeDefinition,
            out Type[] genericArguments)
        {
            if (type == null)
            {
                genericArguments = null;
                return false;
            }

            if (type.IsGenericType)
            {
                Type[] typeArguments = type.GetGenericArguments();

                Type fullGenericType = genericTypeDefinition.MakeGenericType(typeArguments);

                if (type == fullGenericType)
                {
                    genericArguments = typeArguments;
                    return true;
                }
            }

            return IsAssignableFromGenericTypeDefinition(
                type.BaseType,
                genericTypeDefinition,
                out genericArguments);
        }

        /// <summary>
        /// Check if the type is a generic type, and if the type contains
        /// exactly same generic arguments as specified.
        /// </summary>
        /// <param name="self">The type.</param>
        /// <param name="types">
        /// The generic arguments to check.
        /// E.g. T1 and T2 of IEnumerable{T1, T2}
        /// </param>
        /// <returns>True if the type is a generic type of the given types.</returns>
        public static bool IsGenericTypeOf(
            this Type self,
            params Type[] types)
        {
            if (self == null)
            {
                return false;
            }

            if (self.IsGenericType == false)
            {
                return false;
            }

            Type[] typeArguments = self.GetGenericArguments();
            if (typeArguments == null ||
                types == null ||
                typeArguments.Length != types.Length)
            {
                return false;
            }

            for (int i = 0; i < typeArguments.Length; ++i)
            {
                if (typeArguments[i] != types[i] &&
                    typeArguments[i].IsSubclassOf(types[i]) == false)
                {
                    return false;
                }
            }

            return true;
        }

        /// <summary>
        /// Finds the common ancestor type or two different types.
        /// </summary>
        /// <param name="types">All the types to find their common ancestor.</param>
        /// <returns>Returns the common ancestor of all the types.</returns>
        public static Type FindCommonAncestorType(params Type[] types)
        {
            if (types.Length < 2)
            {
                throw new ArgumentException("types argument must contain at least two items.");
            }

            var array = new Type[types.Length][];
            int count = int.MaxValue;

            for (int i = 0; i < types.Length; i++)
            {
                array[i] = types[i].FlattenClassHierarchy();
                count = Math.Min(count, array[i].Length);
            }

            Type currentType = null;

            for (int i = 0; i < count; i++)
            {
                bool same = true;
                for (int j = 0; j < array.Length - 1; j++)
                {
                    if (array[j][i] != array[j + 1][i])
                    {
                        same = false;
                        break;
                    }

                    currentType = array[0][i];
                }

                if (same == false)
                {
                    break;
                }
            }

            return currentType;
        }

        /// <summary>
        /// Flatten the class hierarchy of a type.
        /// </summary>
        /// <param name="childType">Type to flattend its class hierarchy.</param>
        /// <returns>Returns the flattened hierarchy of the type.</returns>
        public static Type[] FlattenClassHierarchy(this Type childType)
        {
            if (childType == null)
            {
                throw new ArgumentNullException("childType");
            }

            var list = new List<Type>();
            FlattenClassHierarchy(childType, list);
            return list.ToArray();
        }

        /// <summary>
        /// Flatten the class hierarchy of a type.
        /// </summary>
        /// <param name="childType">Type to flattend its class hierarchy.</param>
        /// <param name="list">The list in which the flattened hierarchy is added.</param>
        private static void FlattenClassHierarchy(Type childType, List<Type> list)
        {
            if (childType == null)
            {
                return;
            }

            FlattenClassHierarchy(childType.BaseType, list);
            list.Add(childType);
        }
    }
}
