﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Drawing;
using System.Linq;
using System.Text;
using EffectMaker.Foundation.Interfaces;
using EffectMaker.Foundation.Primitives;

namespace EffectMaker.Foundation.Extensions
{
    /// <summary>
    /// Class containing extension methods for Rectangle.
    /// </summary>
    public static class RectangleExtensions
    {
        /// <summary>
        /// Get the center point of the rectangle.
        /// </summary>
        /// <param name="rect">The rectangle to create values from.</param>
        /// <returns>The sequence of values from the primitive.</returns>
        public static Point GetCenter(this Rectangle rect)
        {
            return new Point((rect.Right + rect.Left) / 2, (rect.Bottom + rect.Top) / 2);
        }

        /// <summary>
        /// Union this rectangle with another rectangle.
        /// </summary>
        /// <param name="self">This rectangle.</param>
        /// <param name="unionWith">Another rectangle to union with.</param>
        /// <returns>The rectangle after union.</returns>
        public static Rectangle Union(this Rectangle self, Rectangle unionWith)
        {
            int minX = Math.Min(self.Left, unionWith.Left);
            int minY = Math.Min(self.Top, unionWith.Top);
            int maxX = Math.Max(self.Right, unionWith.Right);
            int maxY = Math.Max(self.Bottom, unionWith.Bottom);

            return new Rectangle(minX, minY, maxX - minX, maxY - minY);
        }
    }

    /// <summary>
    /// Class containing extension methods for RectangleF.
    /// </summary>
    public static class RectangleFExtensions
    {
        /// <summary>
        /// Get the center point of the rectangle.
        /// </summary>
        /// <param name="rect">The rectangle to create values from.</param>
        /// <returns>The point of values from the rectangle.</returns>
        public static PointF GetCenter(this RectangleF rect)
        {
            return new PointF((rect.Right + rect.Left) * 0.5f, (rect.Bottom + rect.Top) * 0.5f);
        }

        /// <summary>
        /// Union this rectangle with another rectangle.
        /// </summary>
        /// <param name="self">This rectangle.</param>
        /// <param name="unionWith">Another rectangle to union with.</param>
        /// <returns>The rectangle after union.</returns>
        public static RectangleF Union(this RectangleF self, RectangleF unionWith)
        {
            float minX = Math.Min(self.Left, unionWith.Left);
            float minY = Math.Min(self.Top, unionWith.Top);
            float maxX = Math.Max(self.Right, unionWith.Right);
            float maxY = Math.Max(self.Bottom, unionWith.Bottom);

            return new RectangleF(minX, minY, maxX - minX, maxY - minY);
        }
    }
}
