﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;

using EffectMaker.Foundation.Extensions;
using EffectMaker.Foundation.Log;

namespace EffectMaker.Foundation.Debugging.Profiling
{
    /// <summary>
    /// Measure the time span since the constructor is called
    /// until it's destruction or when the Stop() method is
    /// called.
    /// </summary>
    public class ProfileTimer : IDisposable
    {
        /// <summary>The default output format.</summary>
        private const string DefaultOutputFormat =
            "ProfileTimer => {0:F6} seconds spent for '{1}'";

        /// <summary>The dictionary that stores global profilers.</summary>
        private static readonly Dictionary<string, EffectMaker.Foundation.Debugging.Profiling.ProfileTimer> GlobalProfilers =
            new Dictionary<string, EffectMaker.Foundation.Debugging.Profiling.ProfileTimer>();

        /// <summary>The counter to prevent nested start timer calls.</summary>
        private int startCount = 0;

        /// <summary>The time when the timer starts.</summary>
        private DateTime startTime;

        /// <summary>The time when the timer stopped.</summary>
        private DateTime endTime;

        /// <summary>The elapsed time.</summary>
        private TimeSpan elapsedTime = new TimeSpan(0);

        /// <summary>The destination to output the elapsed time.</summary>
        private string outputDestination = null;

        /// <summary>The log level to output the elapsed time.</summary>
        private LogLevels outputLogLevel = LogLevels.Profile;

        /// <summary>The format string to output the elapsed time.</summary>
        private string outputFormat = DefaultOutputFormat;

        /// <summary>
        /// Constructor.
        /// </summary>
        /// <param name="tag">The tag for the timer.</param>
        public ProfileTimer(string tag)
        {
            this.Tag = tag;
            this.Resume();
        }

        /// <summary>
        /// Constructor.
        /// </summary>
        /// <param name="outputDestination">The destination to output the elapsed time.</param>
        /// <param name="outputLogLevel">The log level to output the elapsed time.</param>
        /// <param name="outputFormat">The format string to output the elapsed time.</param>
        public ProfileTimer(
            string outputDestination,
            LogLevels outputLogLevel,
            string outputFormat)
        {
            this.Tag = string.Empty;
            this.outputDestination = outputDestination;
            this.outputLogLevel = outputLogLevel;
            this.outputFormat = outputFormat;
            this.Resume();
        }

        /// <summary>
        /// Get or set the format string to output the elapsed time.
        /// </summary>
        public string OutputFormat
        {
            get { return this.outputFormat; }
            set { this.outputFormat = value; }
        }

        /// <summary>
        /// Get the flag indicating whether this timer is stopped.
        /// </summary>
        public bool IsStopped
        {
            get { return this.startCount <= 0; }
        }

        /// <summary>
        /// Get the tag of the timer.
        /// </summary>
        public string Tag { get; set; }

        /// <summary>
        /// Get the elapsed time since the timer started.
        /// </summary>
        public TimeSpan ElapsedTime
        {
            get
            {
                if (this.IsStopped == false)
                {
                    return this.elapsedTime + (DateTime.Now - this.startTime);
                }
                else
                {
                    return this.elapsedTime + (this.endTime - this.startTime);
                }
            }
        }

        /// <summary>
        /// Start a global profiler, all the time counted with the same tag will be added up.
        /// </summary>
        /// <param name="tag">The tag for the profiler.</param>
        public static void StartGlobalProfiler(string tag)
        {
            ProfileTimer profiler;
            if (GlobalProfilers.TryGetValue(tag, out profiler) == false)
            {
                profiler = new ProfileTimer(tag);
                GlobalProfilers.Add(tag, profiler);
            }

            profiler.Resume();
        }

        /// <summary>
        /// Stops a global profiler, all the time counted with the same tag will be added up.
        /// </summary>
        /// <param name="tag">The tag for the profiler.</param>
        public static void StopGlobalProfiler(string tag)
        {
            ProfileTimer profiler;
            if (GlobalProfilers.TryGetValue(tag, out profiler) == true)
            {
                profiler.Pause();
            }
        }

        /// <summary>
        /// Stops the total elapsed time for the global profiler.
        /// </summary>
        /// <param name="tag">The tag for the profiler.</param>
        public static void OutputGlobalProfiler(string tag)
        {
            ProfileTimer profiler;
            if (GlobalProfilers.TryGetValue(tag, out profiler) == true)
            {
                profiler.OutputElapsedTime();
            }
        }

        /// <summary>
        /// Stops the total elapsed time for the global profiler.
        /// </summary>
        public static void OutputAllGlobalProfilers()
        {
            GlobalProfilers.ForEach(p => p.Value.OutputElapsedTime());
        }

        /// <summary>
        /// Clear all the global profiler.
        /// </summary>
        public static void ClearAllGlobalProfilers()
        {
            GlobalProfilers.Clear();
        }

        /// <summary>
        /// Stop and dispose the timer.
        /// </summary>
        public void Dispose()
        {
            this.Stop(false);

#if DEBUG
            this.OutputElapsedTime();
#endif
        }

        /// <summary>
        /// Pause the timer.
        /// </summary>
        public void Pause()
        {
            --this.startCount;

            if (this.startCount == 0)
            {
                this.endTime = DateTime.Now;
            }
        }

        /// <summary>
        /// Pause the timer.
        /// </summary>
        public void Resume()
        {
            if (this.startCount == 0)
            {
                this.elapsedTime += this.endTime - this.startTime;
                this.startTime = DateTime.Now;
            }

            ++this.startCount;
        }

        /// <summary>
        /// Stop the timer.
        /// </summary>
        /// <param name="outputLog">True to output elapsed seconds to the debug console.</param>
        public void Stop(bool outputLog = true)
        {
            this.Pause();
#if DEBUG
            if (outputLog == true)
            {
                this.OutputElapsedTime();
            }
#endif
        }

        /// <summary>
        /// Output elapsed time.
        /// </summary>
        public void OutputElapsedTime()
        {
            if (string.IsNullOrEmpty(this.outputDestination) == true)
            {
                Logger.LogWithDepth(
                    2,
                    this.outputLogLevel,
                    this.outputFormat,
                    this.ElapsedTime.TotalSeconds,
                    this.Tag);
            }
            else
            {
                Logger.LogWithDepth(
                    2,
                    this.outputDestination,
                    this.outputLogLevel,
                    this.outputFormat,
                    this.ElapsedTime.TotalSeconds,
                    this.Tag);
            }
        }
    }
}
