﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef __ShaderConverterHelper_H__
#define __ShaderConverterHelper_H__

#include "ShaderCodeListCs.h"
#include "ShaderConverterWrapper_PCH.h"


using namespace System::Collections::Generic;
using namespace System::Runtime::InteropServices;
using namespace EffectMaker::ShaderConverterLib;

namespace EffectMaker {
namespace ShaderConverterWrapper {

public ref class ShaderConverterHelper
{
public :
    /// <summary>
    /// Constructor.
    /// </summary>
    /// <param name="converterFolderPath">The converter folder path.</param>
    /// <param name="shaderFolderPath">The shader folder path.</param>
    /// <param name="pAppMsgForwarder">The message forwarder to forward message to the applicaiton.</param>
    /// <param name="isCommandLineMode">The flag indicating whether the application is running under command line.</param>
    /// <param name="shaderAlignment">シェーダアライメント</param>
    /// <param name="binaryPosition">バイナリ書き込み位置</param>
    /// <param name="binaryPosition">バイナリ書き込み位置</param>
    ShaderConverterHelper(String ^converterFolderPath,
                          String ^shaderFolderPath,
                          IDotNetMessageForwarder ^pAppMsgForwarder,
                          int convertMode,
                          bool isCommandLineMode,
                          int jobsNumber,
                          int shaderAlignment,
                          int binaryPosition,
                          bool isSubBinaryConverting );

    /// <summary>
    /// Destructor.
    /// </summary>
    virtual ~ShaderConverterHelper();

    /// <summary>
    /// シェーダコードを設定します.
    /// </summary>
    void SetShaderCodes(ShaderCodeListCs ^pShaderCodes);

    /// <summary>
    /// Convert shader binary.
    /// </summary>
    /// <param name="pDataArray">The array of input emitter data.</param>
    /// <param name="pShaderBinary">The output shader binary data.</param>
    /// <param name="pErrorList">The array of compile error information.</param>
    /// <param name="shaderCount">The number of shaders being compiled.</param>
    /// <returns>True on success.</returns>
    bool Convert(cli::array<ShaderConversionInputData^> ^pDataArray,
                 [Out] cli::array<System::Byte>^% pShaderBinary,
                 [Out] List<ShaderCompileErrorInfo^>^% pErrorList,
                 [Out] int% shaderCount);

    /// <summary>
    /// Generate shader code for the given emitter.
    /// </summary>
    /// <param name="pInputData">The input emitter data.</param>
    /// <param name="pVertexShader">The output vertex shader code.</param>
    /// <param name="pFragmentShader">The output fragment shader code.</param>
    /// <param name="pComputeShader">The output compute shader code.</param>
    /// <returns>True on success.</returns>
    bool GenerateShaderCode(ShaderConversionInputData ^pInputData,
                            [Out] System::String^% pVertexShader,
                            [Out] System::String^% pFragmentShader,
                            [Out] System::String^% pComputeShader);

    /// <summary>
    /// Generate shader assembly for the given emitter.
    /// </summary>
    /// <param name="pInputData">The input emitter data.</param>
    /// <param name="pVertexShader">The output vertex shader assembly.</param>
    /// <param name="pFragmentShader">The output fragment shader assembly.</param>
    /// <param name="pComputeShader">The output compute shader assembly.</param>
    /// <param name="pErrorList">The array of compile error information.</param>
    /// <returns>True on success.</returns>
    bool GenerateShaderAssembly(ShaderConversionInputData ^pInputData,
                                [Out] System::String^% pVertexShader,
                                [Out] System::String^% pFragmentShader,
                                [Out] System::String^% pComputeShader,
                                [Out] List<ShaderCompileErrorInfo^>^% pErrorList);

protected :
    /// <summary>
    /// Finalize.
    /// </summary>
    !ShaderConverterHelper();

private :
    /// <summary>
    /// Convert the input data into the native input data buffer
    /// so the shader converter can access it.
    /// </summary>
    /// <param name="pDataArray">The array of input emitter data.</param>
    void ConvertInputData(cli::array<ShaderConversionInputData^> ^pDataArray);

    /// <summary>
    /// Convert the input data into the native input data buffer
    /// so the shader converter can access it.
    /// </summary>
    /// <param name="pInputData">The input emitter data.</param>
    void ConvertInputData(ShaderConversionInputData ^pInputData);

    /// <summary>
    /// Convert and copy the managed input data into the native memory,
    /// so the shader converter can access it.
    /// </summary>
    /// <param name="pManagedData">The managed input data.</param>
    /// <param name="ppEmitterData">The pointer to the native emitter data.</param>
    /// <param name="ppRawData">The pointer to the native emitter binary and custom shaders.</param>
    void SetupNativeInputData(ShaderConversionInputData ^pManagedData,
                              ShaderConverterEmitterData **ppEmitterData,
                              unsigned char **ppRawData);

    /// <summary>
    /// Set up the shader index and it's offset so we can send it back to the
    /// caller.
    /// </summary>
    /// <param name="pDataArray">The array of emitter data to be set up.</param>
    void SetupShaderIndexForOutput(cli::array<ShaderConversionInputData^> ^pDataArray);

    /// <summary>
    /// Release the memory allocated for the native input data.
    /// </summary>
    void ReleaseInputDataBuffer();

    /// <summary>
    /// Release the memory allocated for the shader binary data.
    /// </summary>
    void ReleaseShaderBinaryBuffer();

private :
    ShaderConverter   *m_pConverter;
    IMessageForwarder *m_pNativeMsgForwarder;

    int                m_inputEmitterCount;
    ShaderCodeListCpp *m_pShaderCodes;

    unsigned char     *m_pInputBuffer;
    unsigned char     *m_pShaderBinaryBuffer;

    char              *m_szConverterFolderPath;
    char              *m_szShaderFolderPath;

    int               m_bCommandLineMode;
    int               m_jobsNumber;
    int               m_shaderAlignment;
    int               m_binaryPosition;
    int               m_isSubBinaryConverting;
};

} // namespace ShaderConverterWrapper
} // namespace EffectMaker


#endif // __ShaderConverterHelper_H__
