﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once
#ifndef __GFDFILESHADER_H__
#define __GFDFILESHADER_H__

#ifdef __cplusplus
extern "C"
{
#endif // __cplusplus

/// @addtogroup GFDFileShaderGroup
/// @{

/// \brief Struct for containing GX2 Shaders
///

/// \brief Creates a gsh file based on the passed in GFDShaders
///
/// \param pFilename     Name of the output file
/// \param gpuVer        GPU version
/// \param swapMode      Endian swap mode for shader data (GPU data)
/// \param alignMode     Align mode for adding data specific allignment size padding block before image data (GPU data)
/// \param numShader     Number of shaders in pShaders
/// \param pShaders      Input GFDShaders array pointer used to create gsh file.
///                      If it's includes GX2XXXShader with NULL pointer, it's ignored.
//                       It's used for creating a separate shader binary file.
//
#if CAFE_OS_SDK_VERSION >= 21104
GFD_DECLSPEC bool GFD_API GFDWriteMemShader(char** pShader, GFDGPUVersion gpuVer, GFDEndianSwapMode swapMode, GFDAlignMode alignMode, u32 numShader, const GFDShaders *pShadersOrig, u32 *shaderSize );

/// \brief Creates a gsh file based on the passed in GFDShaders2
///
/// \param pFilename     Name of the output file
/// \param gpuVer        GPU version
/// \param swapMode      Endian swap mode for shader data (GPU data)
/// \param alignMode     Align mode for adding data specific allignment size padding block before image data (GPU data)
/// \param numShader     Number of shaders in pShaders
/// \param pShaders      Input GFDShaders2 array pointer used to create gsh file.
///                      If it's includes GX2XXXShader with NULL pointer, it's ignored.
//                       It's used for creating a separate shader binary file.
/// \return true or false
///
/// \donotcall \notthreadsafe \hostonly \enddonotcall
///
GFD_DECLSPEC bool GFD_API GFDWriteMemShader2(char** pShader, GFDGPUVersion gpuVer, GFDEndianSwapMode swapMode, GFDAlignMode alignMode, u32 numShader, const GFDShaders2 *pShaders, u32 *shaderSize );
#else
GFD_DECLSPEC bool GFD_API GFDWriteMemShader(char** pFilename, GFDGPUVersion gpuVer, GFDEndianSwapMode swapMode, GFDAlignMode alignMode, u32 numShader, GFDShaders *pShaders, u32 *shaderSize );
#endif
/// \brief Appends shaders based on the passed in GFDShaders to an existing gsh file
///
/// \param pFilename     Name of the file to append shaders
/// \param gpuVer        GPU version
/// \param swapMode      Endian swap mode for shader data (GPU data)
/// \param alignMode     Align mode for adding data specific allignment size padding block before image data (GPU data)
/// \param numShader     Number of shaders in pShaders
/// \param pShaders      Input GFDShaders array pointer used to create gsh file.
///                      If it's includes GX2XXXShader with NULL pointer, it's ignored.
//                       It's used for creating a separate shader binary file.
//
GFD_DECLSPEC bool GFD_API GFDAppendWriteMemShader(char* pFilename, GFDGPUVersion gpuVer, GFDEndianSwapMode swapMode, GFDAlignMode alignMode, u32 numShader, GFDShaders *pShaders);

/// \brief Write out a header file containing all the attributes, uniforms, samplers, and even the code as compilable code.
///
/// \param pFilename     Name of the output file
/// \param swapMode      Endian swap mode for shader data (GPU data)
/// \param pShaders      Input GFDShaders array pointer used to create gsh file.
///                      If it's includes GX2XXXShader with NULL pointer, it's ignored.
//                       It's used for creating a separate shader binary file.
//
GFD_DECLSPEC bool GFD_API GFDWriteMemShaderAsCode(char* pFilename, GFDEndianSwapMode swapMode, GFDShaders *pShaders);

/// @}

GX2VertexShader* GFDCopyBlockVSH(GX2VertexShader *pVS);
GX2PixelShader*  GFDCopyBlockPSH(GX2PixelShader *pPS);
u32              GDFGetVertexShaderNum32( void* pData );
u32              GDFGetPixelShaderNum32( void* pData );
GX2VertexShader* GDFGetVertexShader32To64( u32 index , void* pData );
GX2PixelShader*  GDFGetPixelShader32To64( u32 index , void* pData );

/// A block of null terminated strings all stored in the same char array
/// All strings are padded with zeros at the end to be integer multiple of words (4) long.
typedef struct _GFDStringTable
{
    u32    m_n;      ///< Current number of strings
    u32    m_nDB;    ///< Current size of pDataBlock in chars
    u32    m_maxDB;  ///< Max size of data block, again in chars
    char * m_pDB;    ///< Block of data containing all the strings;
} GFDStringTable;

/// A table of data segments, stored contiguously in the same int array as [Size, Data] pairs.
/// Data Tables are very similar to String Tables.  They just point arrays of data
/// rather than arrays null terminated strings.
typedef struct _GFDDataTable
{
    u32    m_n;      ///< Current number of data blocks in hunk (good for error checking)
    u32    m_nDB;    ///< Current size (e.g. next writeable offset) of m_pDB in chars
    u32    m_maxDB;  ///< Max size of data block, again in chars
    char * m_pDB;    ///< Block of data containing all the data
} GFDDataTable;

typedef struct _GFDLoopVar
{
    u32 reg[GX2_NUM_LOOP_VAR_U32_WORDS];
} GFDLoopVar;

#ifdef __cplusplus
}
#endif // __cplusplus

#endif  // __GFDFILESHADER_H__
