﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef __ShaderConverterLib_MessageBase_H__
#define __ShaderConverterLib_MessageBase_H__


namespace EffectMaker {
namespace ShaderConverterLib {

    /// <summary>
    /// The base message class.
    /// The messages can be forwarded to the application for displaying.
    /// </summary>
    class MessageBase
    {
    public :
        /// <summary>
        /// Get the message name.
        /// </summary>
        /// <returns>The message name.</returns>
        virtual const char* GetMessageName() const = 0;

        /// <summary>
        /// Get the argument count of this message.
        /// </summary>
        /// <returns>The argument count.</returns>
        virtual int GetArgumentCount() const = 0;

        /// <summary>
        /// Get the comma-separated argument names.
        /// </summary>
        /// <returns>The argument names.</returns>
        virtual const char* GetArgumentNames() const = 0;

        /// <summary>
        /// Get the argument value at the specified index.
        /// </summary>
        /// <returns>The argument value.</returns>
        virtual const char* GetArgument(int index) const = 0;

        /// <summary>
        /// Send this message to the application for processing or displaying.
        /// </summary>
        void Send();
    };

    /// <summary>
    /// Helper macro for converting __VA_ARGS__ to string.
    /// This makes sure the macro passes compile if __VA_ARGS__ is empty.
    /// </summary>
    #define __VA_ARGS_2_STR(arg, ...) #arg
    #define VA_ARGS_2_STR(...) __VA_ARGS_2_STR(__VA_ARGS__, FAKE_ARG)

    /// <summary>
    /// Define a message class which inherits from MessageBase.
    /// </summary>
    /// <param name="name">The name of the message class.</param>
    /// <param name="...">The arguments for the message class.</param>
    #define DEFINE_MESSAGE(name, ...) \
        struct name : public MessageBase \
        { \
        public: \
            enum ARGS_##name { ARG_BEGIN = -1, __VA_ARGS__, ARG_COUNT }; \
            name() \
            { \
                m_arguments = new char*[ARG_COUNT]; \
                memset(m_arguments, 0, ARG_COUNT * sizeof(char*)); \
            }; \
            name(name &src) \
            { \
                m_arguments = new char*[ARG_COUNT]; \
                size_t size = ARG_COUNT * sizeof(char*); \
                memcpy_s(m_arguments, size, src.m_arguments, size); \
            }; \
            ~name() \
            { \
                for (int i = 0; i < ARG_COUNT; ++i) \
                { \
                    if (m_arguments[i] != NULL) free(m_arguments[i]); \
                } \
                delete[] m_arguments; \
            }; \
            void SetArgument(ARGS_##name argName, const char *szFormat, ...) \
            { \
                va_list args; \
                va_start(args, szFormat); \
                int len = _vscprintf(szFormat, args) + 1; \
                char *pBuffer = (char*)malloc(len * sizeof(char)); \
                vsprintf_s(pBuffer, len, szFormat, args); \
                va_end(args); \
                m_arguments[argName] = pBuffer; \
            }; \
            virtual const char* GetMessageName() const { return #name; }; \
            virtual int GetArgumentCount() const { return ARG_COUNT; }; \
            virtual const char* GetArgumentNames() const { return VA_ARGS_2_STR(__VA_ARGS__); }; \
            virtual const char* GetArgument(int index) const { return m_arguments[index]; }; \
        private: \
            char** m_arguments; \
        }

} // namespace ShaderConverterLib
} // namespace EffectMaker

#endif // __ShaderConverterLib_MessageBase_H__
