﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// これは メイン DLL ファイルです。

#include "stdafx.h"
#include <memory>
//#include <string>
#include "BinaryConvertUtils.h"
#include <msclr/marshal.h>

#include <float.h>
////#pragma fenv_access (on)

#define CVTR_DLL L"3dBinaryConverter.dll"
//#define DLL64 L"win64\\3dBinaryConverter.dll"
#define DLL32 L"3dBinaryConverter.dll"

#define PROC_ADDRESS(handle, name, type)                                \
auto name = reinterpret_cast<type>(GetProcAddress(handle, #name));      \
    if (name == nullptr){ return false;	}								\


namespace BinaryConvertUtils {

const wchar_t* optIgnoreAssign = L"--ignore-assign";
const wchar_t* optEnd = L"";
const wchar_t* optViewer = L"--viewer";
const wchar_t* optSilent = L"--silent";
const wchar_t* optAlignmentSort = L"--disable-attribute-alignment-sort";

const wchar_t* optionsIgnoreAssign[] = { optIgnoreAssign, optEnd,
                                         optSilent, optEnd,
                                         optAlignmentSort, optEnd,
                                         NULL };
const wchar_t* optionsForViewer[] = { optViewer, optEnd,
                                      optSilent, optEnd,
                                      optAlignmentSort, optEnd,
                                      NULL };

//const wchar_t* options[] = {opt, opt2, NULL};

const wchar_t* nullOptions[] = {NULL, NULL};


//-----------------------------------------------------------------------------
// In 3dBinaryConverter.dll, they might changed the floating point control modes
// and set the precision control to use the lowest (24 bits) precision.
// We need to keep NW4F and CE2 build consistent, and since we don't need such
// high precision, we set it to the low precision control mode for CE2 build.
//-----------------------------------------------------------------------------
void BinaryConverter::SetupFloatControlMode()
{
    ////unsigned int control_word;
    ////_controlfp_s( &control_word, _PC_24, _MCW_PC );
}



BinaryConverter::BinaryConverter()
    : isInitialized_(false)
    , hDLL_(nullptr)
    , GetCvtrVersion_(nullptr)
    , Initialize_(nullptr)
    , Clear_(nullptr)
    , Shutdown_(nullptr)
    , SetOptions_(nullptr)
    , AddFile_(nullptr)
    , CalcSize_(nullptr)
    , Convert_(nullptr)
    , Bind_(nullptr)
    , EndianSwap_(nullptr)
{
}

bool BinaryConverter::Initialize(System::String^ baseFolder, bool flg64bit)
{
    System::String^ fullpath = nullptr;
// 	if(flg64bit)
// 	{
// 		hDLL_ = LoadLibrary(DLL64);
// 	}
// 	else
// 	{
 		fullpath = System::IO::Path::Combine(baseFolder, DLL32);
 		const wchar_t* pDllPath =
            (const wchar_t*)(void*)System::Runtime::InteropServices::Marshal::StringToHGlobalUni( fullpath );

        {
            volatile float  fOrigValue1   = 759.76f;
            volatile double fDoubleValue1 = (double)fOrigValue1;
            volatile double fCompare1     = fDoubleValue1 * 100.0;
        }
        ////unsigned int control_word;
        ////_controlfp_s(&control_word, 0, 0);
        hDLL_ = LoadLibrary( pDllPath );
        ////_controlfp_s(&control_word, 0, 0);

        {
            volatile float  fOrigValue2   = 759.76f;
            volatile double fDoubleValue2 = (double)fOrigValue2;
            volatile double fCompare2     = fDoubleValue2 * 100.0;
        }
//		hDLL_ = LoadLibrary( DLL32 );
// 	}
//
    if (hDLL_ == nullptr)
    {
        // エラー
        return false;
    }

    // DLL から関数の読み込み。
 	GetCvtrVersion_	= (FUNC_GetCvtrVersion)GetProcAddress(hDLL_,	"nng3dToolBinaryConverterGetConverterVersion");
    Initialize_     = (FUNC_Initialize)GetProcAddress(hDLL_,		"nng3dToolBinaryConverterInitialize");
 	Clear_			= (FUNC_Clear)GetProcAddress(hDLL_,				"nng3dToolBinaryConverterClear");
    Shutdown_       = (FUNC_Shutdown)GetProcAddress(hDLL_,          "nng3dToolBinaryConverterShutdown");
 	SetOptions_		= (FUNC_SetOptions)GetProcAddress(hDLL_,		"nng3dToolBinaryConverterSetOptions");
 	AddFile_		= (FUNC_AddFile)GetProcAddress(hDLL_,			"nng3dToolBinaryConverterAddFile");
 	CalcSize_		= (FUNC_CalcSize)GetProcAddress(hDLL_,			"nng3dToolBinaryConverterCalculateSize");
 	Convert_		= (FUNC_Convert)GetProcAddress(hDLL_,			"nng3dToolBinaryConverterConvert");
 	Bind_			= (FUNC_Bind)GetProcAddress(hDLL_,				"nng3dToolBinaryConverterBind");
 	EndianSwap_		= (FUNC_EndianSwap)GetProcAddress(hDLL_,		"nng3dToolBinaryConverterSwapEndian");

    // コンバータを初期化
    bool resInitialize = Initialize_();
    if (resInitialize == false)
    {
        return false;
    }

    // 関数読み込みに成功したら、初期化フラグをtrueに
    isInitialized_ = true;

    return true;
}

void BinaryConverter::Destroy()
{
    // 未初期化ならｍなにもしない
    if (isInitialized_ == false)
    {
        return;
    }

    // コンバータを解放
//    Shutdown_();

    // 関数ポインタ解放

    // DLL解放
//	FreeLibrary(hDLL_);

    // 未初期化に
    isInitialized_ = false;
}

bool BinaryConverter::SaveFile(const char* outData, size_t outSize, const wchar_t* path)
{
    FILE* file;
    errno_t err = _wfopen_s(&file, path, L"wb");
    if (err)
    {
        return false;
    }

    fwrite(outData, outSize, 1, file);
    fclose(file);

    return true;
}

int BinaryConverter::AddFileInternal(
    array<Byte>^ pInputStreamBuf, int inputSize, String^ filePath, String^ name, String^ ext, array<String^>^ options)
{
    if(isInitialized_ == false)
    {
        return (int)errCode::notInitialize;
    }

    using namespace msclr::interop;
    marshal_context ^ context = gcnew marshal_context();

    pin_ptr<Byte> pNative = &pInputStreamBuf[0];
    void* str = static_cast<void*>(pNative);
    size_t inputSize_t = inputSize;

    // パスの配列生成
    const wchar_t* path[3];
    path[0] = context->marshal_as<const wchar_t*>(filePath);
    path[1] = context->marshal_as<const wchar_t*>(name);
    path[2] = context->marshal_as<const wchar_t*>(ext);

    // オプションリストを動的に生成するように変更
    const wchar_t** AddFileOptions = NULL;
    if (options != nullptr)
    {
        AddFileOptions = new const wchar_t*[options->Length + 1];
        for (int i = 0; i < options->Length; ++i)
        {
            if (String::IsNullOrEmpty(options[i]))
            {
                AddFileOptions[i] = NULL;
            }
            else
            {
                AddFileOptions[i] = context->marshal_as<const wchar_t*>(options[i]);
            }
        }

        AddFileOptions[options->Length] = NULL;

        const wchar_t *a = AddFileOptions[0];
        const wchar_t *b = AddFileOptions[1];
        const wchar_t *c = AddFileOptions[2];
        const wchar_t *d = AddFileOptions[3];
    }
    else
    {
        AddFileOptions = new const wchar_t*[1];
        AddFileOptions[0] = NULL;
    }

    bool success = true;
    try{
        // ファイルの追加
        success = AddFile_(str, inputSize_t, path, AddFileOptions);
        if (!success)
        {
            delete context;
            delete AddFileOptions;
            return (int)errCode::AddFile;
        }
    }
    catch(...)
    {
        delete context;
        delete AddFileOptions;
        return (int)errCode::UnknownErr;
    }

    delete context;
    delete AddFileOptions;
    return (int)errCode::noErr;
}

int BinaryConverter::PreBinarizeUsingIgnoreAssignOption()
{
    return PreBinarize(optionsIgnoreAssign);
}

int BinaryConverter::PreBinarizeUsingViewerOption()
{
    return PreBinarize(optionsForViewer);
}

int BinaryConverter::PreBinarize(const wchar_t** options)
{
    if (isInitialized_ == false)
    {
        return (int)errCode::notInitialize;
    }

    bool success = SetOptions_(options);

    try {

        Clear_();

        // オプションの登録。
        if (!SetOptions_(options))
        {
            return (int)errCode::SetOptions;
        }

    }
    catch (...)
    {
        return (int)errCode::UnknownErr;
    }

    return (int)errCode::noErr;
}

int BinaryConverter::AddFile(array<Byte>^ pInputStreamBuf, int inputSize, String^ filePath, String^ name, String^ ext)
{
    return AddFileInternal(pInputStreamBuf, inputSize, filePath, name, ext, nullptr);
}

int BinaryConverter::AddExternalFile(array<Byte>^ pInputStreamBuf, int inputSize, String^ filePath, String^ name, String^ ext, int alignment)
{
    array<String^>^ optionList = gcnew array<String^>(4);
    optionList[0] = "--external-file";
    optionList[1] = nullptr;
    optionList[2] = "--external-file-alignment";
    optionList[3] = String::Format("{0}", alignment);

    return AddFileInternal(pInputStreamBuf, inputSize, filePath, name, ext, optionList);
}

int BinaryConverter::Binarize(Stream^ stream)
{
    if(isInitialized_ == false)
    {
        return (int)errCode::notInitialize;
    }

    using namespace msclr::interop;
    marshal_context ^ context = gcnew marshal_context();

    bool success = true;
    try{
        // 出力ファイルサイズの計算。
        size_t outputSize = CalcSize_();
        if (outputSize == 0)
        {
            delete context;
            return (int)errCode::CalcSize;
        }

        // 出力ファイル用バッファ作成
        char* outBuf = static_cast<char*>(_aligned_malloc(outputSize, 0x100));
        if(!outBuf)
        {
            delete context;
            return (int)errCode::MallocOutBuf;
        }

        // 変換処理。
        success = Convert_(outBuf, outputSize);
        if (!success)
        {
            _aligned_free(outBuf);
            delete context;
            return (int)errCode::Convert;
        }

        // バインド
        success = Bind_(outBuf, outputSize);
        if (!success)
        {
            _aligned_free(outBuf);
            delete context;
            return (int)errCode::Bind;
        }

        // エンディアンスワップ
        success = EndianSwap_(outBuf, outputSize);
        if (!success)
        {
            _aligned_free(outBuf);
            delete context;
            return (int)errCode::EndianSwap;
        }

        // char* -> array<Byte>^ に変換
        array<Byte> ^managedBuf = gcnew array<Byte>(static_cast<int>(outputSize));
        System::Runtime::InteropServices::Marshal::Copy((IntPtr)outBuf, managedBuf, 0, static_cast<int>(outputSize));

        // とりあえず、書き込んでみる。
        stream->Write(managedBuf, 0, static_cast<int>(outputSize));

        if(outBuf)
        {
            _aligned_free(outBuf);
        }
    }
    catch(...)
    {
        delete context;
        return (int)errCode::UnknownErr;
    }

    delete context;
    return (int)errCode::noErr;
}


}	// namespace BinaryConvertUtil
