﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Data;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows;
using System.Windows.Forms;
using EffectMaker.Application.Properties;
using EffectMaker.BusinessLogic.Protocol;
using EffectMaker.BusinessLogic.SpecDefinitions;
using EffectMaker.Communicator;
using EffectMaker.Foundation.Attributes;
using EffectMaker.UIControls.Xaml;

namespace EffectMaker.Application.OptionPanes
{
    /// <summary>
    /// スペックの情報を表示するクラス
    /// </summary>
    [DisplayOrder(10)]
    public partial class SpecOptionPane : UserControl, IOptionPane
    {
        /// <summary>
        /// コンストラクタ
        /// </summary>
        public SpecOptionPane()
        {
            InitializeComponent();

            // 全スペック名をCombBoxに入れる.
            foreach (var spec in SpecManager.SpecDefinitions)
            {
                this.specComboBox.Items.Add(spec.Name);
            }

            // ユーザがEffectMaker上で使用しているスペックを代入
            this.specComboBox.CurrentSpecIndex = SpecManager.SelectedIndex;
            this.specComboBox.SelectedIndexChanged += new System.EventHandler(this.OnSelectedSpecChanged);

            // ユーザがEffectMaker上で使用しているスペックを指定させる
            this.specComboBox.SelectedIndex = SpecManager.SelectedIndex;
        }

        /// <summary>
        /// オプションウィンドウのTreeViewを再描画するdelegate
        /// </summary>
        public Action DrawTreeView { get; set; }

        /// <summary>
        /// Gets the option pane identifier.
        /// </summary>
        public string Identifier
        {
            get { return "SpecConfig"; }
        }

        /// <summary>
        /// Gets the identifier of the parent, if any.
        /// </summary>
        public string ChildOf
        {
            get { return null; }
        }

        /// <summary>
        /// Gets the display text of the option pane.
        /// </summary>
        public string DisplayName
        {
            get { return Properties.Resources.OptionSectionSpecConfig; }
        }

        /// <summary>
        /// Called when initialized.
        /// </summary>
        public void OnInitialize()
        {
        }

        /// <summary>
        /// Called when terminated.
        /// </summary>
        public void OnTerminate()
        {
        }

        /// <summary>
        /// Called when the Accept button is clicked.
        /// </summary>
        /// <returns>入力に不正がなければtrue,入力が不正であればfalse.</returns>
        public bool OnAccept()
        {
            return true;
        }

        /// <summary>
        /// Called when the Cancel button is clicked.
        /// </summary>
        public void OnCancel()
        {
        }

        /// <summary>
        /// プロジェクトコンフィグが読み込まれた際に、その内容にUIを更新します。
        /// </summary>
        public void OnProjectSettingChanged()
        {
        }

        /// <summary>
        /// プロジェクトコンフィグのパス設定が有効か無効かを返す。
        /// </summary>
        /// <returns>
        /// 有効ならtrue、無効ならfalse.
        /// </returns>
        public bool OnValidationCheck()
        {
            return true;
        }

        /// <summary>
        /// プロジェクトコンフィグをファイル出力する際に各UIの内容を出力用インスタンスに収集します。
        /// </summary>
        public void OnExportProjectSetting()
        {
        }

        /// <summary>
        /// オプションウィンドウ上で選択されたスペックの情報を表示する
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void OnSelectedSpecChanged(object sender, EventArgs e)
        {
            var currentSpec = SpecManager.SpecDefinitions[this.specComboBox.SelectedIndex];
            this.specTreeView.Nodes.Clear();

            // スペック名
            string name = currentSpec.Name;
            if (currentSpec.AliasList != null && currentSpec.AliasList.Any())
            {
                // エイリアス
                string result = currentSpec.AliasList.Aggregate(string.Empty, (current, a) => current + (a + " "));
                if (result.EndsWith(" "))
                {
                    result = result.Substring(0, result.Length - 1);
                }

                name += " (" + result + ")";
            }

            this.specTreeView.Nodes.Add(new TreeNode("Name: " + name));

            // バイナリヘッダ
            this.specTreeView.Nodes.Add(new TreeNode("BinaryHeader: " + currentSpec.BinaryHeader));

            // エンディアン
            var endian = currentSpec.CpuEndian == 0 ? "Little" : "Big";
            this.specTreeView.Nodes.Add(new TreeNode("CpuEndian: " + endian));
            endian = currentSpec.GpuEndian == 0 ? "Little" : "Big";
            this.specTreeView.Nodes.Add(new TreeNode("GpuEndian: " + endian));

            // グラフィクスAPI
            this.specTreeView.Nodes.Add(new TreeNode("GraphicsApi: " + currentSpec.GraphicsApi));

            // CPU限定
            this.specTreeView.Nodes.Add(new TreeNode("ForceCpuCalc: " + currentSpec.ForceCpuCalc));

            // アーキテクチャ
            this.specTreeView.Nodes.Add(new TreeNode("TargetArchitecture: " + (currentSpec.AddressSize == 4 ? "32bit" : "64bit")));

            // Viewerとの接続関係
            TreeNode connectionNodes = new TreeNode("Connections");
            foreach (var connectionContext in currentSpec.Connections)
            {
                TreeNode connectionContextNode = new TreeNode(connectionContext.Name);
                connectionContextNode.Nodes.Add("Identifier: " + connectionContext.Identifier);
                connectionContextNode.Nodes.Add("Encoding: " + connectionContext.Encoding.ToString());

                string platform;
                switch (connectionContext.Platform)
                {
                    case (int)PlatformType.Windows:
                        platform = "Windows";
                        break;
                    case (int)PlatformType.Cafe:
                        platform = "Cafe";
                        break;
                    case (int)PlatformType.CTR:
                        platform = "CTR";
                        break;
                    case (int)PlatformType.HTC:
                        platform = "HTC";
                        break;
                    default:
                        platform = "Windows";
                        break;
                }

                connectionContextNode.Nodes.Add("Platform: " + platform);
                connectionContextNode.Nodes.Add("IPAddress: " + connectionContext.IPAddress);
                connectionContextNode.Nodes.Add("Port: " + connectionContext.Port);
                connectionContextNode.Nodes.Add("Protocol: " + connectionContext.Protocol);
                connectionContextNode.Nodes.Add("PeerType: " + connectionContext.PeerType);
                connectionContextNode.Nodes.Add("IsBigEndian: " + connectionContext.IsBigEndian);
                connectionContextNode.Nodes.Add("IsReady: " + connectionContext.IsReady);
                connectionNodes.Nodes.Add(connectionContextNode);
            }

            this.specTreeView.Nodes.Add(connectionNodes);

            // Viewer起動連携周り
            if (currentSpec.PCViewerPath != null || currentSpec.TargetViewerPath != null)
            {
                var viewerPathNodes = new TreeNode("ViewerPaths");
                viewerPathNodes.Nodes.Add(new TreeNode("PCViewer: " + currentSpec.PCViewerPath));
                viewerPathNodes.Nodes.Add(new TreeNode("TargetViewer: " + currentSpec.TargetViewerPath));
                this.specTreeView.Nodes.Add(viewerPathNodes);
            }

            // Texture周り
            {
                var textureNodes = new TreeNode("TextureSettings");

                if (currentSpec.TexturePackingOption != null)
                {
                    textureNodes.Nodes.Add(new TreeNode("TileMode: " + currentSpec.TexturePackingOption.TileMode));
                }
                else if (currentSpec.ValidNativeTextureFormats != null && currentSpec.ValidNativeTextureFormats.Any())
                {
                    string formats = currentSpec.ValidNativeTextureFormats.Aggregate(
                        string.Empty, (current, format) => current + (format + " "));
                    textureNodes.Nodes.Add(new TreeNode("ValidNativeTextureFormats: " + formats));
                }

                textureNodes.Nodes.Add(new TreeNode("ForceNativeTexture: " + currentSpec.ForceNativeTexture));

                this.specTreeView.Nodes.Add(textureNodes);
            }

            // ShaderManager周り
            {
                var shaderNodes = new TreeNode("ShaderSettings");

                shaderNodes.Nodes.Add(new TreeNode("ManagerPath: " + currentSpec.ShaderConversionOption.ManagerPath));
                shaderNodes.Nodes.Add(new TreeNode("ManagerTypeName: " + currentSpec.ShaderConversionOption.ManagerTypeName));
                shaderNodes.Nodes.Add(new TreeNode("CodeFolderName: " + currentSpec.ShaderConversionOption.CodeFolderName));
                shaderNodes.Nodes.Add(new TreeNode("SpecDeclarationGeneric: " + currentSpec.ShaderConversionOption.SpecDeclarationGeneric));
                shaderNodes.Nodes.Add(new TreeNode("SpecDeclarationTarget: " + currentSpec.ShaderConversionOption.SpecDeclarationTarget));
                shaderNodes.Nodes.Add(new TreeNode("BinaryAlignment: " + currentSpec.ShaderConversionOption.BinaryAlignment));
                shaderNodes.Nodes.Add(new TreeNode("UboAlignment: " + currentSpec.ShaderConversionOption.UboAlignment));
                shaderNodes.Nodes.Add(new TreeNode("WriteCodeArray: " + currentSpec.ShaderConversionOption.WriteCodeArray));
                shaderNodes.Nodes.Add(new TreeNode("TargetIsPreCompile: " + currentSpec.ShaderConversionOption.TargetIsPreCompile));

                if (currentSpec.ShaderConversionOption.TargetIsPreCompile)
                {
                    shaderNodes.Nodes.Add(new TreeNode("ConverterOption: " + currentSpec.ShaderConversionOption.ConverterOption));
                    shaderNodes.Nodes.Add(new TreeNode("CodeType: " + currentSpec.ShaderConversionOption.CodeType));
                }

                this.specTreeView.Nodes.Add(shaderNodes);
            }

            // 見やすさのために、全ノードを展開しておく
            this.specTreeView.ExpandAll();
        }
    }
}
