﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.IO;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using EffectMaker.Application.Properties;
using EffectMaker.BusinessLogic.IO;
using EffectMaker.BusinessLogic.Options;
using EffectMaker.BusinessLogic.ProjectConfig;
using EffectMaker.Foundation.Attributes;
using EffectMaker.Foundation.Log;
using EffectMaker.Foundation.Serialization;
using EffectMaker.Foundation.Utility;
using EffectMaker.UIControls.Specifics.Behaviors;
using EffectMaker.UIDialogs.MessageDialogs;

namespace EffectMaker.Application.OptionPanes
{
    /// <summary>
    /// Project Basic options.
    /// </summary>
    [DisplayOrder(5)]
    public partial class ProjectBasicOptionPane : UserControl, IOptionPane
    {
        /// <summary>The list of the combiner definition input controls.</summary>
        private List<EffectCombinerDefInput> combinerDefInputControls =
            new List<EffectCombinerDefInput>();

        /// <summary>
        /// Initializes the ProjectBasicOptionPane instance.
        /// </summary>
        public ProjectBasicOptionPane()
        {
            this.InitializeComponent();

            this.btnProjectSettingsLoad.Click += this.OnProjectSettingsLoadButtonClick;
            this.btnProjectSettingsReset.Click += this.OnProjectSettingsClearButtonClick;
            this.btnProjectSettingsReload.Click += this.OnProjectSettingsReloadButtonClick;

            OptionUtil.SetCommentLabelState(this.lblProjectConfigFileComment);

            this.btnPresetLoad.Click += this.OnPresetLoadButtonClick;
            this.btnPresetReset.Click += this.OnPresetClearButtonClick;

            this.btnExportPrjconfig.Click += OnExportProjectConfig;
            OptionUtil.SetCommentLabelState(this.lblExportPrjconfig);

            OptionUtil.SetCommentLabelState(this.lblLinearWorkflowComment);

            OptionUtil.SetCommentLabelState(this.lblBufferOptionComment);

            OptionUtil.SetCommentLabelState(this.lblRegularGravityComment);
            this.chkEnableRegularGravity.CheckedChanged += (sender, args) =>
            {
                var value = this.chkEnableRegularGravity.Checked;
                this.numLength.Enabled = value;
                this.numFps.Enabled = value;
            };

            this.btnDocumentLoad.Click += this.OnDocumentLoadButtonClick;
            this.btnDocumentReset.Click += this.OnDocumentClearButtonClick;
            OptionUtil.SetCommentLabelState(this.lblDocumentFolderComment);
            this.txtDocumentFolder.FilePathChanged += (s, e) =>
            {
                this.btnDocumentReset.Enabled = !string.IsNullOrEmpty(this.txtDocumentFolder.FilePath);
                OptionUtil.OnCheckFilePathExisting(s, e);
                if (this.DrawTreeView != null)
                {
                    this.DrawTreeView();
                }
            };
            this.txtDocumentFolder.BackColor = Color.White;

            this.btnPresetReset.Enabled = false;

            // プリセットのフォルダパスが変更された場合は、フォルダの有無をチェックして、ツリービューを更新する.
            this.txtPresetFolder.FilePathChanged += (s, e) =>
            {
                this.btnPresetReset.Enabled = !string.Equals(this.txtPresetFolder.FilePath, string.Empty);
                OptionUtil.OnCheckFilePathExisting(s, e);

                if (this.DrawTreeView != null)
                {
                    this.DrawTreeView();
                }
            };

            this.btnProjectSettingsReset.Enabled = false;
            this.btnProjectSettingsReload.Enabled = false;

            this.txtProjectSettings.BackColor = Color.White;
            this.txtProjectSettings.FilePathChanged += (s, e) =>
            {
                this.btnProjectSettingsReset.Enabled = !string.IsNullOrEmpty(this.txtProjectSettings.FilePath);
                this.btnProjectSettingsReload.Enabled = !string.IsNullOrEmpty(this.txtProjectSettings.FilePath);
            };

            foreach (var box in this.Controls.OfType<OptionGroupBox>().Where(b => b != this.grpUserSettings))
            {
                box.BorderColor = Color.FromArgb(130, 150, 185);
            }

            this.lblEftCombinerFolderComment.Text = string.Format(
                "{0}{1}{2}",
                Resources.OptionProjectConfigOverwriteComment,
                Environment.NewLine,
                Resources.OptionProjectConfigCombinerDefinitionComment);
            OptionUtil.SetCommentLabelState(this.lblEftCombinerFolderComment);
        }

        /// <summary>
        /// デフォルトのプリセットフォルダパスを返します。
        /// </summary>
        public static string DefaultPresetFolderPath
        {
            get
            {
                return Path.Combine(
                    Path.GetDirectoryName(System.Reflection.Assembly.GetExecutingAssembly().Location),
                    "Presets");
            }
        }

        /// <summary>
        /// オプションウィンドウのTreeViewを再描画するdelegate
        /// </summary>
        public Action DrawTreeView
        {
            get;
            set;
        }

        /// <summary>
        /// Gets the option pane identifier.
        /// </summary>
        public string Identifier
        {
            get { return "ProjectConfig"; }
        }

        /// <summary>
        /// Gets the identifier of the parent, if any.
        /// </summary>
        public string ChildOf
        {
            get { return null; }
        }

        /// <summary>
        /// Gets the display text of the option pane.
        /// </summary>
        public string DisplayName
        {
            get { return Properties.Resources.OptionSectionProjectConfig; } // TODO: make it translatable
        }

        /// <summary>
        /// プロジェクト設定の変更をトリガーします。
        /// </summary>
        public Action TriggerProjectSetting { get; set; }

        /// <summary>
        /// 出力用のプロジェクトコンフィグを更新する処理をトリガーします。
        /// </summary>
        public Action ApplyProjectSettingForExport { get; set; }

        /// <summary>
        /// テクスチャの原点を取得または設定します。
        /// </summary>
        private TextureOriginMode TextureOriginMode
        {
            get
            {
                if (this.btnTextureOriginLowerLeft.Checked)
                {
                    return TextureOriginMode.LowerLeft;
                }
                else
                {
                    return TextureOriginMode.UpperLeft;
                }
            }

            set
            {
                if (value == TextureOriginMode.LowerLeft)
                {
                    this.btnTextureOriginLowerLeft.Checked = true;
                }
                else
                {
                    this.btnTextureOriginUpperLeft.Checked = true;
                }
            }
        }

        /// <summary>
        /// デプスの範囲を取得または設定します。
        /// </summary>
        private DepthMode DepthMode
        {
            get
            {
                if (this.btnDepthNearIsMinusW.Checked)
                {
                    return DepthMode.NearIsMinusW;
                }
                else
                {
                    return DepthMode.NearIsZero;
                }
            }

            set
            {
                if (value == DepthMode.NearIsMinusW)
                {
                    this.btnDepthNearIsMinusW.Checked = true;
                }
                else
                {
                    this.btnDepthNearIsZero.Checked = true;
                }
            }
        }

        /// <summary>
        /// テキストボックスへのドラッグアンドドロップ時の挙動について設定します.
        /// </summary>
        public void InitializeDragAndDrop()
        {
            // プロジェクト設定ファイルのドラッグ&ドロップ処理を設定
            {
                FileDragDropBehavior dragDropBehavior = new FileDragDropBehavior();
                dragDropBehavior.AllowExtensions = RegexUtility.ExtractExtensions(Properties.Resources.OptionDialogFilterXmlFiles);
                dragDropBehavior.DropResultChanged += (s, e) =>
                {
                    this.UpdateProjectSettingsTxt(dragDropBehavior.DropResult);
                };

                this.txtProjectSettings.Behaviors.Add(dragDropBehavior);
            }

            // プリセットフォルダのドラッグ&ドロップ処理を設定
            {
                // 全てのファイルとディレクトリのドラッグ&ドロップを許可する
                FileDragDropBehavior dragDropBehavior = new FileDragDropBehavior();
                dragDropBehavior.AllowAllExtensions = true;
                dragDropBehavior.AllowDirectory = true;
                dragDropBehavior.ConvertToDirectoryPath = true;
                dragDropBehavior.DropResultChanged += (s, e) =>
                {
                    this.txtPresetFolder.FilePath = dragDropBehavior.DropResult;
                };

                this.txtPresetFolder.Behaviors.Add(dragDropBehavior);
            }

            // ドキュメントファイルのドラッグ&ドロップ処理を設定
            {
                // ドキュメントパスは、フォルダとhtmlファイルのみドラッグ&ドロップを許可する
                FileDragDropBehavior dragDropBehavior = new FileDragDropBehavior();
                dragDropBehavior.AllowExtensions = RegexUtility.ExtractExtensions(Properties.Resources.OptionDialogFilterDocumentPath);
                dragDropBehavior.AllowDirectory = true;
                dragDropBehavior.DropResultChanged += (s, e) =>
                {
                    this.txtDocumentFolder.FilePath = dragDropBehavior.DropResult;
                };

                this.txtDocumentFolder.Behaviors.Add(dragDropBehavior);
            }
        }

        /// <summary>
        /// 描画パスの設定
        /// </summary>
        /// <param name="pathList">
        /// パスリスト(省略した場合は現在の設定を読み込む)
        /// </param>
        public void InitializeDrawPath(List<EffectMakerProjectConfig.DrawPath> pathList)
        {
            var drawPaths = pathList;

            this.lstDrawPath.Items.Clear();

            // ヘッダを設定
            this.lstDrawPath.Columns.Clear();
            this.lstDrawPath.Columns.Add("Id");
            this.lstDrawPath.Columns.Add("Text");
            this.lstDrawPath.Columns.Add("Def1");
            this.lstDrawPath.Columns.Add("Def2");

            // 描画パスリストを設定.
            foreach (var drawPath in drawPaths)
            {
                var item = new ListViewItem(new string[]
                {
                    drawPath.Id.ToString(),
                    drawPath.Text,
                    drawPath.ShaderCompileDef1,
                    drawPath.ShaderCompileDef2,
                });

                this.lstDrawPath.Items.Add(item);
            }

            // 列のサイズを更新
            {
                this.lstDrawPath.AutoResizeColumns(ColumnHeaderAutoResizeStyle.ColumnContent);

                // ColumnHeaderAutoResizeStyle.HeaderSizeを指定すると
                // 最後の列が最大化されてしまうので自前でヘッダサイズに合わせる
                int[] minimumWidths = { 26, 36, 37, 37 };
                for (int i = 0; i < lstDrawPath.Columns.Count; ++i)
                {
                    lstDrawPath.Columns[i].Width = Math.Max(lstDrawPath.Columns[i].Width, minimumWidths[i]);
                }
            }
        }

        /// <summary>
        /// Called when initialized.
        /// </summary>
        public void OnInitialize()
        {
            FileEventOptions options = OptionStore.RootOptions.FileEvent;

            OptionUtil.TempConfig = OptionStore.ProjectConfig.Clone() as EffectMakerProjectConfig;

            this.txtProjectSettings.FilePath = options.UserSettingsFilePath;
            if (!string.IsNullOrEmpty(this.txtProjectSettings.FilePath))
            {
                this.CheckAndUpdateProjectConfig();
            }

            // ドラッグアンドドロップ時の設定
            this.InitializeDragAndDrop();

            // プリセットフォルダ
            this.txtPresetFolder.FilePath = OptionStore.ProjectConfig.FileEventProject.PresetFolderPath;

            // 描画パス周りの設定
            this.InitializeDrawPath(OptionStore.ProjectConfig.DrawPaths);

            // リニアワークフロー周りの設定
            this.chkOptionsLinearEditMode.Checked = OptionStore.ProjectConfig.LinearMode;

            // バッファオプションの設定
            this.TextureOriginMode = OptionStore.ProjectConfig.TextureOriginMode;
            this.DepthMode = OptionStore.ProjectConfig.DepthMode;

            // 標準重力の設定
            this.chkEnableRegularGravity.Checked = OptionStore.ProjectConfig.EnableRegularGravity;
            this.numLength.Enabled = OptionStore.ProjectConfig.EnableRegularGravity;
            this.numLength.Value = (decimal) OptionStore.ProjectConfig.UnitLength;
            this.UpdateUnitLengthPrecision();

            this.numFps.Enabled = OptionStore.ProjectConfig.EnableRegularGravity;
            this.numFps.Value = OptionStore.ProjectConfig.FpsForRegularGravity;

            // ドキュメントパス関連設定
            this.txtDocumentFolder.FilePath = OptionStore.ProjectConfig.DocumentPath;

            // エフェクト定義フォルダパスの入力UIを作成
            this.LoadCombinerDefInputControl(OptionStore.ProjectConfig.EffectCombinerBlockDefinitionPaths);

            // プロジェクト読み込みフラグ
            OptionUtil.IsNeedReloadProjectConfig = !this.OnValidationCheck();
        }

        /// <summary>
        /// Called when terminated.
        /// </summary>
        public void OnTerminate()
        {
        }

        /// <summary>
        /// Called when the Accept button is clicked.
        /// </summary>
        /// <returns>入力に不正がなければtrue,入力が不正であればfalse.</returns>
        public bool OnAccept()
        {
            OptionStore.RootOptions.FileEvent.UserSettingsFilePath = this.txtProjectSettings.FilePath;

            // プロジェクトコンフィグの読み直しがあった場合は、UIにない項目が更新されている可能性があるので、
            // TempConfigの内容をそのままセットすることで全項目を適用する。
            // プロジェクトコンフィグ読み込み後にUI上で変更された項目については、
            // 各ペインのOnAcceptでTempConfigと本体Configの両方に値を反映することでフォローしている。
            if (OptionUtil.IsNeedReloadProjectConfig)
            {
                OptionStore.ProjectConfig.Set(OptionUtil.TempConfig);

                // 描画パスの設定に関するエラーをログに表示
                foreach (string error in OptionUtil.TempConfig.DrawPathSettingErrors)
                {
                    Logger.Log("LogView", LogLevels.Warning, error);
                }
            }

            // このペインの内容に関してはProjectConfigへの適用のみで大丈夫
            this.OutputConfig(OptionStore.ProjectConfig);

            return true;
        }

        /// <summary>
        /// Called when the Cancel button is clicked.
        /// </summary>
        public void OnCancel()
        {
        }

        /// <summary>
        /// プロジェクトコンフィグが読み込まれた際に、その内容にUIを更新します。
        /// </summary>
        public void OnProjectSettingChanged()
        {
            this.chkOptionsLinearEditMode.Checked = OptionUtil.TempConfig.LinearMode;
            this.TextureOriginMode = OptionUtil.TempConfig.TextureOriginMode;
            this.DepthMode = OptionUtil.TempConfig.DepthMode;
            this.InitializeDrawPath(OptionUtil.TempConfig.DrawPaths);
            this.chkEnableRegularGravity.Checked = OptionUtil.TempConfig.EnableRegularGravity;
            this.numLength.Value = (decimal)OptionUtil.TempConfig.UnitLength;
            this.UpdateUnitLengthPrecision();
            this.numFps.Value = OptionUtil.TempConfig.FpsForRegularGravity;
            this.LoadCombinerDefInputControl(OptionUtil.TempConfig.EffectCombinerBlockDefinitionPaths);
            this.txtDocumentFolder.FilePath = OptionUtil.TempConfig.DocumentPath;
            this.txtPresetFolder.FilePath = OptionUtil.TempConfig.FileEventProject.PresetFolderPath;
        }

        /// <summary>
        /// プロジェクトコンフィグのパス設定が有効か無効かを返す。
        /// </summary>
        /// <returns>
        /// 有効ならtrue、無効ならfalse.
        /// </returns>
        public bool OnValidationCheck()
        {
            return this.txtProjectSettings.BackColor == Color.White;
        }

        /// <summary>
        /// プロジェクトコンフィグをファイル出力する際に各UIの内容を出力用インスタンスに収集します。
        /// </summary>
        public void OnExportProjectSetting()
        {
            this.OutputConfig(OptionUtil.TempConfig);
        }

        /// <summary>
        /// 小数点以下の桁数を取得します。
        /// </summary>
        /// <param name="v">対象の値です。</param>
        /// <returns>桁数です。</returns>
        private int CalcPrecision(decimal v)
        {
            string text = v.ToString().TrimEnd('0');

            int index = text.IndexOf('.');
            if (index == -1)
            {
                return 0;
            }

            return text.Substring(index + 1).Length;
        }

        /// <summary>
        /// 単位長の小数点以下桁数をアップデートします。
        /// </summary>
        private void UpdateUnitLengthPrecision()
        {
            this.numLength.DecimalPlaces = this.CalcPrecision(this.numLength.Value);
        }

        /// <summary>
        /// UIの状態をコンフィグインスタンスに設定します
        /// </summary>
        /// <param name="config">設定先のコンフィグインスタンス</param>
        private void OutputConfig(EffectMakerProjectConfig config)
        {
            config.LinearMode = this.chkOptionsLinearEditMode.Checked;

            config.TextureOriginMode = this.TextureOriginMode;
            config.DepthMode = this.DepthMode;

            config.EnableRegularGravity = this.chkEnableRegularGravity.Checked;
            config.UnitLength = (float)this.numLength.Value;
            config.FpsForRegularGravity = (int)this.numFps.Value;

            config.DocumentPath = this.txtDocumentFolder.FilePath;
            config.FileEventProject.PresetFolderPath = this.txtPresetFolder.FilePath;

            // コンバイナエディタ周りの設定
            if (config.EffectCombinerBlockDefinitionPaths == null)
            {
                config.EffectCombinerBlockDefinitionPaths = new List<string>();
            }

            var combinerDefPaths =
                from item in this.combinerDefInputControls
                where string.IsNullOrEmpty(item.CombinerDefFolderPath) == false
                select item.CombinerDefFolderPath;

            config.EffectCombinerBlockDefinitionPaths.Clear();
            config.EffectCombinerBlockDefinitionPaths.AddRange(combinerDefPaths);
        }

        /// <summary>
        /// プロジェクト設定ファイルの内容が不正だった場合のメッセージボックスを表示します.
        /// </summary>
        private void ShowWarningMessageBox()
        {
            ThreadSafeMsgBox.Show(
                Resources.InvalidSettingFileProjectConfig,
                Resources.WarningCaption,
                MessageBoxButtons.OK,
                MessageBoxIcon.Warning);
        }

        /// <summary>
        /// エフェクト定義フォルダのUIを生成する.
        /// </summary>
        /// <param name="definitionPaths">エフェクト定義フォルダのパス</param>
        private void LoadCombinerDefInputControl(List<string> definitionPaths)
        {
            // 既存のUIは全て削除する
            this.grpCombinerDefinitionFolders.Controls.Clear();
            this.combinerDefInputControls.Clear();

            // コンバイナ定義フォルダパスの入力UIを作成
            if (definitionPaths == null || definitionPaths.Count <= 0)
            {
                this.AddCombinerDefInputControl();
            }
            else
            {
                foreach (string path in definitionPaths)
                {
                    this.AddCombinerDefInputControl(path);
                }
            }

            // CombinerEditorの有効/無効に合わせてUIの表示/非表示を切り替える
            if (!OptionUtil.TempConfig.IsEftCombinerEditorEnabled &&
                this.grpCombinerDefinitionFolders.Visible)
            {
                // 無効時にVisibleを調整するだけでなく、ペインの高さも削ってスクロールバーの挙動を自然にする
                this.grpCombinerDefinitionFolders.Visible = false;
                int height = this.grpCombinerDefinitionFolders.Location.Y;
                height -= this.grpCombinerDefinitionFolders.Margin.Top;
                height -= this.grpCombinerDefinitionFolders.Padding.Top;
                this.Height = height;
            }
            else if (OptionUtil.TempConfig.IsEftCombinerEditorEnabled &&
                !this.grpCombinerDefinitionFolders.Visible)
            {
                this.grpCombinerDefinitionFolders.Visible = true;
            }
        }

        /// <summary>
        /// プロジェクトファイルをチェックして読み込む.
        /// </summary>
        private void CheckAndUpdateProjectConfig()
        {
            if (!File.Exists(this.txtProjectSettings.FilePath) || !IOConstants.CheckRootElement(this.txtProjectSettings.FilePath, "EffectMakerProjectConfig"))
            {
                OptionUtil.TempConfig = new EffectMakerProjectConfig();
                this.UpdateView(null);
                this.txtProjectSettings.BackColor = Color.Pink;
                this.ShowWarningMessageBox();
                if (this.DrawTreeView != null)
                {
                    this.DrawTreeView();
                }
            }
            else
            {
                this.UpdateView(this.txtProjectSettings.FilePath);
            }

            OptionUtil.IsNeedReloadProjectConfig = true;
        }

        /// <summary>
        /// txtProjectSettings.FilePathを更新する
        /// </summary>
        /// <param name="fileName">更新するファイル名(パス)</param>
        /// <returns>txtProjectSettings.FilePathを更新した場合はtrueを返す. 更新しなかった場合はfalseを返す.</returns>
        private bool UpdateProjectSettingsTxt(string fileName)
        {
            if (!IOConstants.CheckRootElement(fileName, "EffectMakerProjectConfig"))
            {
                this.ShowWarningMessageBox();
                return false;
            }

            this.txtProjectSettings.FilePath = fileName;

            // セッティングツリーをリロード
            this.UpdateView(this.txtProjectSettings.FilePath);

            // オプションウィンドウを閉じるときにプロジェクト設定をリロードさせる
            OptionUtil.IsNeedReloadProjectConfig = true;

            return true;
        }

        /// <summary>
        /// Called when the UserSettingsLoad Button is clicked.
        /// </summary>
        /// <param name="sender">The UserSettingsLoad Button.</param>
        /// <param name="e">Event argument.</param>
        private void OnProjectSettingsLoadButtonClick(object sender, EventArgs e)
        {
            var dlg = new OpenFileDialog
            {
                AutoUpgradeEnabled = !OptionStore.RootOptions.Interface.UseWindowsXPStyle,
                CheckFileExists = true,
                CheckPathExists = true,
                Filter = string.Join(
                    "|",
                    Properties.Resources.OptionDialogFilterXmlFiles,
                    Properties.Resources.OptionDialogFilterAllFiles),
                Multiselect = false,
                FileName = this.txtProjectSettings.FilePath,
            };

            if (dlg.ShowDialog() != DialogResult.OK)
            {
                return;
            }

            this.UpdateProjectSettingsTxt(dlg.FileName);
        }

        /// <summary>
        /// Called when the UserSettingsReload Button is clicked.
        /// </summary>
        /// <param name="sender">The UserSettingsReload Button.</param>
        /// <param name="e">Event argument.</param>
        private void OnProjectSettingsReloadButtonClick(object sender, EventArgs e)
        {
            if (!string.IsNullOrEmpty(this.txtProjectSettings.FilePath))
            {
                this.CheckAndUpdateProjectConfig();
            }
        }

        /// <summary>
        /// Called when the UserSettingsClear Button is clicked.
        /// </summary>
        /// <param name="sender">The UserSettingsClear Button.</param>
        /// <param name="e">Event argument.</param>
        private void OnProjectSettingsClearButtonClick(object sender, EventArgs e)
        {
            this.txtProjectSettings.FilePath = string.Empty;
            this.txtProjectSettings.BackColor = Color.White;
            OptionUtil.IsNeedReloadProjectConfig = true;
            this.UpdateView(this.txtProjectSettings.FilePath);
        }

        /// <summary>
        /// テキストボックスのパスを検証し、ビューを更新します。
        /// </summary>
        /// <param name="prjConfigFilePath">プロジェクトコンフィグのファイルパス</param>
        private void UpdateView(string prjConfigFilePath)
        {
            EffectMakerProjectConfig prjConfig;

            if (string.IsNullOrEmpty(prjConfigFilePath))
            {
                prjConfig = new EffectMakerProjectConfig();
                this.txtProjectSettings.BackColor = Color.White;
            }
            else
            {
                prjConfig = EffectMakerProjectConfig.Load(prjConfigFilePath);
                if (prjConfig == null)
                {
                    prjConfig = new EffectMakerProjectConfig();
                    this.txtProjectSettings.BackColor = Color.Pink;
                    this.ShowWarningMessageBox();
                }
                else
                {
                    this.txtProjectSettings.BackColor = Color.White;
                }
            }

            OptionUtil.TempConfig = prjConfig;
            if (this.TriggerProjectSetting != null)
            {
                this.TriggerProjectSetting();
            }

            if (this.DrawTreeView != null)
            {
                this.DrawTreeView();
            }
        }

        /// <summary>
        /// Add a new combiner definition input control.
        /// </summary>
        /// <param name="definitionPath">The definition folder paht.</param>
        private void AddCombinerDefInputControl(string definitionPath = null)
        {
            // First hide the add combiner definition button of all the existing input controls.
            foreach (EffectCombinerDefInput ctrl in this.combinerDefInputControls)
            {
                ctrl.IsAddButtonVisible = false;
            }

            const int ControlPosY = 31;

            // Create the new input control.
            var control = new EffectCombinerDefInput()
            {
                Width = this.grpCombinerDefinitionFolders.ClientSize.Width - 10,
                Anchor = AnchorStyles.Left | AnchorStyles.Top | AnchorStyles.Right,
                CombinerDefFolderPath = definitionPath,
            };
            control.Location = new Point(6, ControlPosY + control.Height * this.combinerDefInputControls.Count);

            control.AddButtonClicked += (s, e) =>
                this.AddCombinerDefInputControl();

            control.RemoveButtonClicked += (s, e) =>
                this.RemoveCombinerDefInputControl((EffectCombinerDefInput)s);

            this.grpCombinerDefinitionFolders.Controls.Remove(this.lblEftCombinerFolderComment);
            this.grpCombinerDefinitionFolders.Controls.Add(control);
            this.combinerDefInputControls.Add(control);

            control.AllowDrop = true;

            // Adjust the height of the container group and the panel.
            this.lblEftCombinerFolderComment.Location = new Point(14, control.Bottom + 6);
            this.grpCombinerDefinitionFolders.Controls.Add(this.lblEftCombinerFolderComment);
            this.grpCombinerDefinitionFolders.Height = this.lblEftCombinerFolderComment.Bottom + 12;
            this.Height = this.grpCombinerDefinitionFolders.Bottom;
        }

        /// <summary>
        /// Remove the given combiner definition input control.
        /// </summary>
        /// <param name="control">The control to remove.</param>
        private void RemoveCombinerDefInputControl(EffectCombinerDefInput control)
        {
            // Save the current scroll position.
            Point scrollPos = (this.Parent as Panel).AutoScrollPosition;

            // Remove or clear the input control depending on the existing input control count.
            if (this.combinerDefInputControls.Count == 1)
            {
                control.CombinerDefFolderPath = string.Empty;
            }
            else if (this.combinerDefInputControls.Count > 1)
            {
                this.grpCombinerDefinitionFolders.Controls.Remove(control);
                this.combinerDefInputControls.Remove(control);
            }

            this.grpCombinerDefinitionFolders.Controls.Remove(this.lblEftCombinerFolderComment);

            const int ControlPosY = 31;

            // Adjust the position of the remaining input controls.
            int count = this.combinerDefInputControls.Count;
            for (int i = 0; i < count; ++i)
            {
                EffectCombinerDefInput ctrl = this.combinerDefInputControls[i];
                ctrl.Location = new Point(6, ControlPosY + ctrl.Height * i);
                ctrl.IsAddButtonVisible = false;
            }

            // Only show the add button of the last input control.
            EffectCombinerDefInput lastControl = this.combinerDefInputControls[count - 1];
            lastControl.IsAddButtonVisible = true;

            // Adjust the height of the container group and the panel.
            this.lblEftCombinerFolderComment.Location = new Point(14, lastControl.Bottom + 6);
            this.grpCombinerDefinitionFolders.Controls.Add(this.lblEftCombinerFolderComment);
            this.grpCombinerDefinitionFolders.Height = this.lblEftCombinerFolderComment.Bottom + 12;
            this.Height = this.grpCombinerDefinitionFolders.Bottom;

            // Restore the scroll position.
            (this.Parent as Panel).AutoScrollPosition = new Point(scrollPos.X, -scrollPos.Y);
        }

        /// <summary>
        /// 現状のプロジェクト設定を保存する
        /// </summary>
        private void OnExportProjectConfig(object sender, EventArgs eventArgs)
        {
            var dlg = new SaveFileDialog
            {
                AutoUpgradeEnabled = !OptionStore.RootOptions.Interface.UseWindowsXPStyle,
                Filter = string.Join(
                    "|",
                    Properties.Resources.OptionDialogFilterXmlFiles,
                    Properties.Resources.OptionDialogFilterAllFiles),
            };

            if (dlg.ShowDialog() != DialogResult.OK && !string.IsNullOrEmpty(dlg.FileName))
            {
                return;
            }

            if (this.ApplyProjectSettingForExport == null)
            {
                return;
            }

            // 現在のUIの状態に更新
            this.ApplyProjectSettingForExport();

            // 一時的にTempConfigにすげ替えて出力したら戻す
            var temp = OptionStore.ProjectConfig;
            OptionStore.ProjectConfig = OptionUtil.TempConfig;
            OptionStore.SaveProjectConfig(dlg.FileName);
            OptionStore.ProjectConfig = temp;
        }

        /// <summary>
        /// 単位長の値が変更されたとき。
        /// </summary>
        /// <param name="sender">呼び出し元</param>
        /// <param name="e">イベント引数</param>
        private void numLength_ValueChanged(object sender, EventArgs e)
        {
            this.UpdateUnitLengthPrecision();
        }

        /// <summary>
        /// ドキュメントパスの選択
        /// </summary>
        /// <param name="sender">The Preset folder Button.</param>
        /// <param name="e">Event argument.</param>
        private void OnDocumentLoadButtonClick(object sender, EventArgs e)
        {
            var dlg = new OpenFileDialog
            {
                AutoUpgradeEnabled = !OptionStore.RootOptions.Interface.UseWindowsXPStyle,
                CheckFileExists = true,
                CheckPathExists = true,
                Filter = string.Join(
                    "|",
                    Properties.Resources.OptionDialogFilterDocumentPath,
                    Properties.Resources.OptionDialogFilterAllFiles),
                Multiselect = false,
            };

            if (dlg.ShowDialog(this) != DialogResult.OK)
            {
                return;
            }

            this.txtDocumentFolder.FilePath = dlg.FileName;
        }

        /// <summary>
        /// ドキュメントパスの初期化
        /// </summary>
        /// <param name="sender">The Preset folder Button.</param>
        /// <param name="e">Event argument.</param>
        private void OnDocumentClearButtonClick(object sender, EventArgs e)
        {
            this.txtDocumentFolder.FilePath = string.Empty;
        }

        /// <summary>
        /// プリセットフォルダパスの選択
        /// </summary>
        /// <param name="sender">The Preset folder Button.</param>
        /// <param name="e">Event argument.</param>
        private void OnPresetLoadButtonClick(object sender, EventArgs e)
        {
            string originalPath = string.Empty;
            if (string.IsNullOrEmpty(this.txtPresetFolder.FilePath) == false)
            {
                originalPath =
                    System.IO.Path.GetDirectoryName(this.txtPresetFolder.FilePath + Path.DirectorySeparatorChar);
            }

            var dlg = new FolderBrowserDialog
            {
                ShowNewFolderButton = true,
                SelectedPath = originalPath,
            };

            if (dlg.ShowDialog() != DialogResult.OK)
            {
                return;
            }

            this.txtPresetFolder.FilePath = dlg.SelectedPath;
        }

        /// <summary>
        /// プリセットフォルダパスの初期化
        /// </summary>
        /// <param name="sender">The Preset folder Button.</param>
        /// <param name="e">Event argument.</param>
        private void OnPresetClearButtonClick(object sender, EventArgs e)
        {
            this.txtPresetFolder.FilePath = string.Empty;
        }
    }
}
