﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Drawing;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows.Forms;

using EffectMaker.BusinessLogic.IO;

using EffectMaker.Foundation.Extensions;
using EffectMaker.Foundation.Log;
using EffectMaker.Foundation.Utility;

namespace EffectMaker.Application.OptionPanes
{
    /// <summary>
    /// A tree view class for displaying file hierarchy, designed
    /// mainly for custom action and custom shader option panels.
    /// </summary>
    internal class FileHierarchyTreeView : TreeView
    {
        /// <summary>
        /// Constructor.
        /// </summary>
        public FileHierarchyTreeView() : base()
        {
            this.DrawMode = TreeViewDrawMode.OwnerDrawText;
        }

        /// <summary>
        /// Clear all the nodes.
        /// </summary>
        public void Clear()
        {
            this.Nodes.Clear();
        }

        /// <summary>
        /// Add a new text node to the specified parent node.
        /// </summary>
        /// <param name="parent">The parent node.</param>
        /// <param name="text">The displayed text.</param>
        /// <returns>The created tree node.</returns>
        public TreeNode AddText(TreeNode parent, string text)
        {
            if (string.IsNullOrEmpty(text) == true)
            {
                return null;
            }

            // Create the node.
            TreeNode node = new TreeNode(text);
            node.Tag = null;

            // Expand the node.
            node.Expand();

            // Add the created node to the tree view or the specified parent.
            if (parent == null)
            {
                this.Nodes.Add(node);
            }
            else
            {
                parent.Nodes.Add(node);
            }

            return node;
        }

        /// <summary>
        /// Add a new file to the specified parent node.
        /// </summary>
        /// <param name="parent">The parent node.</param>
        /// <param name="name">The displayed name (label) for the file.</param>
        /// <param name="baseFolder">The base folder to use if the file path is a relative path.</param>
        /// <param name="filePath">The file path.</param>
        /// <returns>The created tree node.</returns>
        public TreeNode AddFile(
            TreeNode parent,
            string name,
            string baseFolder,
            string filePath)
        {
            string rootedPath = PathUtility.GetRootedPath(baseFolder, filePath);
            return this.AddFile(parent, name, rootedPath);
        }

        /// <summary>
        /// Add a new file to the specified parent node.
        /// </summary>
        /// <param name="parent">The parent node.</param>
        /// <param name="name">The displayed name (label) for the file.</param>
        /// <param name="filePath">The file path.</param>
        /// <returns>The created tree node.</returns>
        public TreeNode AddFile(TreeNode parent, string name, string filePath)
        {
            // Choose the node collection to add.
            TreeNodeCollection nodes = null;
            if (parent == null)
            {
                nodes = this.Nodes;
            }
            else
            {
                nodes = parent.Nodes;
            }

            TreeNode node;
            if (string.IsNullOrEmpty(name) == false)
            {
                // This node is only for the name of the file.
                var nameNode = new TreeNode(name);
                nameNode.Tag = null;

                // Create a child node for the file path.
                node = new TreeNode(filePath);
                if (string.IsNullOrEmpty(filePath) == false)
                {
                    node.Tag = new FilePathNodeData(filePath);
                }

                // Add the created nodes to the parent.
                nameNode.Nodes.Add(node);
                nodes.Add(nameNode);
            }
            else
            {
                // The node itself shows the file path.
                node = new TreeNode(filePath);
                if (string.IsNullOrEmpty(filePath) == false)
                {
                    node.Tag = new FilePathNodeData(filePath);
                }

                // Add the created nodes to the parent.
                nodes.Add(node);
            }

            // Expand the node.
            this.ExpandAll();

            return node;
        }

        /// <summary>
        /// Handle NodeMouseDoubleClick event.
        /// </summary>
        /// <param name="e">Event argument.</param>
        protected override void OnNodeMouseDoubleClick(TreeNodeMouseClickEventArgs e)
        {
            base.OnNodeMouseDoubleClick(e);

            if (e.Node == null)
            {
                return;
            }

            var data = e.Node.Tag as FilePathNodeData;
            if (data != null &&
                string.IsNullOrEmpty(data.FullPath) == false &&
                File.Exists(data.FullPath) == true)
            {
                ApplicationIOManager.OpenTextFileWithExternalTextEditor(data.FullPath);
            }
        }

        /// <summary>
        /// Called before a node selection changes.
        /// </summary>
        /// <param name="e">Event argument.</param>
        protected override void OnBeforeSelect(TreeViewCancelEventArgs e)
        {
            e.Cancel = true;
            base.OnBeforeSelect(e);
        }

        /// <summary>
        /// Handle DrawNode event for the tree view for custom rendering.
        /// </summary>
        /// <param name="e">The event arguments.</param>
        protected override void OnDrawNode(DrawTreeNodeEventArgs e)
        {
            base.OnDrawNode(e);

            var data = e.Node.Tag as FilePathNodeData;
            if (data == null)
            {
                // This is a name node, just render with the default method.
                e.DrawDefault = true;
                return;
            }

            e.DrawDefault = false;

            // Use the font for the node or for the tree view?
            var nameFont = e.Node.NodeFont;
            if (nameFont == null)
            {
                nameFont = this.Font;
            }

            // Decide the brush color.
            Brush nameBrush = data.FileExists == true ? Brushes.Green : Brushes.Red;
            Brush dirBrush = Brushes.Gray;
            if ((e.State & TreeNodeStates.Selected) != 0)
            {
                nameBrush = SystemBrushes.HighlightText;
                dirBrush = SystemBrushes.HighlightText;
            }
            else
            {
                nameBrush = data.FileExists == true ? Brushes.Green : Brushes.Red;
                dirBrush = Brushes.Gray;
            }

            // Measure text width for the file name.
            int fileNameWidth =
                (int)Math.Ceiling(e.Graphics.MeasureString(data.FileName, nameFont).Width);

            // Compute the render rectangles.
            Rectangle fileNameRect = new Rectangle(
                e.Bounds.Location,
                new Size(fileNameWidth, e.Bounds.Height));

            // Create the string format for rendering.
            var fileNameFormat = new StringFormat()
            {
                Alignment = StringAlignment.Near,
                LineAlignment = StringAlignment.Center,
            };

            // Render!
            e.Graphics.DrawString(
                data.FileName,
                nameFont,
                nameBrush,
                fileNameRect,
                fileNameFormat);

            if (string.IsNullOrEmpty(data.DirectoryName) == false)
            {
                // The font for the directory name is smaller.
                var dirFont = new Font(nameFont.FontFamily, nameFont.Size - 1.0f);

                // Compute the render rectangles.
                Rectangle dirNameRect = new Rectangle(
                    fileNameRect.Right,
                    fileNameRect.Y,
                    e.Bounds.Right - fileNameRect.Right,
                    e.Bounds.Height);

                // Create the string format for rendering.
                var dirNameFormat = new StringFormat()
                {
                    Alignment = StringAlignment.Near,
                    LineAlignment = StringAlignment.Near,
                };

                e.Graphics.DrawString(
                    "(" + data.DirectoryName + ")",
                    dirFont,
                    dirBrush,
                    dirNameRect,
                    dirNameFormat);
            }
        }

        /// <summary>
        /// Holds data needed for the file path tree nodes.
        /// </summary>
        private class FilePathNodeData
        {
            /// <summary>The full path of the file.</summary>
            private string fullPath;

            /// <summary>
            /// Constructor.
            /// </summary>
            /// <param name="fullPath">The full path of the file.</param>
            public FilePathNodeData(string fullPath)
            {
                this.FullPath = fullPath;
            }

            /// <summary>
            /// Get or set the flag indicating whether the file exists.
            /// </summary>
            public bool FileExists { get; private set; }

            /// <summary>
            /// Get or set the full path.
            /// </summary>
            public string FullPath
            {
                get
                {
                    return this.fullPath;
                }

                set
                {
                    this.fullPath = value;
                    this.DirectoryName = Path.GetDirectoryName(this.fullPath);
                    this.FileName = Path.GetFileName(this.fullPath);
                    this.CheckFileExistence();
                }
            }

            /// <summary>
            /// Get or set the directory name of the file.
            /// </summary>
            public string DirectoryName { get; private set; }

            /// <summary>
            /// Get or set the name of the file.
            /// </summary>
            public string FileName { get; private set; }

            /// <summary>
            /// Check if the file exists and update the FileExists flag.
            /// </summary>
            /// <returns>True if the file exists.</returns>
            public bool CheckFileExistence()
            {
                try
                {
                    this.FileExists = File.Exists(this.fullPath);
                }
                catch (Exception ex)
                {
                    Logger.Log(
                        LogLevels.Warning,
                        "An exception occurred while checking the file existence for " + this.fullPath);

                    Logger.Log(
                        LogLevels.Warning,
                        "Exception : " + ex.ToString());

                    this.FileExists = false;
                }

                return this.FileExists;
            }
        }
    }
}
