﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows.Forms;
using EffectMaker.Application.Properties;
using EffectMaker.BusinessLogic.IO;
using EffectMaker.BusinessLogic.Options;
using EffectMaker.Foundation.Utility;

namespace EffectMaker.Application
{
    /// <summary>
    /// 今日のヒント
    /// </summary>
    public partial class TipsDialog : Form
    {
        /// <summary>
        /// 画像ごとのテキスト
        /// </summary>
        private Dictionary<int, Tuple<string, string>> textDictionary = null;

        /// <summary>
        /// 現在表示している画像のインデックス
        /// </summary>
        private int imageIndex = 0;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="isStartUp">スタートアップ時の表示ならtrue</param>
        public TipsDialog(bool isStartUp)
        {
            this.InitializeComponent();

            this.Icon = Icon.FromHandle(Resources.Icon_Hint.GetHicon());

            this.Text = Resources.HelpDialog_Caption;
            this.btnClose.Text = Resources.TipsDialogOK;
            this.btnClose.Click += (sender, args) => this.Close();

            this.chkUseTips.Text = Resources.TipsDialogCheckBoxShowNext;
            this.chkUseTips.Checked = OptionStore.RootOptions.Interface.UseTipsDialog;

            this.IsStartUp = isStartUp;
            this.InitTextDictionary();

            this.btnPrev.ToolTipText = Resources.TipsDialogPrevHint;
            this.btnNext.ToolTipText = Resources.TipsDialogNextHint;
            this.btnPrev.Click += (sender, args) =>
            {
                --this.imageIndex;
                this.UpdateImage();
            };

            this.btnNext.Click += (sender, args) =>
            {
                ++this.imageIndex;
                this.UpdateImage();
            };

            Showing = true;
        }

        /// <summary>
        /// 表示中に立つフラグ
        /// </summary>
        public static bool Showing { get; private set; }

        /// <summary>
        /// スタートアップ時の表示か否かを取得・設定
        /// </summary>
        private bool IsStartUp { get; set; }

        /// <summary>
        /// 表示位置調整
        /// </summary>
        /// <param name="mainRect">メインフォームの矩形</param>
        public void AdjustPosition(Rectangle mainRect)
        {
            if (mainRect == Rectangle.Empty)
            {
                this.StartPosition = FormStartPosition.CenterScreen;
            }
            else if (mainRect.X < 0 || mainRect.Y < 0)
            {
                this.StartPosition = FormStartPosition.WindowsDefaultLocation;
            }
            else
            {
                this.StartPosition = FormStartPosition.Manual;
                int x = (mainRect.Right + mainRect.Left) / 2;
                int y = (mainRect.Top + mainRect.Bottom) / 2;
                x -= this.Width / 2;
                y -= this.Height / 2;
                this.Location = new Point(x, y);
            }
        }

        /// <summary>
        /// フォームを表示します。
        /// </summary>
        /// <param name="owner">オーナーウィンドウ</param>
        public new void Show(IWin32Window owner)
        {
            // StartPosition の設定が反映されないため位置を手動で設定する
            Rectangle mainRect = OptionStore.RootOptions.Interface.MainFormBounds;
            this.AdjustPosition(mainRect);

            base.Show(owner);
        }

        /// <summary>
        /// Shownイベント
        /// </summary>
        /// <param name="e">イベント引数</param>
        protected override void OnShown(EventArgs e)
        {
            if (this.textDictionary.Count == 0)
            {
                this.Close();
                return;
            }

            base.OnShown(e);
        }

        /// <summary>
        /// Closingイベント
        /// </summary>
        /// <param name="e">イベント引数</param>
        protected override void OnClosing(CancelEventArgs e)
        {
            Showing = false;
            OptionStore.RootOptions.Interface.UseTipsDialog = this.chkUseTips.Checked;
            base.OnClosing(e);
        }

        /// <summary>
        /// 画像とメッセージの辞書を初期化
        /// </summary>
        private void InitTextDictionary()
        {
            string imgFolder = Path.Combine(IOConstants.ExecutableFolderPath, @"Tips\");
            this.textDictionary = new Dictionary<int, Tuple<string, string>>();
            using (var sr = new StreamReader(Path.Combine(imgFolder, "ImageDoc.csv"), Encoding.UTF8))
            {
                while (true)
                {
                    var line = sr.ReadLine();
                    if (line == null)
                    {
                        break;
                    }

                    var splitted = line.Split('\t');
                    string str;
                    if (ResourceUtility.GetStringValue(this.GetType().Assembly, splitted[1], out str) == false)
                    {
                        str = string.Format("<{0}>", splitted[1]);
                    }

                    string imgPath = Path.Combine(imgFolder, splitted[1] + ".png");

                    this.textDictionary[int.Parse(splitted[0])] = new Tuple<string, string>(imgPath, str);
                }
            }

            if (this.IsStartUp)
            {
                this.imageIndex = OptionStore.RootOptions.Interface.HintIndex + 1;
            }
            else
            {
                this.imageIndex = 1;
            }

            this.UpdateImage();
        }

        /// <summary>
        /// 表示画像の更新
        /// </summary>
        private void UpdateImage()
        {
            if (this.imageIndex < 1)
            {
                this.imageIndex = this.textDictionary.Count;
            }

            if (this.imageIndex > this.textDictionary.Count)
            {
                this.imageIndex = 1;
            }

            var targetItem = this.textDictionary.ToList()[this.imageIndex - 1];
            this.picTipsImage.ImageLocation = targetItem.Value.Item1;

            var num = targetItem.Key;
            var maxNum = this.textDictionary.Max(x => x.Key);
            this.lblNumber.Text = string.Format("{0}/{1}", num, maxNum);
            if (this.IsStartUp)
            {
                OptionStore.RootOptions.Interface.HintIndex = this.imageIndex;
            }

            this.Text = string.Format(
                "{0}{1}", Resources.HelpDialog_Caption, targetItem.Value.Item2);
        }
    }
}
