﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Drawing;
using System.IO;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using EffectMaker.BusinessLogic.IO;
using EffectMaker.BusinessLogic.Options;
using EffectMaker.BusinessLogic.UserData;
using EffectMaker.Foundation.Log;
using EffectMaker.UIDialogs.CommandHistoryDialog;
using EffectMaker.UIDialogs.CurveEditorDialog;
using EffectMaker.UIDialogs.EffectBrowserDialog;
using EffectMaker.UIDialogs.SearchDialog;
using EffectMaker.UIDialogs.TextureViewerDialog;
using EffectMaker.UIDialogs.EmbededViewerDialog;
using EffectMaker.UILogic.Manager;
using EffectMaker.UILogic.ViewModels;
using WeifenLuo.WinFormsUI.Docking;

namespace EffectMaker.Application
{
    /// <summary>
    /// メインフォームのドックパネルに追加するビューです。
    /// </summary>
    public class DockContents
    {
        /// <summary>
        /// コンフィグファイルパスです。
        /// </summary>
        public const string ConfigFilePath = "DockInfo.xml";

        /// <summary>
        /// メインフォームです。
        /// </summary>
        private MainForm mainForm;

        /// <summary>
        /// メインフォームのドックパネルです。
        /// </summary>
        private DockPanel dockPanel;

        /// <summary>
        /// ルートビューモデルです。
        /// </summary>
        private WorkspaceRootViewModel rootViewModel;

        /// <summary>
        /// バインドアイテムです。
        /// </summary>
        private MainFormExecutableHolder executableHolder;

        /// <summary>
        /// 履歴ダイアログです。
        /// </summary>
        private CommandHistoryDialog commandHistoryDialog;

        /// <summary>
        /// プロパティダイアログです。
        /// </summary>
        private PropertyDialog propertyDialog;

        /// <summary>
        /// テクスチャダイアログです。
        /// </summary>
        private TextureViewerDialog textureViewerDialog;

        /// <summary>
        /// エフェクトブラウザダイアログです。
        /// </summary>
        private EffectBrowserDialog effectBrowserDialog;

        /// <summary>
        /// カーブエディタダイアログです。
        /// </summary>
        private CurveEditorDialog curveEditorDialog;

        /// <summary>
        /// 検索置換ダイアログです。
        /// </summary>
        private SearchDialog searchDialog;

        /// <summary>
        /// ビューア埋め込みダイアログです。
        /// </summary>
        private EmbededViewerDialog embededViewerDialog;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="mainForm">メインフォーム</param>
        /// <param name="dockPanel">メインフォームのドックパネル</param>
        /// <param name="rootViewModel">ルートビューモデル</param>
        /// <param name="executableHolder">バインドアイテム</param>
        public DockContents(
            MainForm mainForm,
            DockPanel dockPanel,
            WorkspaceRootViewModel rootViewModel,
            MainFormExecutableHolder executableHolder)
        {
            this.mainForm = mainForm;
            this.dockPanel = dockPanel;
            this.rootViewModel = rootViewModel;
            this.executableHolder = executableHolder;

            // ドックパネルにウィンドウをアタッチしたときウィンドウを隠さないようにする
            this.dockPanel.ContentAdded += this.OnContentAdded;
        }

        /// <summary>
        /// ドッキングウィンドウの表示状態を保存。
        /// </summary>
        public void SaveDockInfo()
        {
            // ファイルパスを取得
            string filePath = Path.Combine(IOConstants.ExecutableFolderPath, ConfigFilePath);

            // 表示状態を保存
            try
            {
                this.dockPanel.SaveAsXml(filePath);
            }
            catch
            {
                Logger.Log("LogView", LogLevels.Error, Properties.Resources.ErrorFailedSavingDockInfo);
            }
        }

        /// <summary>
        /// ドッキングウィンドウの表示状態を復元。
        /// </summary>
        /// <returns>成功したときtrueを返します。</returns>
        public bool LoadDockInfo()
        {
            // ファイルパスを取得
            string filePath = Path.Combine(IOConstants.ExecutableFolderPath, ConfigFilePath);

            // 表示状態を復元
            try
            {
                if (File.Exists(filePath) == true)
                {
                    this.dockPanel.LoadFromXml(filePath, this.GetContentFromPersistString);
                }
            }
            catch
            {
            }

            return true;
        }

        /// <summary>
        /// クラス名に対応するクラスインスタンスを取得します。
        /// ドッキングウィンドウの状態を復元するためのメソッドです。
        /// </summary>
        /// <param name="persistString">クラス名</param>
        /// <returns>クラスインスタンスを返します。</returns>
        public IDockContent GetContentFromPersistString(string persistString)
        {
            if (persistString == typeof(PropertyDialog).FullName)
            {
                return this.GetPropertyDialog(true);
            }
            else if (persistString == typeof(CommandHistoryDialog).FullName)
            {
                return this.GetCommandHistoryDialog(true);
            }
            else if (persistString == typeof(CurveEditorDialog).FullName)
            {
                return this.GetCurveEditorDialog(true);
            }
            else if (persistString == typeof(TextureViewerDialog).FullName)
            {
                return this.GetTextureViewerDialog(true);
            }
            else if (persistString == typeof(EffectBrowserDialog).FullName)
            {
                return this.GetEffectBrowserDialog(true);
            }
            else if (persistString == typeof(EmbededViewerDialog).FullName)
            {
                return this.GetEmbededViewerDialog(true);
            }

            return null;
        }

        /// <summary>
        /// プロパティダイアログを取得します。
        /// </summary>
        /// <param name="autoCreate">ダイアログの自動作成フラグ</param>
        /// <returns>プロパティダイアログを返します。</returns>
        public PropertyDialog GetPropertyDialog(bool autoCreate)
        {
            if (this.propertyDialog == null && autoCreate == true)
            {
                this.propertyDialog = new PropertyDialog(this.rootViewModel, this.executableHolder);
                // this.propertyDialog.Owner = this.mainForm;
                this.propertyDialog.Activated += this.OnDockContentsActivated;
            }

            return this.propertyDialog;
        }

        /// <summary>
        /// 履歴ダイアログを取得します。
        /// </summary>
        /// <param name="autoCreate">ダイアログの自動作成フラグ</param>
        /// <returns>履歴ダイアログを返します。</returns>
        public CommandHistoryDialog GetCommandHistoryDialog(bool autoCreate)
        {
            // 履歴ダイアログがnullのとき自動で作る
            if (this.commandHistoryDialog == null && autoCreate == true)
            {
                this.commandHistoryDialog = new CommandHistoryDialog();
                this.commandHistoryDialog.Activated += this.OnDockContentsActivated;
            }

            return this.commandHistoryDialog;
        }

        /// <summary>
        /// カーブエディタダイアログを取得します。
        /// </summary>
        /// <param name="autoCreate">ダイアログの自動作成フラグ</param>
        /// <returns>カーブエディタダイアログを返します。</returns>
        public CurveEditorDialog GetCurveEditorDialog(bool autoCreate)
        {
            // カーブエディタダイアログがnullのとき自動で作る
            if (this.curveEditorDialog == null && autoCreate == true)
            {
                this.curveEditorDialog = new CurveEditorDialog();
                this.curveEditorDialog.Activated += this.OnDockContentsActivated;
            }

            return this.curveEditorDialog;
        }

        /// <summary>
        /// テクスチャダイアログを取得します。
        /// </summary>
        /// <param name="autoCreate">ダイアログの自動作成フラグ</param>
        /// <returns>テクスチャダイアログを返します。</returns>
        public TextureViewerDialog GetTextureViewerDialog(bool autoCreate)
        {
            // テクスチャダイアログがnullのとき自動で作る
            if (this.textureViewerDialog == null && autoCreate == true)
            {
                this.textureViewerDialog = new TextureViewerDialog();
                this.textureViewerDialog.Activated += this.OnDockContentsActivated;

                this.textureViewerDialog.VisibleChanged +=
                    (ss, ee) => TextureViewerDialogManager.IsVisibleDialog = this.textureViewerDialog.IsHidden == false;

                // 表示するテクスチャが変更があった時
                Action textureChanged = () =>
                {
                    this.textureViewerDialog.ColorChannel = TextureViewerDialogManager.TargetColorChannel;
                    this.textureViewerDialog.TargetFilePath = TextureViewerDialogManager.TargetTexturePath;

                    this.ShowTextureViewerDialog();
                };

                TextureViewerDialogManager.TargetTextureChanged += (s, ee) => textureChanged();
                TextureViewerDialogManager.TargetColorChannelChanged += (ss, eee) => textureChanged();
            }

            return this.textureViewerDialog;
        }

        /// <summary>
        /// エフェクトブラウザダイアログを取得します。
        /// </summary>
        /// <param name="autoCreate">ダイアログの自動作成フラグ</param>
        /// <returns>エフェクトブラウザダイアログを返します。</returns>
        public EffectBrowserDialog GetEffectBrowserDialog(bool autoCreate)
        {
            // エフェクトブラウザダイアログがnullのとき自動で作る
            if (this.effectBrowserDialog == null && autoCreate == true)
            {
                this.effectBrowserDialog = new EffectBrowserDialog();
                this.effectBrowserDialog.Activated += this.OnDockContentsActivated;

                TextureViewerDialog textureViewerDialog = this.GetTextureViewerDialog(true);

                this.effectBrowserDialog.TextureSelected += (s, ee) =>
                {
                    textureViewerDialog.ColorChannel = ee.Channel;
                    textureViewerDialog.TargetFilePath = ee.FilePath;

                    this.ShowTextureViewerDialog();
                };

                this.effectBrowserDialog.EsetEditing += (s, ee) =>
                    this.executableHolder.OnFileOpenExecutable.Execute(ee.FilePaths.ToArray());

                this.effectBrowserDialog.PrevEditing += (s, ee) =>
                    this.executableHolder.OnFileOpenExecutable.Execute(ee.FilePaths.ToArray());

                this.effectBrowserDialog.WorkspaceOpening += (s, ee) =>
                    this.executableHolder.OnFileOpenExecutable.Execute(ee.FilePaths.ToArray());

                // this.effectBrowserDialog.EsetExporting += (s, ee) => MessageBox.Show("未実装");
            }

            return this.effectBrowserDialog;
        }

        /// <summary>
        /// 検索置換ダイアログを取得します。
        /// </summary>
        /// <param name="autoCreate">ダイアログの自動作成フラグ</param>
        /// <returns>検索置換ダイアログを返します。</returns>
        public SearchDialog GetSearchDialog(bool autoCreate)
        {
            // 検索置換ダイアログがnullのとき自動で作る
            if (this.searchDialog == null && autoCreate == true)
            {
                this.searchDialog = new SearchDialog();
                // this.searchDialog.Activated += this.OnDockContentsActivated;
                this.searchDialog.FormClosed += (s, e) => this.searchDialog = null;

                this.ReloadXPathTable();

                OptionStore.OptionChanged += (s, e) => this.ReloadXPathTable();
            }

            return this.searchDialog;
        }

        /// <summary>
        /// ビューア埋め込みダイアログを取得します。
        /// </summary>
        /// <param name="autoCreate">ダイアログの自動作成フラグ</param>
        /// <returns>ビューア埋め込みダイアログを返します。</returns>
        public EmbededViewerDialog GetEmbededViewerDialog(bool autoCreate)
        {
            // ビューア埋め込みダイアログがnullのとき自動で作る
            if (this.embededViewerDialog == null && autoCreate == true)
            {
                this.embededViewerDialog = new EmbededViewerDialog();
                this.embededViewerDialog.Activated += this.OnDockContentsActivated;
                this.embededViewerDialog.FormClosed += (s, e) => this.embededViewerDialog = null;

                // データコンテキストを設定
                this.embededViewerDialog.DataContext = WorkspaceRootViewModel.Instance.ViewerViewModel?.ViewerBasicViewModel.ViewerBasicSceneViewModel;

                // ワークスペースビューモデルが変わったときにデータコンテキストを更新するようにイベントハンドラを登録
                WorkspaceRootViewModel.Instance.ViewerViewModelChanged += (s, e) =>
                {
                    this.embededViewerDialog.DataContext = WorkspaceRootViewModel.Instance.ViewerViewModel?.ViewerBasicViewModel.ViewerBasicSceneViewModel;
                };
            }

            return this.embededViewerDialog;
        }

        /// <summary>
        /// ビューア埋め込みダイアログを表示します。
        /// </summary>
        /// <returns>表示したビューア埋め込みダイアログを返します。</returns>
        public EmbededViewerDialog ShowEmbededViewerDialog()
        {
            EmbededViewerDialog dialog = this.GetEmbededViewerDialog(true);

            if (dialog.Visible == false)
            {
                if (dialog.DockPanel == null)
                {
                    dialog.Show(this.dockPanel, DockState.Document);
                }
                else
                {
                    dialog.Show();
                }
            }

            return dialog;
        }

        /// <summary>
        /// プロパティダイアログを表示します。
        /// </summary>
        /// <returns>表示したプロパティダイアログを返します。</returns>
        public PropertyDialog ShowPropertyDialog()
        {
            PropertyDialog dialog = this.GetPropertyDialog(true);

            if (dialog.Visible == false)
            {
                if (dialog.DockPanel == null)
                {
                    dialog.Show(this.dockPanel, DockState.Document);
                }
                else
                {
                    dialog.Show();
                }
            }

            return dialog;
        }

        /// <summary>
        /// 履歴ダイアログを表示します。
        /// </summary>
        /// <returns>表示した履歴ダイアログを返します。</returns>
        public CommandHistoryDialog ShowCommandHistoryDialog()
        {
            CommandHistoryDialog dialog = this.GetCommandHistoryDialog(true);

            if (dialog.Visible == false)
            {
                if (dialog.DockPanel == null)
                {
                    dialog.Show(this.dockPanel, DockState.Float);
                }
                else
                {
                    dialog.Show();
                }

                // 非表示の時は内容がアップデートされないので、ここで更新する。
                dialog.UpdateCommandHistory();
            }

            return dialog;
        }

        /// <summary>
        /// カーブエディタダイアログを表示します。
        /// </summary>
        /// <returns>表示したカーブエディタダイアログを返します。</returns>
        public CurveEditorDialog ShowCurveEditorDialog()
        {
            CurveEditorDialog dialog = this.GetCurveEditorDialog(true);

            if (dialog.Visible == false)
            {
                if (dialog.DockPanel == null)
                {
                    dialog.Show(this.dockPanel, DockState.Float);
                    dialog.FloatPane.FloatWindow.ClientSize = new Size(964, 480);
                }
                else
                {
                    dialog.Show();
                }
            }

            return dialog;
        }

        /// <summary>
        /// テクスチャダイアログを表示します。
        /// </summary>
        /// <returns>表示したテクスチャダイアログを返します。</returns>
        public TextureViewerDialog ShowTextureViewerDialog()
        {
            TextureViewerDialog dialog = this.GetTextureViewerDialog(true);

            if (dialog.Visible == false)
            {
                if (dialog.DockPanel == null)
                {
                    dialog.Show(this.dockPanel, DockState.DockTop);
                }
                else
                {
                    dialog.Show();
                }
            }

            return dialog;
        }

        /// <summary>
        /// エフェクトブラウザダイアログを表示します。
        /// </summary>
        /// <returns>表示したエフェクトブラウザダイアログを返します。</returns>
        public EffectBrowserDialog ShowEffectBrowserDialog()
        {
            EffectBrowserDialog dialog = this.GetEffectBrowserDialog(true);

            if (dialog.Visible == false)
            {
                if (dialog.DockPanel == null)
                {
                    dialog.Show(this.dockPanel, DockState.Float);
                    dialog.FloatPane.FloatWindow.ClientSize = new Size(1300, 800);
                }
                else
                {
                    dialog.Show();
                }
            }

            return dialog;
        }

        /// <summary>
        /// 検索置換ダイアログを表示します。
        /// 現在、ドックパネルには対応していませｎ。
        /// </summary>
        /// <returns>表示した検索置換ダイアログを返します。</returns>
        public SearchDialog ShowSearchDialog()
        {
            SearchDialog dialog = this.GetSearchDialog(true);

            if (dialog.Visible == false)
            {
                dialog.Show();
            }

            return dialog;
        }

        /// <summary>
        /// 検索置換ダイアログ用の XPath テーブルをリロードします。
        /// </summary>
        private void ReloadXPathTable()
        {
            if (this.searchDialog == null)
            {
                return;
            }

            this.searchDialog.ClearXPathTable();

            // メインの XPath テーブルを読み込む
            try
            {
                string mainTableFilePath = Path.Combine(IOConstants.ExecutableFolderPath, @"Addins\XPathTables\MainTable.xml");

                using (FileStream stream = new FileStream(mainTableFilePath, FileMode.Open, FileAccess.Read))
                {
                    this.searchDialog.AddXPathTable(stream);
                }
            }
            catch
            {
            }

            // 列挙値定義データを読み込む
            try
            {
                string enumDefinitionsFilePath = Path.Combine(IOConstants.ExecutableFolderPath, @"Addins\XPathTables\EnumDefinitions.xml");

                using (FileStream stream = new FileStream(enumDefinitionsFilePath, FileMode.Open, FileAccess.Read))
                {
                    this.searchDialog.AddEnumDefinitions(stream);
                }
            }
            catch
            {
            }

            // カスタムアクションの XPath テーブルを読み込む
            foreach (UserDataInfo info in (CustomActionUserDataManager.EnumerateCustomActionUserDataInfo()))
            {
                if (info != null && string.IsNullOrEmpty(info.XPathTable) == false)
                {
                    MemoryStream memoryStream = new MemoryStream(Encoding.UTF8.GetBytes(info.XPathTable));
                    this.searchDialog.AddXPathTable(memoryStream);
                }
            }

            // カスタムシェーダの XPath テーブルを読み込む
            foreach (UserDataInfo info in CustomShaderUserDataManager.EnumerateCustomShaderUserDataInfo())
            {
                if (info != null && string.IsNullOrEmpty(info.XPathTable) == false)
                {
                    MemoryStream memoryStream = new MemoryStream(Encoding.UTF8.GetBytes(info.XPathTable));
                    this.searchDialog.AddXPathTable(memoryStream);
                }
            }

            // エミッタ拡張パラメータの XPath テーブルを読み込む
            {
                UserDataInfo info = EmitterExtParamsUserDataManager.FindEmitterExtParamsUserDataInfo();

                if (info != null && string.IsNullOrEmpty(info.XPathTable) == false)
                {
                    MemoryStream memoryStream = new MemoryStream(Encoding.UTF8.GetBytes(info.XPathTable));
                    this.searchDialog.AddXPathTable(memoryStream);
                }
            }

            // カスタムの XPath テーブルを読み込む（暫定対応）
            // TODO : カスタムアクション・シェーダーと同様の実行時にXPathTableを生成する仕組みにし
            // 選択肢の文字列を取得可能にする
            try
            {
                string customTableFilePath = Path.Combine(IOConstants.ExecutableFolderPath, @"Addins\XPathTables\CustomTable.xml");

                using (FileStream stream = new FileStream(customTableFilePath, FileMode.Open, FileAccess.Read))
                {
                    this.searchDialog.AddXPathTable(stream);
                }
            }
            catch
            {
            }

            // プラグイン形式の検索条件を読み込む
            searchDialog.AddXPathTablePlugin();

            // UI を更新する
            searchDialog.SetupUI();
        }

        /// <summary>
        /// メインフォームのドックパネルにDockContentを追加したときの処理を行います.
        /// DockContentをメインフォームにドッキングしたとき、メインフォームが
        /// 他のアプリケーションウィンドウの後ろに回り込むのを防ぎます.
        /// </summary>
        /// <param name="sender">The sender object.</param>
        /// <param name="e">The event argments.</param>
        private void OnContentAdded(object sender, EventArgs e)
        {
            // メインフォームをアクティブ化する
            this.mainForm.Activate();
        }

        /// <summary>
        /// DockContentsがアクティブ化したときの処理を行います。
        /// ウィンドウをドッキングしたときにメインウィンドウが
        /// 他のアプリケーションウィンドウの後ろに回り込むのを防ぎます。
        /// </summary>
        /// <param name="sender">The sender object.</param>
        /// <param name="e">The event argments.</param>
        private void OnDockContentsActivated(object sender, EventArgs e)
        {
            var content = (WeifenLuo.WinFormsUI.Docking.DockContent)sender;

            // メインフォームにドッキングしているときは何もしない
            if (content.Pane.FloatWindow == null)
            {
                return;
            }

            // ウィンドウをアクティブ化する
            if (content.Pane.IsActivated == false)
            {
                content.Pane.Activate();
            }
        }
    }
}
