﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using EffectMaker.BusinessLogic.Manager;
using EffectMaker.BusinessLogic.UserData;
using EffectMaker.UILogic.ViewModels;
using EffectMaker.DataModelLogic.Utilities;
using EffectMaker.Foundation.Log;
using ModelData = EffectMaker.Foundation.Model.ModelData;
using TextureData = EffectMaker.Foundation.Texture.TextureData;

namespace EffectMaker.Application.CommandLine
{
    /// <summary>
    /// バイナリコンバータの静的解析を行います。
    /// </summary>
    public class StaticProfiler
    {
        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public StaticProfiler()
        {
            this.globalProfile = new GlobalProfile();
            this.emitterProfiles = new List<EmitterProfile>();
        }

        /// <summary>
        /// 静的解析を行います。
        /// </summary>
        /// <param name="emitterSetList">エミッタセットリスト</param>
        /// <param name="session">バイナリコンバートセッション</param>
        public void Analyze(List<EmitterSetViewModel> emitterSetList, WriteBinaryDataSession session)
        {
            // バイナリコンバートセッションの解析情報を取得
            this.globalProfile.BinarySize = session.Profiler.BinarySize;
            this.globalProfile.EmitterSetBlockSize = session.Profiler.EsetBinarySize;

            this.globalProfile.PrimitiveBlockSize = session.Profiler.PrimitiveBlockInfo.Size;
            this.globalProfile.TextureBlockSize = session.Profiler.TextureBlockInfo.Size;
            this.globalProfile.ShaderBinaryBlockSize = session.Profiler.ShaderBinBlockInfo.Size;
            this.globalProfile.ShaderSourceBlockSize = session.Profiler.ShaderSrcBlockInfo.Size;

            // エミッタごとに処理して必要な情報を集める
            // テクスチャやプリミティブの情報は不要であれば消してください
            foreach (EmitterSetViewModel emitterSet in emitterSetList)
            {
                // エミッタごとに処理
                foreach (EmitterViewModel emitter in emitterSet.Emitters)
                {
                    // CPU / GPU / GPU+SO の内訳を計算
                    int processType = emitter.EmitterBasicViewModel.EmitterBasicBasicViewModel.DataModel.ProcessType;

                    switch (processType)
                    {
                        case 0:  // CPU
                            ++globalProfile.NumCpuEmitters;
                            break;

                        case 1:  // GPU
                            ++globalProfile.NumGpuEmitters;
                            break;

                        case 2:  // GPU+SO
                            ++globalProfile.NumGpuSoEmitters;
                            break;
                    }

                    EmitterTextureViewModel[] textureViewModels = GetTextureViewModels(emitter.EmitterTextureGroupViewModel);

                    // テクスチャ情報を取得
                    foreach (EmitterTextureViewModel textureViewModel in textureViewModels)
                    {
                        string texturePath = textureViewModel.FilePath;

                        // テクスチャパスが指定されているときプリミティブ情報を取得する
                        if (string.IsNullOrEmpty(texturePath) == false)
                        {
                            bool isTextureAdded = textures.ContainsKey(texturePath);

                            // 未登録のテクスチャ情報を取得する
                            if (isTextureAdded == false)
                            {
                                var result = TextureManager.Instance.LoadTexture(texturePath, true);

                                if (result.ResultCode == Foundation.Texture.LoadTextureResultCode.Success)
                                {
                                    textures.Add(texturePath, result.TextureData);
                                }
                            }
                        }
                    }

                    // パーティクル形状がプリミティブのときプリミティブ情報を取得
                    if (emitter.EmitterParticleViewModel.EmitterParticleShapeViewModel.ShapeType == 1)  // 1: Primitive
                    {
                        string primitivePath = emitter.EmitterParticleViewModel.EmitterParticleShapeViewModel.PrimitiveFilePath;

                        // プリミティブパスが指定されているときプリミティブ情報を取得する
                        if (string.IsNullOrEmpty(primitivePath) == false)
                        {
                            ModelData model;
                            primitives.TryGetValue(primitivePath, out model);

                            // 未登録のモデル情報を取得する
                            if (model == null) {
                                var result = PrimitiveManager.Instance.LoadModelWithData(primitivePath, true, out model);

                                if (result == Foundation.Model.Types.LoadModelResults.Success)
                                {
                                    primitives.Add(primitivePath, model);
                                }
                            }
                        }
                    }

                    // エミッタ形状がプリミティブのときプリミティブ情報を取得
                    if (emitter.EmitterEmitterViewModel.EmitterEmitterShapeViewModel.DataModel.EmitterType == 15)
                    {
                        string primitivePath = emitter.EmitterEmitterViewModel.EmitterEmitterShapeViewModel.PrimitiveFilePath;

                        // プリミティブパスが指定されているときプリミティブ情報を取得する
                        if (string.IsNullOrEmpty(primitivePath) == false)
                        {
                            ModelData primitive;
                            primitives.TryGetValue(primitivePath, out primitive);

                            // 未登録のモデル情報を取得する
                            if (primitive == null) {
                                var result = PrimitiveManager.Instance.LoadModelWithData(primitivePath, true, out primitive);

                                if (result == Foundation.Model.Types.LoadModelResults.Success)
                                {
                                    primitives.Add(primitivePath, primitive);
                                }
                            }
                        }
                    }
                }
            }

            // エミッタごとに処理
            foreach (EmitterSetViewModel emitterSet in emitterSetList)
            {
                foreach (EmitterViewModel emitter in emitterSet.Emitters)
                {
                    this.emitterProfiles.Add(new EmitterProfile(emitter));
                }
            }
        }

        /// <summary>
        /// 解析結果をファイルに出力します。
        /// </summary>
        public void Output()
        {
            List<string> globalHeader = new List<string>();
            List<string> globalData = new List<string>();

            // グローバル解析情報をまとめる
            // CPU / GPU / GPU+SO の内訳をテーブルに追加
            globalHeader.Add("NumCpuEmitters");
            globalData.Add(globalProfile.NumCpuEmitters.ToString());

            globalHeader.Add("NumGpuEmitters");
            globalData.Add(globalProfile.NumGpuEmitters.ToString());

            globalHeader.Add("NumGpuSoEmitters");
            globalData.Add(globalProfile.NumGpuSoEmitters.ToString());

            // バイナリサイズをテーブルに追加
            globalHeader.Add("BinarySize");
            globalData.Add(globalProfile.BinarySize.ToString());

            // エミッタセットブロックサイズをテーブルに追加
            globalHeader.Add("EmitterSetSize");
            globalData.Add(globalProfile.EmitterSetBlockSize.ToString());

            // シェーダソースコードブロックサイズをテーブルに追加
            globalHeader.Add("ShaderSourceSize");
            globalData.Add(globalProfile.ShaderSourceBlockSize.ToString());

            // シェーダバイナリブロックサイズをテーブルに追加
            globalHeader.Add("ShaderBinarySize");
            globalData.Add(globalProfile.ShaderBinaryBlockSize.ToString());

            // テクスチャブロックサイズをテーブルに追加
            globalHeader.Add("TextureSize");
            globalData.Add(globalProfile.TextureBlockSize.ToString());

            // プリミティブブロックサイズをテーブルに追加
            globalHeader.Add("PrimitiveSize");
            globalData.Add(globalProfile.PrimitiveBlockSize.ToString());

            List<string> emitterHeader = new List<string>();
            List<List<string>> emitterData = new List<List<string>>(this.emitterProfiles.Count);

            // エミッタ解析情報をまとめる
            for (int i = 0; i < this.emitterProfiles.Count; ++i)
            {
                List<string> item = new List<string>();
                emitterData.Add(item);

                // エミッタ名をテーブルに追加
                item.Add(this.emitterProfiles[i].EmitterName);
                if (i == 0)
                {
                    emitterHeader.Add("EmitterName");
                }

                // 親エミッタセット名をテーブルに追加
                item.Add(this.emitterProfiles[i].EmitterSetName);
                if (i == 0)
                {
                    emitterHeader.Add("EmitterSetName");
                }

                // プロセスタイプをテーブルに追加
                item.Add(this.emitterProfiles[i].ProcessType);
                if (i == 0)
                {
                    emitterHeader.Add("ProcessType");
                }

                // カスタムアクションIDをテーブルに追加
                item.Add(this.emitterProfiles[i].CustomActionId);
                if (i == 0)
                {
                    emitterHeader.Add("CustomActionId");
                }

                // カスタムシェーダIDをテーブルに追加
                item.Add(this.emitterProfiles[i].CustomShaderId);
                if (i == 0)
                {
                    emitterHeader.Add("CustomShaderId");
                }

                // コンバイナシェーダのファイルパスをテーブルに追加
                item.Add(this.emitterProfiles[i].CombinerShaderFilePath);
                if (i == 0)
                {
                    emitterHeader.Add("CombinerShaderFilePath");
                }

                // テクスチャ0のファイルパスをテーブルに追加
                item.Add(this.emitterProfiles[i].Texture0FilePath);
                if (i == 0)
                {
                    emitterHeader.Add("Texture0FilePath");
                }

                // テクスチャ1のファイルパスをテーブルに追加
                item.Add(this.emitterProfiles[i].Texture1FilePath);
                if (i == 0)
                {
                    emitterHeader.Add("Texture1FilePath");
                }

                // テクスチャ2のファイルパスをテーブルに追加
                item.Add(this.emitterProfiles[i].Texture2FilePath);
                if (i == 0)
                {
                    emitterHeader.Add("Texture2FilePath");
                }

                // エミッタ形状プリミティブのファイルパスをテーブルに追加
                item.Add(this.emitterProfiles[i].EmitterShapePrimitiveFilePath);
                if (i == 0)
                {
                    emitterHeader.Add("EmitterShapePrimitiveFilePath");
                }

                // パーティクル形状プリミティブのファイルパスをテーブルに追加
                item.Add(this.emitterProfiles[i].ParticleShapePrimitiveFilePath);
                if (i == 0)
                {
                    emitterHeader.Add("ParticleShapePrimitiveFilePath");
                }

                List<ReservedShaderDefinition> reservedShaderDefinitions = ReservedShaderUserDataManager.Definition.ReservedShaderDefinitions;
                bool[] hasReservedShaders = this.emitterProfiles[i].HasReservedShaders;

                // エミッタプラグインの有無をテーブルに追加
                for (int j = 0; j < reservedShaderDefinitions.Count; ++j)
                {
                    item.Add(hasReservedShaders[j] ? "○" : string.Empty);
                    if (i == 0)
                    {
                        string text = reservedShaderDefinitions[j].NameEn.Replace(" ", string.Empty);
                        emitterHeader.Add("Has" + text);
                    }
                }

                // 推定パーティクル最大数をテーブルに追加
                item.Add(this.emitterProfiles[i].MaxParticleCount.ToString());
                if (i == 0)
                {
                    emitterHeader.Add("MaxParticleCount");
                }
            }

            // 出力ファイル名前をチェック
            if (string.IsNullOrEmpty(this.OutputPath))
            {
                Logger.Log(LogLevels.Error, "Output path is empty.");
            }

            FileStream stream = null;
            StreamWriter writer = null;

            try
            {
                // 出力ファイルを開く
                // Excelで文字化けしないようにUTF-8を指定してBOMを出力する
                stream = new FileStream(this.OutputPath, FileMode.Create, FileAccess.Write);
                writer = new StreamWriter(stream, Encoding.GetEncoding("utf-8"));

                // グローバルヘッダを出力
                for (int i = 0; i < globalHeader.Count; ++i)
                {
                    if (i != globalHeader.Count - 1)
                    {
                        string field = this.StringToCsvField(globalHeader[i]);
                        writer.Write(field + ",");
                    }
                    else
                    {
                        string field = this.StringToCsvField(globalHeader[i]);
                        writer.Write(field + "\r\n");
                    }
                }

                // グローバルデータを出力
                for (int i = 0; i < globalData.Count; ++i)
                {
                    if (i != globalData.Count - 1)
                    {
                        string field = this.StringToCsvField(globalData[i]);
                        writer.Write(field + ",");
                    }
                    else
                    {
                        string field = this.StringToCsvField(globalData[i]);
                        writer.Write(field + "\r\n");
                    }
                }

                writer.Write("\r\n");

                // エミッタヘッダを出力
                for (int i = 0; i < emitterHeader.Count; ++i)
                {
                    if (i != emitterHeader.Count - 1)
                    {
                        string field = this.StringToCsvField(emitterHeader[i]);
                        writer.Write(field + ",");
                    }
                    else
                    {
                        string field = this.StringToCsvField(emitterHeader[i]);
                        writer.Write(field + "\r\n");
                    }
                }

                // エミッタデータを出力
                foreach (List<string> item in emitterData)
                {
                    for (int i = 0; i < item.Count; ++i)
                    {
                        if (i != item.Count - 1)
                        {
                            string field = this.StringToCsvField(item[i]);
                            writer.Write(field + ",");
                        }
                        else
                        {
                            string field = this.StringToCsvField(item[i]);
                            writer.Write(field + "\r\n");
                        }
                    }
                }

                writer.Close();
                stream.Close();
            }
            catch (Exception e)
            {
                // エラー出力
                Logger.Log("Console", LogLevels.Error, Properties.Resources.ConsoleMsgFailWriteProfile);
                Logger.Log(LogLevels.Warning, "Caught Exception: {0}", e.Message);
            }
            finally
            {
                // 出力ファイルを閉じる
                if (writer != null)
                {
                    writer.Close();
                }

                if (stream != null)
                {
                    stream.Close();
                }
            }
        }

        /// <summary>
        /// プロファイラをリセットします。
        /// </summary>
        public void Reset()
        {
            this.OutputPath = string.Empty;
        }

        /// <summary>
        /// テクスチャビューモデルリストを取得します。
        /// </summary>
        /// <param name="textureGroup">テクスチャグループ</param>
        /// <returns>テクスチャビューモデルリストを返します。</returns>
        private EmitterTextureViewModel[] GetTextureViewModels (EmitterTextureGroupViewModel textureGroup)
        {
            EmitterTextureViewModel[] textures = new EmitterTextureViewModel[3];
            textures[0] = textureGroup.Texture0;
            textures[1] = textureGroup.Texture1;
            textures[2] = textureGroup.Texture2;

            return textures;
        }

        /// <summary>
        /// 文字列をCSVのフィールドデータに変換します。
        /// コンマ、両端のスペース、改行等が適切に記録されるようにします。
        /// </summary>
        /// <param name="text">文字列</param>
        /// <returns>CSVのフィールドデータを返します。</returns>
        private string StringToCsvField(string text)
        {
            char[] findCh = { '"', ',', '\r', '\n' };
            bool needQuotes = false;

            needQuotes |= text.IndexOfAny(findCh) != -1;
            needQuotes |= text.StartsWith(" ") || text.StartsWith("\t");
            needQuotes |= text.EndsWith(" ") || text.EndsWith("\t");

            if (needQuotes == false)
            {
                return text;
            }

            string field = "\"" + text.Replace("\"", "\"\"") + "\"";

            return field;
        }

        /// <summary>
        /// 出力ファイルパスを取得または設定します。
        /// </summary>
        public string OutputPath { get; set; }

        /// <summary>
        /// グローバル解析情報です。
        /// </summary>
        private GlobalProfile globalProfile;

        /// <summary>
        /// エミッタ解析情報です。
        /// </summary>
        private List<EmitterProfile> emitterProfiles;

        /// <summary>
        /// テクスチャリストです。
        /// </summary>
        Dictionary<string, TextureData> textures = new Dictionary<string, TextureData>();

        /// <summary>
        /// プリミティブリストです。
        /// </summary>
        Dictionary<string, ModelData> primitives = new Dictionary<string, ModelData>();

        /// <summary>
        /// グローバル解析情報です。
        /// </summary>
        private class GlobalProfile
        {
            /// <summary>
            /// バイナリサイズを取得または設定します。
            /// </summary>
            public long BinarySize { get; set; }

            /// <summary>
            /// エミッタセットブロックの容量を取得または設定します。
            /// </summary>
            public long EmitterSetBlockSize { get; set; }

            /// <summary>
            /// テクスチャブロックの容量を取得または設定します。
            /// </summary>
            public long TextureBlockSize { get; set; }

            /// <summary>
            /// プリミティブブロックの容量を取得または設定します。
            /// </summary>
            public long PrimitiveBlockSize { get; set; }

            /// <summary>
            /// シェーダバイナリブロックの容量を取得または設定します。
            /// </summary>
            public long ShaderBinaryBlockSize { get; set; }

            /// <summary>
            /// シェーダソースコードプロックの容量を取得または設定します。
            /// </summary>
            public long ShaderSourceBlockSize { get; set; }

            /// <summary>
            /// GPUエミッタの数を取得または設定します。
            /// </summary>
            public int NumGpuEmitters { get; set; }

            /// <summary>
            /// CPUエミッタの数を取得または設定します。
            /// </summary>
            public int NumCpuEmitters { get; set; }

            /// <summary>
            /// GPU+SOエミッタの数を取得または設定します。
            /// </summary>
            public int NumGpuSoEmitters { get; set; }
        }

        /// <summary>
        /// エミッタ解析情報です。
        /// </summary>
        private class EmitterProfile
        {
            /// <summary>
            /// コンストラクタです。
            /// </summary>
            /// <param name="vm">ビューモデル</param>
            public EmitterProfile(EmitterViewModel vm)
            {
                this.Vm = vm;
            }

            /// <summary>
            /// エミッタビューモデルを取得します。
            /// </summary>
            public EmitterViewModel Vm { get; private set; }

            /// <summary>
            /// エミッタ名を取得します。
            /// </summary>
            public string EmitterName
            {
                get
                {
                    return this.Vm.Name;
                }
            }

            /// <summary>
            /// 親エミッタセットの名前を取得します。
            /// </summary>
            public string EmitterSetName
            {
                get
                {
                    EmitterSetViewModel parent = HierarchyViewModel.GetParent<EmitterSetViewModel>(this.Vm);
                    if (parent == null)
                    {
                        Logger.Log(LogLevels.Warning, "Parent emitter set is null.");
                        return null;
                    }

                    return parent.DataModel.Name;
                }
            }

            /// <summary>
            /// プロセスタイプを取得します。
            /// </summary>
            public string ProcessType
            {
                get
                {
                    // プロセスタイプを取得
                    int processType = this.Vm.EmitterBasicViewModel.EmitterBasicBasicViewModel.DataModel.ProcessType;

                    // プロセスタイプを文字列にして返す
                    switch (processType)
                    {
                        case 0:
                            return "○ CPU";

                        case 1:
                            return "◎ GPU";

                        case 2:
                            return "● GPU+SO";

                        default:
                            return "Invalid";
                    }
                }
            }

            /// <summary>
            /// カスタムアクションIDを取得します。
            /// </summary>
            public string CustomActionId
            {
                get
                {
                    int customActionId = -1;

                    // カスタムアクションIDを取得
                    CustomActionViewModel customAction = this.Vm.GetChild<CustomActionViewModel>();
                    if (customAction != null && customAction.DataModel.EnableConvert)
                    {
                        customActionId = customAction.DataModel.SelectedSettingIndex;
                    }

                    if (customActionId == -1)
                    {
                        return string.Empty;  // 未使用のときは空文字を返す
                    }
                    else
                    {
                        return customActionId.ToString();
                    }
                }
            }

            /// <summary>
            /// カスタムシェーダIDを取得します。
            /// </summary>
            public string CustomShaderId
            {
                get
                {
                    // カスタムシェーダIDを取得
                    int customShaderId = this.Vm.EmitterCustomShaderViewModel.EmitterCustomShaderBasicViewModel.DataModel.SelectedSettingIndex;

                    // カスタムアクションIDを取得
                    if (customShaderId == -1)
                    {
                        return string.Empty;  // 未使用のときは空文字を返す
                    }
                    else
                    {
                        return customShaderId.ToString();
                    }
                }
            }

            /// <summary>
            /// コンバイナシェーダのファイルパスを取得します。
            /// </summary>
            public string CombinerShaderFilePath
            {
                get
                {
                    string path = this.Vm.EmitterCombinerViewModel.EmitterCombinerEditorViewModel.CombinerEditorProjectPath;
                    if (string.IsNullOrEmpty(path) == false)
                    {
                        path = Path.GetFileName(path);
                    }

                    return path;
                }
            }

            /// <summary>
            /// テクスチャ0のファイルパス取得します。
            /// </summary>
            public string Texture0FilePath
            {
                get
                {
                    string path = this.Vm.EmitterTextureGroupViewModel.Texture0.EmitterTextureFileViewModel.FilePath;
                    if (string.IsNullOrEmpty(path) == false)
                    {
                        path = Path.GetFileName(path);
                    }

                    return path;
                }
            }

            /// <summary>
            /// テクスチャ1のファイルパスを取得します。
            /// </summary>
            public string Texture1FilePath
            {
                get
                {
                    string path = this.Vm.EmitterTextureGroupViewModel.Texture1.EmitterTextureFileViewModel.FilePath;
                    if (string.IsNullOrEmpty(path) == false)
                    {
                        path = Path.GetFileName(path);
                    }

                    return path;
                }
            }

            /// <summary>
            /// テクスチャ2のファイルパスを取得します。
            /// </summary>
            public string Texture2FilePath
            {
                get
                {
                    string path = this.Vm.EmitterTextureGroupViewModel.Texture2.EmitterTextureFileViewModel.FilePath;
                    if (string.IsNullOrEmpty(path) == false)
                    {
                        path = Path.GetFileName(path);
                    }

                    return path;
                }
            }

            /// <summary>
            /// エミッタ形状プリミティブのファイルパスを取得します。
            /// </summary>
            public string EmitterShapePrimitiveFilePath
            {
                get
                {
                    string path = this.Vm.EmitterEmitterViewModel.EmitterEmitterShapeViewModel.PrimitiveFilePath;
                    if (string.IsNullOrEmpty(path) == false)
                    {
                        path = Path.GetFileName(path);
                    }

                    return path;
                }
            }

            /// <summary>
            /// パーティクル形状プリミティブのファイルパスを取得します。
            /// </summary>
            public string ParticleShapePrimitiveFilePath
            {
                get
                {
                    string path = this.Vm.EmitterParticleViewModel.EmitterParticleShapeViewModel.PrimitiveFilePath;
                    if (string.IsNullOrEmpty(path) == false)
                    {
                        path = Path.GetFileName(path);
                    }

                    return path;
                }
            }

            /// <summary>
            /// エミッタプラグインの有無を取得します。
            /// この配列は書き換え無効です。
            /// </summary>
            public bool[] HasReservedShaders
            {
                get
                {
                    List<ReservedShaderDefinition> reservedShaderDefinitions = ReservedShaderUserDataManager.Definition.ReservedShaderDefinitions;
                    bool[] hasReservedShaders = new bool[reservedShaderDefinitions.Count];

                    // エミッタプラグインビューモデルを取得。
                    ReservedShaderNodeViewModel reservedShaderVm = this.Vm.GetChild<ReservedShaderNodeViewModel>();
                    if (reservedShaderVm == null)
                    {
                        return hasReservedShaders;
                    }

                    ReservedShaderDefinition definition = ReservedShaderUserDataManager.FindReservedShaderDefinition(reservedShaderVm.DataModel.PageData.ContentsData);
                    int index = 0;

                    // エミッタプラグインに対応するインデックスのフラグを立てる
                    while (definition != reservedShaderDefinitions[index])
                    {
                        ++index;
                    }

                    if (index < reservedShaderDefinitions.Count) {
                        hasReservedShaders[index] = true;
                    }

                    return hasReservedShaders;
                }
            }

            /// <summary>
            /// 推定パーティクル最大数を取得します。
            /// </summary>
            public int MaxParticleCount
            {
                get
                {
                    // 放出間隔
                    // this.Vm.EmitterEmissionViewModel.EmitterEmissionTimingViewModel.DataModel.EmitTimeDistUnit

                    // 放出レート
                    // this.Vm.EmitterEmissionViewModel.EmitterEmissionTimingViewModel.DataModel.EmitTimeEmissionRate

                    return -1;
                }
            }
        }
    }
}
