﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.IO;
using System.Threading;
using EffectMaker.BusinessLogic.Options;
using EffectMaker.BusinessLogic.ProjectConfig;
using EffectMaker.BusinessLogic.UserData;
using EffectMaker.Foundation.Log;
using EffectMaker.Foundation.Utility;

namespace EffectMaker.Application.CommandLine
{
    /// <summary>
    /// プロジェクト設定オプションを処理します。
    /// </summary>
    public class ProjectConfigProcessor : CommandLineProcessorBase
    {
        /// <summary>
        /// EffectMaker起動時に読み込まれるプロジェクト設定
        /// </summary>
        private EffectMakerProjectConfig defaultProjectConfig;

        /// <summary>
        /// プロジェクト設定ファイルです。
        /// </summary>
        private string projectConfigFile;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="workingData">作業データ</param>
        /// <param name="projectConfigFile">プロジェクト設定ファイル</param>
        public ProjectConfigProcessor(CommandLineWorkingData workingData, string projectConfigFile)
            : base(workingData)
        {
            this.projectConfigFile = projectConfigFile;
        }

        /// <summary>
        /// エラーログを取得します。
        /// </summary>
        public override string ErrorReport　{ get { return string.Empty; }　}

        /// <summary>
        /// 設定処理を行います。
        /// </summary>
        /// <returns>処理が正常に完了したときtrue, それ以外はfalseを返します。</returns>
        public override bool Setup()
        {
            // プロジェクト設定を保存
            this.defaultProjectConfig = new EffectMakerProjectConfig(OptionStore.ProjectConfig);

            // プロジェクト設定ファイルの絶対パスを取得
            this.WorkingData.ProjectConfigFilePath =
                PathUtility.ToAbsolutePath(this.projectConfigFile, TheApp.WorkingFolder);

            Logger.Log(
                "Console",
                LogLevels.Information,
                string.Format("Loading project config...\r\n->{0}", this.WorkingData.ProjectConfigFilePath));

            if (File.Exists(this.WorkingData.ProjectConfigFilePath) == false)
            {
                Logger.Log(
                    "Console",
                    LogLevels.Error,
                    Properties.Resources.ConsoleMsgProjectConfigFileNotFound,
                    this.WorkingData.ProjectConfigFilePath);

                return false;
            }

            // プロジェクト設定ファイルをロード
            try
            {
                TheApp.Mx.WaitOne();

                bool loaded = OptionStore.LoadProjectConfig(this.WorkingData.ProjectConfigFilePath);
                if (!loaded)
                {
                    Logger.Log(
                        "Console",
                        LogLevels.Error,
                        Properties.Resources.ConsoleMsgFailLoadProjectConfigFile,
                        this.WorkingData.ProjectConfigFilePath);

                    return false;
                }

                // ファイルイベントのマージを行う
                OptionUtil.MergeFileEvent();
            }
            finally
            {
                TheApp.Mx.ReleaseMutex();
            }

            Logger.Log(
                "Console",
                LogLevels.Information,
                "The config is loaded.");

            // 描画パスの表示
            if (OptionStore.ProjectConfig.DrawPaths != null)
            {
                Logger.Log(
                    "Console",
                    LogLevels.Information,
                    "\r\nDefined draw paths as follows...");
                foreach (var drawPath in OptionStore.ProjectConfig.DrawPaths)
                {
                    Logger.Log(
                        "Console",
                        LogLevels.Information,
                        string.Format("[{0}]:{1}", drawPath.Id, drawPath.Text));
                }

                if (OptionStore.ProjectConfig.DefaultDrawPath != null)
                {
                    Logger.Log(
                        "Console",
                        LogLevels.Information,
                        string.Format("The default draw path is {0}", OptionStore.ProjectConfig.DefaultDrawPath.Text));
                }
            }

            if (!string.IsNullOrEmpty(OptionStore.ProjectConfig.CustomShaderPath))
            {
                try
                {
                    TheApp.Mx.WaitOne();

                    if (UserDataHelper.Instance.LoadCustomShaderUserData(
                        OptionStore.ProjectConfig.CustomShaderPath, false) != null)
                    {
                        Logger.Log(
                            "Console",
                            LogLevels.Information,
                            "\r\nCustom Shader {0} is loaded.",
                            OptionStore.ProjectConfig.CustomShaderPath);
                    }
                    else
                    {
                        Logger.Log(
                            "Console",
                            LogLevels.Information,
                            "\r\nError: Custom Shader {0} is NOT loaded.",
                            OptionStore.ProjectConfig.CustomShaderPath);

                        return false;
                    }
                }
                finally
                {
                    TheApp.Mx.ReleaseMutex();
                }
            }

            if (!string.IsNullOrEmpty(OptionStore.ProjectConfig.CustomActionPath))
            {
                try
                {
                    TheApp.Mx.WaitOne();

                    if (UserDataHelper.Instance.LoadCustomActionUserData(
                        OptionStore.ProjectConfig.CustomActionPath, false) != null)
                    {
                        Logger.Log(
                            "Console",
                            LogLevels.Information,
                            "\r\nCustom Action {0} is loaded.",
                            OptionStore.ProjectConfig.CustomActionPath);
                    }
                    else
                    {
                        Logger.Log(
                            "Console",
                            LogLevels.Information,
                            "\r\nError: Custom Action {0} is NOT loaded.",
                            OptionStore.ProjectConfig.CustomActionPath);

                        return false;
                    }
                }
                finally
                {
                    TheApp.Mx.ReleaseMutex();
                }
            }

            if (!string.IsNullOrEmpty(OptionStore.ProjectConfig.EmitterExtParamsPath))
            {
                try
                {
                    TheApp.Mx.WaitOne();

                    if (UserDataHelper.Instance.LoadEmitterExtParamsUserData(
                        OptionStore.ProjectConfig.EmitterExtParamsPath, false) != null)
                    {
                        Logger.Log(
                            "Console",
                            LogLevels.Information,
                            "\r\nEmitter Extension Parameters {0} is loaded.",
                            OptionStore.ProjectConfig.EmitterExtParamsPath);
                    }
                    else
                    {
                        Logger.Log(
                            "Console",
                            LogLevels.Information,
                            "\r\nError: Emitter Extension Parameters {0} is NOT loaded.",
                            OptionStore.ProjectConfig.EmitterExtParamsPath);

                        return false;
                    }
                }
                finally
                {
                    TheApp.Mx.ReleaseMutex();
                }
            }

            // 描画パス設定に関するエラーを表示
            foreach (string error in OptionStore.ProjectConfig.DrawPathSettingErrors)
            {
                Logger.Log("Console", LogLevels.Warning, error);
            }

            // エフェクト定義フォルダを表示
            if (OptionStore.ProjectConfig.EffectCombinerBlockDefinitionPaths != null)
            {
                Logger.Log(
                    "Console",
                    LogLevels.Information,
                    "\r\nDefined effect combiner paths as follows...");
                foreach (var path in OptionStore.ProjectConfig.EffectCombinerBlockDefinitionPaths)
                {
                    Logger.Log("Console", LogLevels.Information, path);
                }
            }

            return true;
        }

        /// <summary>
        /// コマンドライン処理を行います。
        /// </summary>
        /// <returns>処理が正常に完了したときtrue, それ以外はfalseを返します。</returns>
        public override bool Process()
        {
            return true;
        }

        /// <summary>
        /// 書き換えた設定などを復元します。
        /// </summary>
        public override void Cleanup()
        {
            try
            {
                TheApp.Mx.WaitOne();

                // プロジェクト設定を復元
                if (this.defaultProjectConfig != null)
                {
                    OptionStore.ProjectConfig.Set(this.defaultProjectConfig);

                    UserDataHelper.Instance.LoadAndSetupCustomShaderAndAction(
                        OptionStore.ProjectConfig.CustomShaderPath,
                        OptionStore.ProjectConfig.CustomActionPath);
                    UserDataHelper.Instance.LoadEmitterExtParamsUserData(
                        OptionStore.ProjectConfig.EmitterExtParamsPath, true);

                    this.defaultProjectConfig = null;
                }
            }
            finally
            {
                TheApp.Mx.ReleaseMutex();
            }
        }
    }
}
