﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.IO;
using System.Text;
using EffectMaker.Foundation.Log;
using EffectMaker.Foundation.Utility;
using EffectMaker.UILogic.ViewModels;

namespace EffectMaker.Application.CommandLine
{
    /// <summary>
    /// バイナリ出力オプションを処理します。
    /// </summary>
    public class BinaryOutputProcessor : CommandLineProcessorBase
    {
        /// <summary>
        /// 出力バイナリファイルです。
        /// </summary>
        private string outputBinaryFile;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="workingData">作業データ</param>
        /// <param name="outputBinaryFile">出力バイナリファイル</param>
        public BinaryOutputProcessor(CommandLineWorkingData workingData, string outputBinaryFile)
            : base(workingData)
        {
            this.outputBinaryFile = outputBinaryFile;
        }

        /// <summary>
        /// エラーが何も発生せずに終了できたか
        /// </summary>
        public bool IsCompletedSuccessfully
        {
            get { return this.WorkingData.Converter.IsCompletedSuccessfully; }
        }

        /// <summary>
        /// エラーログを取得します。
        /// </summary>
        public override string ErrorReport
        {
            get
            {
                var combinedLog = new StringBuilder();
                foreach (var log in this.WorkingData.Converter.ErrorLog)
                {
                    combinedLog.AppendLine(log);
                }

                return combinedLog.ToString();
            }
        }

        /// <summary>
        /// 設定処理を行います。
        /// </summary>
        /// <returns>処理が正常に完了したときtrue, それ以外はfalseを返します。</returns>
        public override bool Setup()
        {
            // バイナリ出力オプションが指定されているのに、--api-typeオプションが指定されていない場合はエラーを返す
            if (this.WorkingData.ApiTypeOptionEnabled == false)
            {
                Logger.Log("Console", LogLevels.Error, Properties.Resources.ConsoleMsgApiTypeNotSpecified);
                return false;
            }

            // 出力バイナリファイルパスを取得
            this.WorkingData.Converter.BinaryPath =
                PathUtility.ToAbsolutePath(this.outputBinaryFile, TheApp.WorkingFolder);

            // 出力ディレクトリが存在するかチェック
            string dirPath = PathUtility.GetDirectoryName(this.WorkingData.Converter.BinaryPath);

            if (Directory.Exists(dirPath) == false)
            {
                Logger.Log(
                    "Console",
                    LogLevels.Error,
                    Properties.Resources.ConsoleMsgOutputDirNotFound,
                    dirPath);

                return false;
            }

            // --resident-texture-listオプションが、
            // --input-resident-asset-listオプションか--output-resident-asset-listオプションと同時に指定されていないかチェック
            if ( !string.IsNullOrEmpty(this.WorkingData.Converter.ResidentTextureListPath) &&
                 (!string.IsNullOrEmpty(this.WorkingData.Converter.InputResidentAssetListPath) || !string.IsNullOrEmpty(this.WorkingData.Converter.OutputResidentAssetListPath)) )
            {
                Logger.Log(
                    "Console",
                    LogLevels.Error,
                    Properties.Resources.ConsoleMsgOutputInvalidOptionCombination,
                    dirPath);

                return false;
            }

            // ヘッダファイルの出力オプションが指定されているとき
            if (this.WorkingData.Converter.OutputHeaderEnabled == true)
            {
                // エミッタセットとエミッタのインデックスを管理するインスタンス
                this.WorkingData.Converter.EmitterSetIndexes = new List<KeyValuePair<string, int>>();
                this.WorkingData.Converter.EmitterIndexes = new List<KeyValuePair<string, int>>();
            }

            return true;
        }

        /// <summary>
        /// コマンドライン処理を行います。
        /// </summary>
        /// <returns>処理が正常に完了したときtrue, それ以外はfalseを返します。</returns>
        public override bool Process()
        {
            return this.WorkingData.Converter.Convert();
        }

        /// <summary>
        /// 書き換えた設定などを復元します。
        /// </summary>
        public override void Cleanup()
        {
        }
    }
}
