﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using EffectMaker.Application.Properties;
using EffectMaker.BusinessLogic.SpecDefinitions;
using EffectMaker.Foundation.Log;

namespace EffectMaker.Application.CommandLine
{
    /// <summary>
    /// APIタイプ(プラットフォーム)オプションを処理します。
    /// </summary>
    public class ApiTypeProcessor : CommandLineProcessorBase
    {
        /// <summary>
        /// APIタイプ名です。
        /// </summary>
        private readonly string apiType;

        /// <summary>
        /// コードタイプ名です。
        /// </summary>
        private readonly string codeType;

        /// <summary>
        /// タイリングモード名です。
        /// </summary>
        private readonly string tileMode;

        /// <summary>
        /// タイリング最適化モード名です。
        /// </summary>
        private readonly string tileOptimize;

        /// <summary>
        /// タイリングサイズ最適化のしきい値です。
        /// </summary>
        private readonly int? tileSizeThreshold;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="workingData">作業データ</param>
        /// <param name="apiType">APIタイプ名</param>
        /// <param name="codeType">コードタイプ名</param>
        /// <param name="tileMode">タイリングモード名</param>
        /// <param name="tileOptimize">タイリング最適化モード名</param>
        /// <param name="tileSizeThreshold">タイリングサイズ最適化のしきい値</param>
        public ApiTypeProcessor(CommandLineWorkingData workingData, string apiType, string codeType, string tileMode, string tileOptimize, int? tileSizeThreshold)
            : base(workingData)
        {
            this.apiType = apiType;
            this.codeType = codeType;
            this.tileMode = tileMode;
            this.tileOptimize = tileOptimize;
            this.tileSizeThreshold = tileSizeThreshold;
        }

        /// <summary>
        /// エラーログを取得します。
        /// </summary>
        public override string ErrorReport { get { return string.Empty; } }

        /// <summary>
        /// 設定処理を行います。
        /// </summary>
        /// <param name="args">パラメーター</param>
        /// <returns>処理が正常に完了したときtrue, それ以外はfalseを返します。</returns>
        public override bool Setup()
        {
            // オプションで指定された環境を探す
            var specName = this.apiType.ToLower();
            int i;
            for (i = 0; i < SpecManager.SpecDefinitions.Count; i++)
            {
                var spec = SpecManager.SpecDefinitions[i];
                if (spec.IsSpecified(specName))
                {
                    SpecManager.SelectedIndex = i;
                    break;
                }
            }

            // オプションで指定された環境がなかったら、エラー
            if (i >= SpecManager.SpecDefinitions.Count)
            {
                Logger.Log("Console", LogLevels.Error, Properties.Resources.ConsoleMsgInvalidApiTypeOption, this.apiType);
                return false;
            }

            // EFTBは特殊対応でコードタイプとタイリングモードの指定は無しでOKとする
            if (SpecManager.CurrentSpec.BinaryHeader == "EFTB")
            {
                // むしろ指定されていたらエラーとする
                if (!string.IsNullOrEmpty(this.codeType))
                {
                    Logger.Log(
                        "Console",
                        LogLevels.Error,
                        Resources.ConsoleMsgCantSpecifiedCodeType);
                    return false;
                }

                if (!string.IsNullOrEmpty(this.tileMode))
                {
                    Logger.Log(
                        "Console",
                        LogLevels.Error,
                        Resources.ConsoleMsgCantSpecifiedTileMode);
                    return false;
                }

                return true;
            }

            if (!string.IsNullOrEmpty(this.codeType))
            {
                if (SpecManager.CurrentSpec.IsValidCodeType(this.codeType))
                {
                    SpecManager.CurrentSpec.ShaderConversionOption.CodeType = this.codeType;
                }
                else
                {
                    Logger.Log(
                        "Console",
                        LogLevels.Error,
                        Properties.Resources.ConsoleMsgInvalidCodeTypeOption,
                        this.codeType);
                    return false;
                }
            }
            else
            {
                Logger.Log(
                    "Console",
                    LogLevels.Warning,
                    Resources.ConsoleMsgWarningCodeTypeNotSpecified,
                    SpecManager.CurrentSpec.ShaderConversionOption.CodeType);
            }

            if (!string.IsNullOrEmpty(this.tileMode))
            {
                if (SpecManager.CurrentSpec.IsValidTileMode(this.tileMode))
                {
                    SpecManager.CurrentSpec.TexturePackingOption.TileMode = this.tileMode;
                }
                else
                {
                    Logger.Log(
                        "Console",
                        LogLevels.Error,
                        Properties.Resources.ConsoleMsgInvalidTileModeOption,
                        this.tileMode);
                    return false;
                }
            }
            else
            {
                Logger.Log(
                    "Console",
                    LogLevels.Warning,
                    Resources.ConsoleMsgWarningTileModeNotSpecified,
                    SpecManager.CurrentSpec.TexturePackingOption.TileMode);
            }

            if (!string.IsNullOrEmpty(this.tileOptimize))
            {
                if (SpecManager.CurrentSpec.IsValidTileOptimize(this.tileOptimize))
                {
                    SpecManager.CurrentSpec.TexturePackingOption.TileOptimize = this.tileOptimize;
                }
                else
                {
                    Logger.Log(
                        "Console",
                        LogLevels.Error,
                        Resources.ConsoleMsgInvalidTileOptimizeOption,
                        this.tileOptimize);
                    return false;
                }
            }

            if (this.tileSizeThreshold != null)
            {
                if (SpecManager.CurrentSpec.IsValidTileSizeThreshold(this.tileSizeThreshold.Value))
                {
                    SpecManager.CurrentSpec.TexturePackingOption.TileSizeThreshold = this.tileSizeThreshold;
                }
                else
                {
                    Logger.Log(
                        "Console",
                        LogLevels.Error,
                        Resources.ConsoleMsgInvalidTileSizeThresholdOption,
                        this.tileSizeThreshold);
                    return false;
                }
            }

            return true;
        }

        /// <summary>
        /// コマンドライン処理を行います。
        /// </summary>
        /// <returns>処理が正常に完了したときtrue, それ以外はfalseを返します。</returns>
        public override bool Process()
        {
            return true;
        }

        /// <summary>
        /// 書き換えた設定などを復元します。
        /// </summary>
        public override void Cleanup()
        {
        }
    }
}
