﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

// 旧形式で記述されたコマンドラインオプションに対応するためのフラグです。
#define CompatibleCommandLineOption

using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using EffectMaker.Application.Properties;
using EffectMaker.BusinessLogic.BinaryHeaders;
using EffectMaker.BusinessLogic.IO;
using EffectMaker.BusinessLogic.SpecDefinitions;
using EffectMaker.Foundation.Log;
using Nintendo.Foundation.IO;

namespace EffectMaker.Application.CommandLine
{
    /// <summary>
    /// コマンドライン処理を管理するクラスです。
    /// </summary>
    public static class CommandLineApplication
    {
        /// <summary>
        /// 終了コードの定義です。
        /// </summary>
        public enum ExitCodeKind : int
        {
            NoErrors,
            WarningOnConvertProcess,
            InvalidArguments,
            FailedAnySetup,
            FailedAnyProcess,
            ExceptionOccured,
        }

        /// <summary>
        /// 終了コードを取得します。
        /// </summary>
        public static int ExitCode { get; private set; }

        /// <summary>
        /// 引数をチェックしてコマンドラインモードのON/OFFを取得します。
        /// </summary>
        /// <param name="args">コマンドライン引数</param>
        /// <returns>コマンドラインモードのON/OFF</returns>
        public static bool ShouldEnableCommandLine(string[] args)
        {
            // 引数が無い場合はGUIモードとする
            if (!args.Any())
            {
                return false;
            }

            // コマンドライン引数を解析
            CompatibleCommandLineParams parameters = ParseArgs(args, true);

            // 解析に失敗した場合、全ての引数が有効なファイルパスであればGUIモードにする
            // -? ヘルプ呼び出しのみの場合も解析失敗扱いになりCUIでヘルプが表示される
            if (parameters == null)
            {
                bool isValidFilePath = args.Aggregate(true, (current, arg) => current & System.IO.File.Exists(arg));
                return !isValidFilePath;
            }

            // コンバートする場合はCUIモードとする
            if (parameters.ConvertListFile != null)
            {
                return true;
            }

            // 中間ファイルアップデートする場合はCUIモードとする
            if (parameters.UpdateListFile != null || parameters.UpdateCE2ListFile != null)
            {
                return true;
            }

            // その他の場合はGUIモードとする
            return false;
        }

        /// <summary>
        /// コマンドライン処理を行います。
        /// </summary>
        /// <param name="args">コマンドライン引数</param>
        /// <returns>True on success, otherwise, false is returned if any of the processor failed.</returns>
        public static bool Run(string[] args)
        {
            ////System.Windows.Forms.MessageBox.Show("プロセスにアタッチしてください。");

            ExitCode = (int)ExitCodeKind.NoErrors;

            // コンソール画面を初期化
            AppData.ConsoleDisplay = new ConsoleDisplay();
            AppData.ConsoleDisplay.Initialize();

            Logger.RegisterLogHandler(AppData.ConsoleDisplay);

            // コマンドライン引数を解析
            CompatibleCommandLineParams parameters = ParseArgs(args, false);

            if (parameters == null)
            {
                // コンソール画面を破棄
                Logger.UnregisterLogHandler(AppData.ConsoleDisplay);
                AppData.ConsoleDisplay.Release();
                AppData.ConsoleDisplay = null;

                ExitCode = (int)ExitCodeKind.InvalidArguments;

                return false;
            }

            // 位置引数はコマンドラインで無効なため警告メッセージを出す
            if (parameters.StartupFiles != null)
            {
                bool invalidArgs = false;

                foreach (var file in parameters.StartupFiles)
                {
                    Logger.Log(
                        "Console",
                        LogLevels.Warning,
                        Resources.ConsoleMsgWarningInvalidArg,
                        file);

                    invalidArgs = true;
                }

                if (invalidArgs)
                {
                    ExitCode = (int)ExitCodeKind.InvalidArguments;

                    return false;
                }
            }

            CommandLineWorkingData workingData = new CommandLineWorkingData();

            // プロセッサを作成
            List<CommandLineProcessorBase> processors = CreateProcessors(workingData, parameters);

            try
            {
                // プロセッサを初期化
                foreach (CommandLineProcessorBase processor in processors)
                {
                    if (processor.Setup() == false)
                    {
                        ExitCode = (int)ExitCodeKind.FailedAnySetup;
                        throw new Exception(
                            string.Format("setup of {0}.{1}{2}",
                            processor.GetType().Name,
                            Environment.NewLine,
                            processor.ErrorReport));
                    }
                }

                // プロセッサの処理を実行
                foreach (CommandLineProcessorBase processor in processors)
                {
                    if (processor.Process() == false)
                    {
                        ExitCode = (int)ExitCodeKind.FailedAnyProcess;
                        throw new Exception(
                            string.Format("process of {0}.{1}{2}",
                            processor.GetType().Name,
                            Environment.NewLine,
                            processor.ErrorReport));
                    }
                }
            }
            catch (Exception e)
            {
                if (ExitCode == (int)ExitCodeKind.NoErrors)
                {
                    ExitCode = (int)ExitCodeKind.ExceptionOccured;
                }

                Logger.Log(
                    "Console",
                    LogLevels.Error,
                    string.Format("{0}Abort on {1}", Environment.NewLine, e.Message));
            }
            finally
            {
                // プロセッサが書き換えた設定を復元
                foreach (CommandLineProcessorBase processor in processors)
                {
                    processor.Cleanup();
                }

                // コンソール画面を破棄
                Logger.UnregisterLogHandler(AppData.ConsoleDisplay);
                AppData.ConsoleDisplay.Release();
                AppData.ConsoleDisplay = null;
            }

            // BinaryOutputProcessorで何かエラーがあったかチェックする
            var binaryOutputProcessor = (BinaryOutputProcessor)processors.FirstOrDefault(proc => proc is BinaryOutputProcessor);
            if (binaryOutputProcessor != null && !binaryOutputProcessor.IsCompletedSuccessfully)
            {
                if (ExitCode == (int)ExitCodeKind.NoErrors)
                {
                    ExitCode = (int)ExitCodeKind.WarningOnConvertProcess;
                }
            }

            return ExitCode == (int)ExitCodeKind.NoErrors;
        }

        /// <summary>
        /// コマンドライン引数をパースします。
        /// </summary>
        /// <param name="args">コマンドライン引数</param>
        /// <param name="silent">出力を無効にするか</param>
        /// <returns>解析結果を返します。</returns>
        public static CompatibleCommandLineParams ParseArgs(string[] args, bool silent)
        {
#if CompatibleCommandLineOption
            // argsを書き換えるためクローンを作成
            args = (string[])args.Clone();

            // ハイフン1個のロングネームオプションはパーサーが対応してないのでハイフン2個にしてあげる
            {
                HashSet<string> obsoleteParams = new HashSet<string>()
                {
                    "-convlist",
                    "-prjconfig",
                    "-update",
                    "-shaderOpt",
                    "-noShaderSrc",
                    "-spec"
                };

                bool findObsoleteParam = false;

                for (int i = 0; i < args.Length; ++i)
                {
                    if (obsoleteParams.Contains(args[i]))
                    {
                        args[i] = '-' + args[i];
                        findObsoleteParam = true;
                    }
                }

                if (!silent && findObsoleteParam)
                {
                    Logger.Log("Console", LogLevels.Warning, Resources.ConsoleMsgUsingObsoleteLongNameOption);
                }
            }

            // -?オプションは個別に置換してあげる
            {
                bool findObsoleteHelp = false;

                for (int i = 0; i < args.Length; ++i)
                {
                    if (args[i] == "-?")
                    {
                        args[i] = "--help";
                        findObsoleteHelp = true;
                    }
                }

                if (!silent && findObsoleteHelp)
                {
                    string warningMsg = string.Format(Resources.ConsoleMsgReplaceOptionName, "-?", "-h, --help");
                    Logger.Log("Console", LogLevels.Warning, warningMsg);
                }
            }
#endif

            // バイナリ情報を取得
            System.Reflection.Assembly assembly = System.Reflection.Assembly.GetExecutingAssembly();
            System.Diagnostics.FileVersionInfo assemblyInfo = System.Diagnostics.FileVersionInfo.GetVersionInfo(assembly.Location);

            // パーサーを作成
            StringWriter helpWriter = new StringWriter();
            StringWriter errorWriter = new StringWriter();

            var settings = new CommandLineParserSettings()
            {
                HelpWriter = helpWriter.Write,
                ErrorWriter = errorWriter.Write,
                ApplicationName = assemblyInfo.ProductName,
                Version = string.Format("{0}.{1}.{2}", assemblyInfo.ProductMajorPart, assemblyInfo.ProductMinorPart, assemblyInfo.ProductBuildPart),
                ApplicationDescription = Resources.ConsoleDescApplicationDescription
            };

            var commandLineParser = new CommandLineParser(settings);

            bool resParse = false;
            CompatibleCommandLineParams parameters = null;

            // パース処理を実行
            try
            {
                //// ParseArgs()の挙動
                //// パース成功     : params = input, return = true , throw = false
                //// パースエラー   : params = null , return = false, throw = true
                //// ヘルプ表示     : params = null , return = false, throw = false
                //// バージョン表示 : params = null , return = false, throw = false
                resParse = commandLineParser.ParseArgs(args, out parameters);
            }
            catch (Exception e)
            {
                // エラーメッセージを表示
                string errorText = errorWriter.ToString();

                if (!silent)
                {
                    if (errorText.Length != 0)
                    {
                        // パースエラーを表示
                        Logger.Log("Console", LogLevels.Error, errorText);
                    }
                    else if (!silent)
                    {
                        // DLLが読み込めないときなどのアプリケーションエラーを表示
                        Logger.Log("Console", LogLevels.Error, e.Message);
                    }
                }

                return null;
            }

            // ヘルプ情報またはバージョン情報を表示
            if (resParse == false)
            {
                if (!silent)
                {
#if CompatibleCommandLineOption
                    // 旧形式のオプションの説明はいらないので新形式だけのヘルプ文字列を取得する
                    helpWriter = new StringWriter();
                    errorWriter = new StringWriter();

                    settings.HelpWriter = helpWriter.Write;
                    settings.ErrorWriter = errorWriter.Write;

                    var helpParameters = new CommandLineParams();

                    commandLineParser.ParseArgs(args, out helpParameters);
#endif

                    string helpText = helpWriter.ToString();
                    Logger.Log("Console", LogLevels.Information, helpText);
                }

                return null;
            }

#if CompatibleCommandLineOption
            // 旧形式のコマンドが使われていたときに警告を表示
            if (!silent)
            {
                if (parameters.UsingObsoleteConvertListFile)
                {
                    string warningMsg = string.Format(Resources.ConsoleMsgReplaceOptionName, "-convlist", "--convert-list");
                    Logger.Log("Console", LogLevels.Warning, warningMsg);
                }

                if (parameters.UsingObsoleteProjectConfigFile)
                {
                    string warningMsg = string.Format(Resources.ConsoleMsgReplaceOptionName, "-prjconfig", "--team-setting-file");
                    Logger.Log("Console", LogLevels.Warning, warningMsg);
                }

                if (parameters.UsingObsoleteShaderConvertOption)
                {
                    string warningMsg = string.Format(Resources.ConsoleMsgReplaceOptionName, "-shaderOpt", "--shader-option");
                    Logger.Log("Console", LogLevels.Warning, warningMsg);
                }

                if (parameters.UsingObsoleteNoShaderSource)
                {
                    string warningMsg = string.Format(Resources.ConsoleMsgReplaceOptionName, "-noShaderSrc", "--no-shader-source");
                    Logger.Log("Console", LogLevels.Warning, warningMsg);
                }

                if (parameters.UsingObsoleteVerboseMode)
                {
                    string warningMsg = string.Format(Resources.ConsoleMsgReplaceOptionName, "-d", "-v, --verbose");
                    Logger.Log("Console", LogLevels.Warning, warningMsg);
                }

                if (parameters.UsingObsoleteSpec)
                {
                    string warningMsg = string.Format(Resources.ConsoleMsgReplaceOptionName, "-spec", "-a, --api-type");
                    Logger.Log("Console", LogLevels.Warning, warningMsg);
                }

                if (parameters.UsingObsoletePlatform)
                {
                    var currentSpec = SpecManager.SpecDefinitions.FirstOrDefault(s => s.IsSpecified(parameters.ApiType));
                    if (currentSpec == null || currentSpec.BinaryHeader == "VFXB")
                    {
                        string warningMsg = string.Format(Resources.ConsoleMsgReplaceOptionName, "-p, --platform", "-a, --api-type");
                        Logger.Log("Console", LogLevels.Warning, warningMsg);
                    }
                }
            }
#endif

            return parameters;
        }

        /// <summary>
        /// コマンドラインプロセッサを作成します。
        /// </summary>
        /// <param name="workingData">作業データ</param>
        /// <param name="parameters">コマンドラインパラメータ</param>
        /// <returns>コマンドラインプロセッサを返します。</returns>
        private static List<CommandLineProcessorBase> CreateProcessors(CommandLineWorkingData workingData, CompatibleCommandLineParams parameters)
        {
            var processors = new List<CommandLineProcessorBase>();

            if (parameters.ProjectConfigFile != null)
            {
                var processor = new ProjectConfigProcessor(workingData, parameters.ProjectConfigFile);
                processors.Add(processor);
            }

            if (parameters.ShaderConvertOption != null)
            {
                var processor = new ShaderOptionProcessor(workingData, parameters.ShaderConvertOption);
                processors.Add(processor);
            }

            if (parameters.ConvertListFile != null)
            {
                var processor = new ConvertListProcessor(workingData, parameters.ConvertListFile);
                processors.Add(processor);
                workingData.Converter.ConvertListFilePath = parameters.ConvertListFile;
            }

            if (parameters.OutputBinaryFile != null)
            {
                var processor = new BinaryOutputProcessor(workingData, parameters.OutputBinaryFile);
                processors.Add(processor);

                workingData.Converter.OutputBinaryEnabled = true;
            }

            if (parameters.EsetHeaderFile != null)
            {
                var processor = new EsetHeaderProcessor(workingData, parameters.EsetHeaderFile);
                processors.Add(processor);

                workingData.Converter.OutputHeaderEnabled = true;
            }

            if (parameters.OutputResidentAssetListFile != null)
            {
                workingData.Converter.OutputResidentAssetListPath = parameters.OutputResidentAssetListFile;
            }

            if (parameters.InputResidentAssetListFile != null)
            {
                workingData.Converter.InputResidentAssetListPath = parameters.InputResidentAssetListFile;
            }

            if (parameters.ResidentTextureListFile != null)
            {
                workingData.Converter.ResidentTextureListPath = parameters.ResidentTextureListFile;
            }

            if (parameters.OutputShaderBinaryName != null)
            {
                workingData.Converter.OutputShaderBinaryEnabled = true;
                workingData.Converter.ShaderBinaryOutputFilePath = parameters.OutputShaderBinaryName;
            }

            if (parameters.OutputComputeShaderBinaryName != null)
            {
                workingData.Converter.OutputComputeShaderBinaryEnabled = true;
                workingData.Converter.ComputeShaderBinaryOutputFilePath = parameters.OutputComputeShaderBinaryName;
            }

            if (parameters.OutputTextureBinaryName != null)
            {
                workingData.Converter.OutputTextureBinaryEnabled = true;
                workingData.Converter.TextureBinaryOutputFilePath = parameters.OutputTextureBinaryName;
            }

            if (parameters.OutputG3dPrimitiveBinaryName != null)
            {
                workingData.Converter.OutputG3dPrimitiveBinaryEnabled = true;
                workingData.Converter.G3dPrimitiveBinaryOutputFilePath = parameters.OutputG3dPrimitiveBinaryName;
            }

            if (parameters.IncludeEmitterSetFilePathArray == true)
            {
                workingData.Converter.IncludeEmitterSetFilePathArray = true;
            }

            if (parameters.UpdateListFile != null)
            {
                var processor = new UpdateListFileProcessor(workingData, parameters.UpdateListFile);
                processors.Add(processor);
            }

            if (parameters.UpdateCE2ListFile != null)
            {
                var processor = new UpdateCE2ListFileProcessor(workingData, parameters.UpdateCE2ListFile);
                processors.Add(processor);
            }

            if (parameters.Udd2Convert != null)
            {
                var processor = new Udd2ConvertProcessor(workingData, parameters.Udd2Convert);
                processors.Add(processor);
            }

            if (parameters.OverwriteBinaryFileVersion != null)
            {
                try
                {
                    uint version = Convert.ToUInt32(parameters.OverwriteBinaryFileVersion, 16);
                    BinaryFileHeader.OverwriteVersion = version;
                }
                catch (Exception)
                {
                }
            }

            if (parameters.IsVerboseOutputMode == true)
            {
                IOConstants.DetailCommandLineOutputMode = true;
            }

            if (parameters.IsSilentMode == true)
            {
                AppData.ConsoleDisplay.IsSilentMode = true;
            }

            if (parameters.NoShaderSource == true)
            {
                var processor = new NoShaderSourceProcessor(workingData);
                processors.Add(processor);
            }

            if (parameters.ShaderCacheDirectory != null)
            {
                var processor = new ShaderCacheDirectoryProcessor(workingData, parameters.ShaderCacheDirectory);
                processors.Add(processor);
            }

            if (parameters.JobsNumber != 0)
            {
                var processor = new JobsNumberProcessor(workingData, parameters.JobsNumber);
                processors.Add(processor);
            }

            if (parameters.ApiType != null)
            {
                var processor = new ApiTypeProcessor(workingData, parameters.ApiType, parameters.CodeType, parameters.TileMode, parameters.TileOptimize, parameters.TileSizeThreshold);
                processors.Add(processor);

                workingData.ApiTypeOptionEnabled = true;
            }

            if (parameters.ProfileOutputPath != null)
            {
                var processor = new ProfileModeProcessor(workingData, parameters.ProfileOutputPath);
                processors.Add(processor);

                workingData.Converter.StaticProfileEnabled = true;
            }

            return processors;
        }
    }
}
