﻿/*---------------------------------------------------------------------------*
Project:  NintendoWare
File:     eft_Stripe.vsh

Copyright (C)2011-2013 Nintendo All rights reserved.

These coded instructions, statements, and computer programs contain proprietary
information of Nintendo and/or its licensed developers and are protected by
national and international copyright laws. They may not be disclosed to third
parties or copied or duplicated in any form, in whole or in part, without the
prior written consent of Nintendo.

The content herein is highly confidential and should be handled accordingly.
*---------------------------------------------------------------------------*/

// 定数バッファ
#define sysStripeRandom         sysEmitterPluginUniformBlockParam0      // ストライプ単位のランダム（xyzw）
#define sysStripeHeadAlpha      sysEmitterPluginUniformBlockParam1.x    // 先端α
#define sysStripeTailAlpha      sysEmitterPluginUniformBlockParam1.y    // 末端α
#define sysStripeHeadScale      sysEmitterPluginUniformBlockParam1.z    // 先端スケール
#define sysStripeTailScale      sysEmitterPluginUniformBlockParam1.w    // 末端スケール
#define sysStripeColor0         sysEmitterPluginUniformBlockParam2      // ストライプカラー0
#define sysStripeColor1         sysEmitterPluginUniformBlockParam3      // ストライプカラー1
#define sysStripeTime           sysEmitterPluginUniformBlockParam4.x    // ストライプ時間
#define sysStripePointSum       sysEmitterPluginUniformBlockParam4.y    // ストライプの履歴点の数
#define sysStripeCrossMode      sysEmitterPluginUniformBlockParam4.z    // クロスモード
#define sysStripeLife           sysEmitterPluginUniformBlockParam4.w    // 寿命
#ifndef _VFX_NSDK01215_COMPATIBLE
#define sysStripeHeadPos        sysEmitterPluginUniformBlockParam5.xyz  // ストライプの先端
#endif
#define sysOffsetParam          ubCenter.z                              // デプス、カメラオフセット距離
// アトリビュート
#define inSpinePosition         sysEmitterPluginAttr0.xyz               // 背骨の位置（xyz）
#define inWingLength            sysEmitterPluginAttr0.w                 // 展開する翼の長さ
#define inSpineDirection        sysEmitterPluginAttr1.xyz               // 背骨の方向（xyz）
#define inWingDirection         sysEmitterPluginAttr2.xyz               // ポリゴンの展開方向（翼の方向）
//#define inDummy               sysEmitterPluginAttr2.w                 //
#define inTexUvOffset0          sysEmitterPluginAttr3.x                 // テクスチャ座標0
#define inTexUvOffset1          sysEmitterPluginAttr3.y                 // テクスチャ座標1
#define inTexUvOffset2          sysEmitterPluginAttr3.z                 // テクスチャ座標2
#define inEmitterMatrixAxisY    sysEmitterPluginAttr4.xyz               // エミッタ行列Y軸成分（必要な時だけ値が入る）

#ifdef _USE_NN_VFX
#define sysStripePointID        ( sysEmitterPluginAttr1.w )
#else
#define sysStripePointID        ( gl_VertexID / 2 )
#endif

//---------------------------------------------------
// Zバッファの奥行きを取得して、指定のz値が隠れているならを、見えているなら1を返す
//---------------------------------------------------
#ifndef USE_USR_CHECK_DEPTH_BUFFER
float CheckDepthBuffer( vec2 ofs, vec4 outDepthTexCoordProj, float pz )
{
    float tz = GetDepthValueFromTexture( outDepthTexCoordProj + vec4( ofs, 0, 0 ) );
    float dis = tz - pz;
    return step( 0.0, dis );
}
#endif

//---------------------------------------------------
// 遮蔽チェック
//---------------------------------------------------
#ifndef USE_USR_VERTEX_MASKING_CHECK
float CalcMaskingCheck( vec4 ptclFragCoordCenter, float sysPtclCenterDepth )
{
    float alphaRate = 1.0;

    vec4 outDepthTexCoordProj = ptclFragCoordCenter;

    // ピクセルZを取得する
    float cpz = sysPtclCenterDepth;

    // 外側と内側のサンプリング点のデプスバッファとの前後関係を取得
    float r = alphaFunc0.y;
    float out_sum = 0.0;
    float in_sum = 0.0;

    // 外側の点
    out_sum += 2.0 * CheckDepthBuffer( vec2( -0.8, -0.7 )*1.0*r, outDepthTexCoordProj, cpz );
    out_sum += 1.0 * CheckDepthBuffer( vec2( -1, 0 )*1.0*r, outDepthTexCoordProj, cpz );
    out_sum += 2.0 * CheckDepthBuffer( vec2( 0.7, 0.8 )*1.0*r, outDepthTexCoordProj, cpz );
    out_sum += 1.0 * CheckDepthBuffer( vec2( 0, -1 )*1.0*r, outDepthTexCoordProj, cpz );

    out_sum += 1.0 * CheckDepthBuffer( vec2( 0, 1 )*1.0*r, outDepthTexCoordProj, cpz );
    out_sum += 2.0 * CheckDepthBuffer( vec2( 0.7, -0.8 )*1.0*r, outDepthTexCoordProj, cpz );
    out_sum += 1.0 * CheckDepthBuffer( vec2( 1, 0 )*1.0*r, outDepthTexCoordProj, cpz );
    out_sum += 2.0 * CheckDepthBuffer( vec2( 0.8, 0.7 )*1.0*r, outDepthTexCoordProj, cpz );

    out_sum += 2.0 * CheckDepthBuffer( vec2( -0.8, -0.7 )*0.7*r, outDepthTexCoordProj, cpz );
    out_sum += 1.0 * CheckDepthBuffer( vec2( -1, 0 )*0.7*r, outDepthTexCoordProj, cpz );
    out_sum += 2.0 * CheckDepthBuffer( vec2( 0.7, 0.8 )*0.7*r, outDepthTexCoordProj, cpz );
    out_sum += 1.0 * CheckDepthBuffer( vec2( 0, -1 )*0.7*r, outDepthTexCoordProj, cpz );

    out_sum += 1.0 * CheckDepthBuffer( vec2( 0, 1 )*0.7*r, outDepthTexCoordProj, cpz );
    out_sum += 2.0 * CheckDepthBuffer( vec2( 0.7, -0.8 )*0.7*r, outDepthTexCoordProj, cpz );
    out_sum += 1.0 * CheckDepthBuffer( vec2( 1, 0 )*0.7*r, outDepthTexCoordProj, cpz );
    out_sum += 2.0 * CheckDepthBuffer( vec2( 0.8, 0.7 )*0.7*r, outDepthTexCoordProj, cpz );

    out_sum += 2.0 * CheckDepthBuffer( vec2( -0.8, -0.7 )*0.4*r, outDepthTexCoordProj, cpz );
    out_sum += 1.0 * CheckDepthBuffer( vec2( -1, 0 )*0.4*r, outDepthTexCoordProj, cpz );
    out_sum += 2.0 * CheckDepthBuffer( vec2( 0.7, 0.8 )*0.4*r, outDepthTexCoordProj, cpz );
    out_sum += 1.0 * CheckDepthBuffer( vec2( 0, -1 )*0.4*r, outDepthTexCoordProj, cpz );

    out_sum += 1.0 * CheckDepthBuffer( vec2( 0, 1 )*0.4*r, outDepthTexCoordProj, cpz );
    out_sum += 2.0 * CheckDepthBuffer( vec2( 0.7, -0.8 )*0.4*r, outDepthTexCoordProj, cpz );
    out_sum += 1.0 * CheckDepthBuffer( vec2( 1, 0 )*0.4*r, outDepthTexCoordProj, cpz );
    out_sum += 2.0 * CheckDepthBuffer( vec2( 0.8, 0.7 )*0.4*r, outDepthTexCoordProj, cpz );

    // 内側の点
    in_sum += 3.0 * CheckDepthBuffer( vec2( -0.2, -0.4 )*1.2*r, outDepthTexCoordProj, cpz );
    in_sum += 3.0 * CheckDepthBuffer( vec2( 0.2, 0.4 )*1.2*r, outDepthTexCoordProj, cpz );
    in_sum += 3.0 * CheckDepthBuffer( vec2( 0.4, -0.2 )*1.2*r, outDepthTexCoordProj, cpz );
    in_sum += 3.0 * CheckDepthBuffer( vec2( -0.4, 0.2 )*1.2*r, outDepthTexCoordProj, cpz );

    in_sum += 3.0 * CheckDepthBuffer( vec2( -0.1, -0.2 )*1.0*r, outDepthTexCoordProj, cpz );
    in_sum += 3.0 * CheckDepthBuffer( vec2( 0.1, 0.2 )*1.0*r, outDepthTexCoordProj, cpz );
    in_sum += 3.0 * CheckDepthBuffer( vec2( 0.2, -0.1 )*1.0*r, outDepthTexCoordProj, cpz );
    in_sum += 3.0 * CheckDepthBuffer( vec2( -0.2, 0.1 )*1.0*r, outDepthTexCoordProj, cpz );

    in_sum += 5.0 * CheckDepthBuffer( vec2( 0, 0 )*r, outDepthTexCoordProj, cpz );

    // 外側の遮蔽率と、内側の遮蔽率を個別に乗算して減衰(内側が全点隠れたら消えるようにするため)
    alphaRate *= clamp( out_sum / 35.0, 0.0, 1.0 );
    alphaRate *= clamp( in_sum / 7.0, 0.0, 1.0 );

    return alphaRate;
}
#endif

//--------------------------------------------------------------
// 内部処理用の関数定義
void CalcVertexPosition( float relativePos );
void CalcVertexColor();
void CalcTextureUV();
void _CalcTextureUV_SphereMap();
void _CalcTextureUV_InvertedRandom();
void _CalcTextureUV_PatternAnime();
void _CalcTextureUV_SRTAnime();

//--------------------------------------------------------------
// グローバル変数（テクスチャ計算用）
vec4 g_InUv0;               // テクスチャ0: UV座標
vec4 g_InUv1;               // テクスチャ1: UV座標
vec4 g_InUv2;               // テクスチャ2: UV座標
int  g_PtclTexturePtn0;     // テクスチャパターン 0
int  g_PtclTexturePtn1;     // テクスチャパターン 1
int  g_PtclTexturePtn2;     // テクスチャパターン 2

//--------------------------------------------------------------
// スーパーストライプのメイン関数
//--------------------------------------------------------------
#define USE_USR_VERTEX_MAIN             // デフォルトのメイン関数を上書きします。
void main()
{
    float relativePos = sysStripePointID / ( sysStripePointSum - 1 );

    //--------------------------------------------------------------
    // 全体を通して使う変数
    sysPtclTime = sysStripeTime;        // パーティクル時間

    // 履歴点をPtcl代表点として記録
    sysPtclWorldPos.xyz = inSpinePosition;

    //--------------------------------------------------------------
    // 頂点位置の決定
    CalcVertexPosition( relativePos );

    //--------------------------------------------------------------
    // ★頂点位置決定後のカスタムシェーダの呼び出し
    AdjustmentWorldPosition();

    //--------------------------------------------------------------
    // ここからスクリーン座標系に変換
    {
        sysWorldPositionVary = gl_Position.xyz;     // ワールド座標を保存
        gl_Position *= sysViewProjectionMatrix;      // スクリーン座標系へ変換

        // デプスオフセットを追加
        if ( CHECK_BIT_FLAG( BIT_FLAG_OFFSET_TYPE_DEPTH ) )
        {
            gl_Position.z -= sysOffsetParam * ( 1 - gl_Position.z / gl_Position.w );
        }
    }

    // 視線ベクトル
    sysEyeVecVary.xyz = sysWorldPositionVary.xyz - eyePos.xyz;

    //--------------------------------------------------------------
    // 頂点カラーの決定
    CalcVertexColor();
    sysPrimitiveColorVary = vec4( 1 );
    //--------------------------------------------------------------
    // テクスチャ: パターン・座標アニメーション処理
    CalcTextureUV();

    //--------------------------------------------------------------
    // ピクセルシェーダ用変数の設定
    sysFragCoordVary = sysTexCoordTransformMatrix * gl_Position;  // ピクセルシェーダでの、この頂点のスクリーン座標

    // パーティクルの中心のデプス値を計算
    sysPtclFragCoordCenter = sysTexCoordTransformMatrix * ( vec4( inSpinePosition, 1 ) * sysViewProjectionMatrix );
    sysPtclVertexDepthValueCenter = GetDepthValue( sysPtclFragCoordCenter );

    sysPtclAlphaRate = 1.0;
    float nearDistAlphaRate = 1.0;
    float farDistAlphaRate = 1.0;

#ifndef _VFX_NSDK01215_COMPATIBLE
    //--------------------------------------------------------------
    // 遮蔽チェック
#ifdef _MASKING_CHECK
    vec3 stripeHeadPos = sysStripeHeadPos;
    vec4 stripeHeadFragCoordCenter = sysTexCoordTransformMatrix * ( vec4( stripeHeadPos, 1 ) * sysViewProjectionMatrix );
    float stripeHeadDepthValue = GetDepthValue( stripeHeadFragCoordCenter );
    float maskAlphaRate = CalcMaskingCheck( stripeHeadFragCoordCenter, stripeHeadDepthValue );
    sysPtclAlphaRate *= maskAlphaRate;
#endif
#endif

    //--------------------------------------------------------------
    // 近距離アルファ
#ifdef _NEAR_DIST_ALPHA
#define nearAlphaMin    alphaFunc1.x
#define nearAlphaMax    alphaFunc1.y
    nearDistAlphaRate = ( sysPtclVertexDepthValueCenter - nearAlphaMin ) / ( nearAlphaMax - nearAlphaMin );
    nearDistAlphaRate = clamp( nearDistAlphaRate, 0.0, 1.0 );
    sysPtclAlphaRate *= nearDistAlphaRate;
#endif

    //--------------------------------------------------------------
    // 遠距離アルファ
#ifdef _FAR_DIST_ALPHA
#define farAlphaMin    alphaFunc1.z
#define farAlphaMax    alphaFunc1.w
    farDistAlphaRate = ( sysPtclVertexDepthValueCenter - farAlphaMin ) / ( farAlphaMax - farAlphaMin );
    farDistAlphaRate = clamp( farDistAlphaRate, 0, 1 );
    farDistAlphaRate = 1 - farDistAlphaRate;
    sysPtclAlphaRate *= farDistAlphaRate;
#endif

    //--------------------------------------------------------------
    // フレネル処理
#ifdef _VERTEX_FRESNEL_ALPHA
    float fresnel = CalcFresnelRate( sysEyeVecVary.xyz, sysWorldNormalVary );
    sysPtclAlphaRate *= fresnel;
#endif

    //--------------------------------------------------------------
    // マスターアルファ値を計算
    sysMasterAlpha = relativePos * ( sysStripeTailAlpha - sysStripeHeadAlpha ) + sysStripeHeadAlpha;
    sysMasterAlpha *= sysPtclAlphaRate * sysEmitterFadeAlpha;

    //--------------------------------------------------------------
    // 寿命係数
    sysLifeRatio = sysStripeTime / sysStripeLife;

    //--------------------------------------------------------------
    // シェーダアニメーションを計算
#if defined( _SHADER_ANIM ) || defined( _VFX_COMBINER_SHADER_ANIM_ENABLED )
    vec3 vShaderAnim = Calculate8KeyAnimShader( shaderAnimKeyNum, sysStripeTime, 0, 0, sysStripeLife, sysStripeRandom.x ).xyz;
    sysShaderAnim  = vShaderAnim.x;
    sysShaderAnimY = vShaderAnim.y;
    sysShaderAnimZ = vShaderAnim.z;
#else
    sysShaderAnim  = shaderAnim.v[0].x;
    sysShaderAnimY = shaderAnim.v[0].y;
    sysShaderAnimZ = shaderAnim.v[0].z;
#endif

    // 乱数をフラグメントシェーダに送る
    sysParticleRandom = sysStripeRandom;

    //--------------------------------------------------------------
    // ★最終カスタムシェーダの呼び出し
    FinalAdjustmentVertexProcess();

#ifdef _TARGET_VK
    gl_Position.y = -gl_Position.y;      // For Vulkan.
    #ifdef _DEPTH_MODE_NEAR_IS_MINUS_W
    gl_Position.z = (gl_Position.z + gl_Position.w) / 2.0;
    #endif
#endif
}

//--------------------------------------------------------------
// 頂点座標の計算
//--------------------------------------------------------------
void CalcVertexPosition( float relativePos )
{
    // カメラオフセット
    vec3 center;
    center.x = sysEmitterMatrix[0].w;
    center.y = sysEmitterMatrix[1].w;
    center.z = sysEmitterMatrix[2].w;
    vec3 camOffset = CalculateCameraOffset( center );

    // 先端／末端スケール
    float scale = relativePos * ( sysStripeTailScale - sysStripeHeadScale ) + sysStripeHeadScale;

    //--------------------------------
    // ビルボードストライプ
    //--------------------------------
#ifdef _STRIPE_BILLBOARD
    vec3 spineDir = normalize( inSpineDirection );
    vec3 lookDir = normalize( eyePos.xyz - inSpinePosition );       // 視線方向（物体→カメラの向き）
#ifdef _VFX_NSDK01215_COMPATIBLE
    vec3 wingDir = cross( normalize( inWingDirection ), lookDir );
#else
    vec3 wingDir = cross( lookDir, normalize( inWingDirection ) );
#endif
    wingDir = normalize( wingDir );
    if( sysStripeCrossMode == 1 )
    {
        // MEMO: この計算は直前の wingDir の計算でベクトルが反転した結果が連動するのでそのまま。
        wingDir = normalize( cross( wingDir, spineDir ) );
    }
#ifdef _VFX_NSDK01215_COMPATIBLE
    sysWorldNormalVary = cross( wingDir, spineDir );     // 法線を設定
#else
    // 法線計算も反転させる必要があった
    sysWorldNormalVary = cross( spineDir, wingDir );     // 法線を設定
#endif
#endif

    //--------------------------------
    // エミッタマトリクス
    //--------------------------------
#ifdef _STRIPE_EMITTER_MATRIX
    vec3 spineDir = normalize( inSpineDirection );
    vec3 wingDir = normalize( cross( inEmitterMatrixAxisY, spineDir ) );
    if( sysStripeCrossMode == 1 )
    {
        wingDir = normalize( cross( wingDir, spineDir ) );
    }
    sysWorldNormalVary = normalize( cross( spineDir, wingDir ) );    // 法線を設定
#endif

    //--------------------------------
    // エミッタ上下
    //--------------------------------
#ifdef _STRIPE_EMITTER_UPDOWN
    vec3 spineDir = normalize( inSpineDirection );
    vec3 wingDir = inEmitterMatrixAxisY;
    if( sysStripeCrossMode == 1 )
    {
        wingDir = normalize( cross( wingDir, spineDir ) );
    }
    sysWorldNormalVary = normalize( cross( spineDir, wingDir ) );    // 法線を設定
#endif

    //--------------------------------
    // リボン
    //--------------------------------
#ifdef _STRIPE_RIBBON
#if 0
    // 検証用: wing 方向をそのまま出すシェーダ
    vec3 wingDir = normalize( inWingDirection );
    if( sysStripeCrossMode == 1 )
    {
        wingDir = normalize( cross( wingDir, inSpineDirection ) );
    }
    sysWorldNormalVary = normalize( cross( wingDir, inSpineDirection ) );   // 法線を設定
#else
    // 代表的な3方向について、どの方向を強く出すべきかを係数化した上で、合成して正規化するアプローチ
    vec3 vDir    = normalize( inSpineDirection.xyz );
    vec3 vEye    = normalize( sysEyeVector.xyz );
    vec3 vWing   = normalize( inWingDirection.xyz );
    vec3 vBld    = normalize( cross( vDir, vEye ) );
    vec3 vNormal = normalize( cross( vWing, vDir ) );

    float d1 = abs( dot( vNormal, vBld ) );     // 元々の法線方向と、視線から見たビルボード方向の類似度
    float d2 = abs( dot( vEye, vDir ) );        // 視線方向と、背骨の方向の類似度
    {
        float pi = 3.141592;
        float pi_2 = pi * 0.5;
        d1 = ( acos( d1 ) / pi_2 );
        d2 = ( acos( d2 ) / pi_2 );
    }
    // 元々の wing 方向か、その法線方向かどちらに伸ばすべきかの多数決方式
    vec3 wingDir = normalize( ( 2.0 - d2 - d1 ) * vNormal + ( d2 + d1 ) * vWing );

    if( sysStripeCrossMode == 1 )
    {
        wingDir = normalize( cross( wingDir, vDir ) );
    }
    sysWorldNormalVary = normalize( cross( wingDir, vDir ) );   // 法線を決定
#endif
#endif

    // 頂点位置を決定
    gl_Position.xyz = inSpinePosition - ( wingDir * inWingLength * scale * 0.5 );
    gl_Position.w   = 1;

#ifdef _STRIPE_FOLLOW_EMITTER_MATRIX
    // エミッタに完全追従の場合、ローカル系からワールド系に変換
    gl_Position.xyz = ( vec4( gl_Position.xyz, 1 ) * sysEmitterMatrix ).xyz;
#ifdef _VFX_NSDK01215_COMPATIBLE
    sysWorldNormalVary.xyz = ( vec4( sysWorldNormalVary, 1 ) * sysEmitterMatrix ).xyz;
#else
    // MEMO: 法線なので、平行移動成分を加味しないように修正。
    sysWorldNormalVary.xyz = ( vec4( sysWorldNormalVary, 0 ) * sysEmitterMatrix ).xyz;
#endif
#endif

    // 接線・従法線の計算
    sysWorldTangentVary = normalize( wingDir );
    sysWorldBinormalVary = normalize( cross( sysWorldTangentVary, sysWorldNormalVary ) );

    // カメラオフセット
    gl_Position.xyz += camOffset;
}

//--------------------------------------------------------------
// 頂点頂点カラーの計算
//--------------------------------------------------------------
void CalcVertexColor()
{
    sysColor0Vary = sysStripeColor0 * sysEmitterColor0;    // カラー0
    sysColor1Vary = sysStripeColor1 * sysEmitterColor1;    // カラー1
}

//--------------------------------------------------------------
// テクスチャUV座標の計算
//--------------------------------------------------------------
void CalcTextureUV()
{
    //--------------------------------------------------------------
    // 入力となるテクスチャ座標を決定

    float u = 1.0;
    if( inWingLength > 0.0 ) { u = 0.0; }
    vec2 defuv0 = vec2( u, -inTexUvOffset0 );
    vec2 defuv1 = vec2( u, -inTexUvOffset1 );
    vec2 defuv2 = vec2( u, -inTexUvOffset2 );

    g_InUv0   = vec4( defuv0, 0, 0 );
    g_InUv1   = vec4( defuv1, 0, 0 );
    g_InUv2   = vec4( defuv2, 0, 0 );

#if 0
    //--------------------------------------------------------------
    // スフィアマップ
    _CalcTextureUV_SphereMap();
#endif

    //--------------------------------------------------------------
    // 反転ランダム
    _CalcTextureUV_InvertedRandom();

    //--------------------------------------------------------------
    // テクスチャパターンアニメーション
    _CalcTextureUV_PatternAnime();

    //--------------------------------------------------------------
    // テクスチャSRTアニメーション
    _CalcTextureUV_SRTAnime();

    //--------------------------------------------------------------
    // 最終結果のUV座標をピクセルシェーダへ渡す
    sysTexCoord01Vary.x = g_InUv0.x;
    sysTexCoord01Vary.y = g_InUv0.y;
    sysTexCoord01Vary.z = g_InUv1.x;
    sysTexCoord01Vary.w = g_InUv1.y;
    sysTexCoord2Vary.x  = g_InUv2.x;
    sysTexCoord2Vary.y  = g_InUv2.y;
}

//--------------------------------------------------------------
// テクスチャ: スフィアマップの計算
//--------------------------------------------------------------
void _CalcTextureUV_SphereMap()
{
    vec2 sphereMapTexCoord;
    vec3 viewPosition = normalize( ( vec4( sysWorldPositionVary, 1 ) * sysViewMatrix ).xyz );
    vec3 viewNormal   = normalize( ( vec4( sysWorldNormalVary,   0 ) * sysViewMatrix ).xyz );

    viewNormal.z = clamp( viewNormal.z, 0.2, 1.0 );     // 暫定処理（zの負数を回避）

    vec3 r = reflect( viewPosition, viewNormal );
    r.z = r.z + 1.0;
    float m = 2.0 * sqrt( dot(r, r) );
    sphereMapTexCoord.x =  (r.x / m) + 0.5;
    sphereMapTexCoord.y = -(r.y / m) + 0.5;

#ifdef _TEX_0_SPHERE_MAP
    g_InUv0.xy = sphereMapTexCoord;
#endif
#ifdef _TEX_1_SPHERE_MAP
    g_InUv1.xy = sphereMapTexCoord;
#endif
#ifdef _TEX_2_SPHERE_MAP
    g_InUv2.xy = sphereMapTexCoord;
#endif
}

//--------------------------------------------------------------
// テクスチャ: 反転ランダムの計算
//--------------------------------------------------------------
void _CalcTextureUV_InvertedRandom()
{
#ifdef _PARTICLE
    if ( CHECK_BIT_FLAG( BIT_FLAG_TEX0_INV_RANDOM_U ) ) { if ( sysStripeRandom.x > 0.5 ) g_InUv0.x = -g_InUv0.x; }
    if ( CHECK_BIT_FLAG( BIT_FLAG_TEX0_INV_RANDOM_V ) ) { if ( sysStripeRandom.y > 0.5 ) g_InUv0.y = -g_InUv0.y; }
    if ( CHECK_BIT_FLAG( BIT_FLAG_TEX1_INV_RANDOM_U ) ) { if ( sysStripeRandom.z > 0.5 ) g_InUv1.x = -g_InUv1.x; }
    if ( CHECK_BIT_FLAG( BIT_FLAG_TEX1_INV_RANDOM_V ) ) { if ( sysStripeRandom.w > 0.5 ) g_InUv1.y = -g_InUv1.y; }
    if ( CHECK_BIT_FLAG( BIT_FLAG_TEX2_INV_RANDOM_U ) ) { if ( sysStripeRandom.y > 0.5 ) g_InUv2.x = -g_InUv2.x; }
    if ( CHECK_BIT_FLAG( BIT_FLAG_TEX2_INV_RANDOM_V ) ) { if ( sysStripeRandom.z > 0.5 ) g_InUv2.y = -g_InUv2.y; }
#endif
#ifdef _PRIMITIVE
    if ( CHECK_BIT_FLAG( BIT_FLAG_TEX0_INV_RANDOM_U ) ) { if ( sysStripeRandom.x > 0.5 ) g_InUv0.x = 1.0 - g_InUv0.x; }
    if ( CHECK_BIT_FLAG( BIT_FLAG_TEX0_INV_RANDOM_V ) ) { if ( sysStripeRandom.y > 0.5 ) g_InUv0.y = 1.0 - g_InUv0.y; }
    if ( CHECK_BIT_FLAG( BIT_FLAG_TEX1_INV_RANDOM_U ) ) { if ( sysStripeRandom.z > 0.5 ) g_InUv1.x = 1.0 - g_InUv1.x; }
    if ( CHECK_BIT_FLAG( BIT_FLAG_TEX1_INV_RANDOM_V ) ) { if ( sysStripeRandom.w > 0.5 ) g_InUv1.y = 1.0 - g_InUv1.y; }
    if ( CHECK_BIT_FLAG( BIT_FLAG_TEX2_INV_RANDOM_U ) ) { if ( sysStripeRandom.y > 0.5 ) g_InUv2.x = 1.0 - g_InUv2.x; }
    if ( CHECK_BIT_FLAG( BIT_FLAG_TEX2_INV_RANDOM_V ) ) { if ( sysStripeRandom.z > 0.5 ) g_InUv2.y = 1.0 - g_InUv2.y; }
#endif
}

//--------------------------------------------------------------
// テクスチャ: パターンアニメーションの計算
//--------------------------------------------------------------
void _CalcTextureUV_PatternAnime()
{
    // テクスチャ パターンアニメーション コマ値を計算
    g_PtclTexturePtn0 = g_PtclTexturePtn1 = g_PtclTexturePtn2 = 0;

    float life = 100;


#ifdef _TEX_0_PATTERN_ANIM
    g_PtclTexturePtn0 = CalculateTexPtnAnim0( sysPtclTime, life, sysStripeRandom.x );
#endif
#ifdef _TEX_1_PATTERN_ANIM
    g_PtclTexturePtn1 = CalculateTexPtnAnim1( sysPtclTime, life, sysStripeRandom.y );
#endif
#ifdef _TEX_2_PATTERN_ANIM
    g_PtclTexturePtn2 = CalculateTexPtnAnim2( sysPtclTime, life, sysStripeRandom.z );
#endif
}

//--------------------------------------------------------------
// テクスチャ: SRTアニメーションの計算
//--------------------------------------------------------------
void _CalcTextureUV_SRTAnime()
{
    ivec3 texShareRandom;   // 乱数を共有するかしないかの設定
    texShareRandom.x = 0;
    texShareRandom.y = 0;
    texShareRandom.z = 0;

    if( CHECK_BIT_FLAG_1( BIT_FLAG_TEX_ANIM_FIXED_RANDOM ) )
    {
        texShareRandom.y = 1;
        texShareRandom.z = 2;
    }


    vec2 animDir = vec2( -1, 1 );   // テクスチャのスクロール方向補正
    vec2 texBaseOffset = vec2( -0.5, 0.5 );

#if defined( _TEX_0_SHIFT_ANIM ) || defined( _TEX_0_PATTERN_ANIM ) || defined( _TEX_0_SHIFT_ANIM_ROTATE )
    g_InUv0 = CalculateTextureShiftAnimWithRotate( g_InUv0.xy, texShiftAnim0, sysPtclTime, g_PtclTexturePtn0, texShareRandom.x, sysStripeRandom );
#else
    g_InUv0.x = g_InUv0.x *  texShiftAnim0.pm4.x * ( 1 / texShiftAnim0.pm4.z );
    g_InUv0.y = g_InUv0.y * -texShiftAnim0.pm4.y * ( 1 / texShiftAnim0.pm4.w );
#endif

#if defined( _TEX_1_SHIFT_ANIM ) || defined( _TEX_1_PATTERN_ANIM ) || defined( _TEX_1_SHIFT_ANIM_ROTATE )
    g_InUv1 = CalculateTextureShiftAnimWithRotate( g_InUv1.xy, texShiftAnim1, sysPtclTime, g_PtclTexturePtn1, texShareRandom.y, sysStripeRandom );
#else
    g_InUv1.x = g_InUv1.x *  texShiftAnim1.pm4.x * ( 1 / texShiftAnim1.pm4.z );
    g_InUv1.y = g_InUv1.y * -texShiftAnim1.pm4.y * ( 1 / texShiftAnim1.pm4.w );
#endif

#if defined( _TEX_2_SHIFT_ANIM ) || defined( _TEX_2_PATTERN_ANIM ) || defined( _TEX_2_SHIFT_ANIM_ROTATE )
    g_InUv2 = CalculateTextureShiftAnimWithRotate( g_InUv2.xy, texShiftAnim2, sysPtclTime, g_PtclTexturePtn2, texShareRandom.z, sysStripeRandom );
#else
    g_InUv2.x = g_InUv2.x *  texShiftAnim2.pm4.x * ( 1 / texShiftAnim2.pm4.z );
    g_InUv2.y = g_InUv2.y * -texShiftAnim2.pm4.y * ( 1 / texShiftAnim2.pm4.w );
#endif
}
