﻿using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Data;
using System.Windows.Documents;
using System.Windows.Input;
using System.Windows.Media;
using System.Windows.Media.Imaging;
using System.Windows.Navigation;
using System.Windows.Shapes;

namespace VfxCombinerNodeGen
{
    /// <summary>
    /// 返り値型の enum. そのまま使うとコード上、予約語に引っかかるので別名で定義して管理。
    /// </summary>
    public enum ValueType
    {
        Float = 0,  // float
        Vec2 = 1,   // vec2
        Vec3 = 2,   // vec3
        Vec4 = 3,   // vec4
    };

    /// <summary>
    /// 実際の型名を文字列として取り出すための拡張
    /// </summary>
    public static class ValueTypeExt
    {
        public static string ToStringExt(this ValueType path)
        {
            switch (path)
            {
                case ValueType.Float:   return "float";
                case ValueType.Vec2:    return "vec2";
                case ValueType.Vec3:    return "vec3";
                case ValueType.Vec4:    return "vec4";
                default: throw new ArgumentOutOfRangeException("e");
            }
        }
    };

    /// <summary>
    /// MainWindow.xaml の相互作用ロジック
    /// </summary>
    public partial class MainWindow : Window
    {
        /// <summary>
        /// ノードの入力の定義
        /// </summary>
        public class InputDesc
        {
            /// <summary>
            /// 入力の型
            /// </summary>
            public ValueType Type { get; set; }
            /// <summary>
            /// 表示名
            /// </summary>
            public string Name { get; set; }

            /// <summary>
            /// 初期値を入れるためのコンストラクタ
            /// </summary>
            public InputDesc()
            {
                Type = ValueType.Float;
                Name = "value";
            }
        };

        /// <summary>
        /// ノードの出力の定義
        /// </summary>
        public class OutputDesc
        {
            /// <summary>
            /// 出力の型
            /// </summary>
            public ValueType Type { get; set; }
            /// <summary>
            /// 表示名
            /// </summary>
            public string Name { get; set; }
            /// <summary>
            /// 出力のマスク指定。基本的に "return" で返り値がそのまま渡される。
            /// 例えば、関数の返り値が vec4 のところで "return.xyz" のように記述すると、
            /// 返り値の xyz 要素で作られた vec3 を取り出すことができる。
            /// これを応用することで、
            /// ・返り値の一部分を取り出す（RGBのR要素だけ取り出す）
            /// ・返り値のチャンネル数別に出す（return.x / return.xxx で、同じ値の float / vec3 を両方出せる）
            /// ・（※処理にロスがあるが）複数の計算結果のマスク
            ///   最大 vec4 まで計算結果を返り値として持たせ、return.* で必要なものだけマスクして渡す
            /// といったトリックが可能。
            /// </summary>
            public string Return { get; set; }

            /// <summary>
            /// 初期値を入れるためのコンストラクタ
            /// </summary>
            public OutputDesc()
            {
                Type = ValueType.Float;
                Name = "value";
                Return = "return";
            }
        };

        private string m_Guid;  // 今使っている GUID

        /// <summary>
        /// GUID を生成。
        /// </summary>
        private string GenGuid()
        {
            // 形式は GUIDGEN.exe の「レジストリ形式」の全部大文字版。
            // 例: {2FE63157-FDAB-4C5F-AA82-A88FCDF9F384}
            return Guid.NewGuid().ToString("B").ToUpper();
        }

        /// <summary>
        /// UI の内容を初期化して反映。
        /// </summary>
        private void ResetAllForms()
        {
            // Guid 再設定
            m_Guid = GenGuid();

            // DataGrid 初期化
            var inputData = new ObservableCollection<InputDesc>();
            var outputData = new ObservableCollection<OutputDesc>();

            // MEMO: Output には最初から一つ項目があって良い
            outputData.Add(new OutputDesc());

            DataGrid_Input.ItemsSource = inputData;
            DataGrid_Output.ItemsSource = outputData;
            DataGrid_Input.ColumnWidth = DataGridLength.Auto;
            DataGrid_Output.ColumnWidth = DataGridLength.Auto;

            // コンボボックスの選択項目を初期化
            ComboBox_ReturnType.SelectedIndex = 0;

            // テンプレ埋め
            TextBox_Tag.Text         = "Test";
            TextBox_Description.Text = "ノードの説明";
            TextBox_NodeName.Text    = "NewNode1";
            TextBox_ShaderCode.Text  = "  " + "return 0.0;";

            // UI を更新
            UpdateGeneratedCode();
        }

        /// <summary>
        /// ノード定義を生成して UI に反映させる。
        /// </summary>
        string UpdateGeneratedCode()
        {
            /*-------------------------------------------------------------------------------
                MEMO: もうそれほど仕様が変わる部分でもないので、
                      ノード定義の記述を単なるテキストとして、テンプレートを埋める形で生成させています。
                      データバインドを使うほどでもないので、UI に更新がある度にその場で全更新をさせています。
            -------------------------------------------------------------------------------*/

            string tag          = TextBox_Tag.Text;
            string funcId       = "Get" + TextBox_NodeName.Text;
            string returnType   = ((ValueType)ComboBox_ReturnType.SelectedIndex).ToStringExt();
            string desc         = TextBox_Description.Text;
            string guid         = m_Guid;
            string nodeName     = TextBox_NodeName.Text;
            string shaderCode   = TextBox_ShaderCode.Text;

            // 不要な部分をトリム
            shaderCode = shaderCode.TrimStart('\r', '\n').TrimEnd('\r', '\n', ' ');

            // ここデータグリッド見る
            string funcInputDesc = "";  // 関数定義の引数
            string ioDesc        = "";  // xml タグ
            string argDesc       = "";  // 関数の引数

            // 入力線
            var input = DataGrid_Input.ItemsSource as ObservableCollection<InputDesc>;
            for (int i = 0; i < input.Count; ++i)
            {
                string name = input[i].Name;
                string type = input[i].Type.ToStringExt();

                // 関数の入力情報
                funcInputDesc += string.Format(
                    "///   <in name=\"{0}\" type=\"{1}\" />\n",
                    name, type);

                // 入力ノード情報
                ioDesc += string.Format(
                    "///   <in displayname=\"{0}\" type=\"{1}\" target=\"{2}\" />\n",
                    name, type, name);

                // シェーダの関数の引数部分の定義
                argDesc += string.Format("{0} {1}", type, name);
                if (i < input.Count - 1)
                {
                    // 最後でなければカンマを付ける
                    argDesc += ", ";
                }
            }

            // 出力線
            var outputs = DataGrid_Output.ItemsSource as ObservableCollection<OutputDesc>;
            for(int i = 0; i<outputs.Count; ++i)
            {
                string name = outputs[i].Name;
                string type = outputs[i].Type.ToStringExt();
                string target = outputs[i].Return;

                // 出力ノード情報
                ioDesc += string.Format(
                    "///   <out displayname=\"{0}\" type=\"{1}\" target=\"{2}\" />\n",
                    name, type, target);
            }


            /*-------------------------------------------------------------------------------
              ノード定義記述のテンプレを埋める。
              例：

              /// <tag name="Sample">
              ///   <func id="GetSampleNode" />
              /// </tag>
              /// <func id="GetSampleNode" name="GetSampleNode">
              ///   <return type="vec4" />
              ///   <in name="v2" type="vec2" />
              ///   <in name="f" type="float" />
              ///   <description>たとえばこんな感じです</description>
              /// </func>
              /// <block guid="{438C3192-36CD-43B6-B0B2-058666D2EC40}" displayname="SampleNode">
              ///   <func id="GetSampleNode"/>
              ///   <in displayname="v2" type="vec2" target="v2" />
              ///   <in displayname="f" type="float" target="f" />
              ///   <out displayname="result" type="vec4" target="return" />
              /// </block>
              vec4 GetSampleNode(vec2 v2, float f)
              {
                return vec4( v2.x, v2.y, f, 0.0 );
              }
            -------------------------------------------------------------------------------*/
            // コメント＋ XML 記述部分の埋め込み
            string code = string.Format((
                "/// <tag name=\"{0}\">\n" +
                "///   <func id=\"{1}\" />\n" +
                "/// </tag>\n" +
                "/// <func id=\"{1}\" name=\"{1}\">\n" +
                "///   <return type=\"{2}\" />\n" +
                "{7}" +
                "///   <description>{3}</description>\n" +
                "/// </func>\n" +
                "/// <block guid=\"{4}\" displayname=\"{5}\">\n" +
                "///   <func id=\"{1}\"/>\n" +
                "{6}" +
                "/// </block>\n"),
                tag, funcId, returnType, desc, guid, nodeName, ioDesc, funcInputDesc);

            // シェーダコードの埋め込み
            string shaderHeader = string.Format(
                "{0} {1}({2})\n" +
                "{{",
                returnType, funcId, argDesc);
            string shaderFooter =
                "}";
            code += string.Format(
                "{0}\n" +
                "{1}\n" +
                "{2}",
                shaderHeader, shaderCode, shaderFooter);

            // UI に反映。
            TextBox_ShaderCode_Header.Text = shaderHeader;
            TextBox_ShaderCode_Footer.Text = shaderFooter;
            TextBox_OutputCode.Text = code;

            // 一応作ったコードを返す。
            return code;
        }

        public MainWindow()
        {
            InitializeComponent();

            // 自分自身のバージョン情報をタイトルバーに出しておく。
            var versionInfo = System.Reflection.Assembly.GetExecutingAssembly().GetName().Version.ToString();
            this.Title += " " + versionInfo;

            // UI を初期化して反映。
            ResetAllForms();
        }

        #region -------- UI インタラクション 定義部分 --------

        private void Button_ResetGuid_Click(object sender, RoutedEventArgs e)
        {
            m_Guid = GenGuid();
            UpdateGeneratedCode();
        }

        private void Button_Reset_Click(object sender, RoutedEventArgs e)
        {
            ResetAllForms();
        }

        private void Button_CopyToClipboard_Click(object sender, RoutedEventArgs e)
        {
            // クリップボードにコピー
            Clipboard.SetDataObject(TextBox_OutputCode.Text);
        }

        private void Update_ShaderDefinition(object sender, TextChangedEventArgs e)
        {
            UpdateGeneratedCode();
        }

        private void DataGrid_CurrentCellChanged(object sender, EventArgs e)
        {
            UpdateGeneratedCode();
        }

        private void ComboBox_ReturnType_SelectionChanged(object sender, SelectionChangedEventArgs e)
        {
            UpdateGeneratedCode();
        }

        #endregion
    }
}
