﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;

namespace Workflow.Core
{
    public interface IWorkflowItemContainer<out TWorkflow, TValue> where TWorkflow : WorkflowItem<TValue>
    {
        TWorkflow WorkflowItem { get; }
    }

    /// <summary>
    /// A WorkflowItem is a WorkflowElement, connected to other WorkflowItem nodes through input and output plugs.
    /// </summary>
    /// <typeparam name="T"></typeparam>
    public class WorkflowItem<T> : WorkflowElement, IEquatable<WorkflowItem<T>>
    {
        public WorkflowItem()
        {
            InputPlugs = new InputPlug<T>[0];
            OutputPlugs = new OutputPlug<T>[0];
        }

        protected void SetOutputValue(int outputIndex, T value)
        {
            lock (outputPlugsSyncRoot)
                outputPlugs[outputIndex].SetValue(value);
        }

        protected void SetOutputValue(OutputPlug<T> outputPlug, T value)
        {
            if (outputPlug == null)
                throw new ArgumentNullException("outputPlug");

            SetOutputValue(outputPlug.Index, value);
        }

        protected void SetOutputValue(Guid outputIdentifier, T value)
        {
            lock (outputPlugsSyncRoot)
                SetOutputValue(outputPlugs.First(i => i.Identifier == outputIdentifier), value);
        }

        public bool IsInputPlugged(int inputIndex)
        {
            lock (inputPlugsSyncRoot)
                return inputPlugs[inputIndex].RemoteOutputPlug != null;
        }

        public T[] GetInputValues()
        {
            lock (inputPlugsSyncRoot)
                return inputPlugs.Select(i => i.Value).ToArray();
        }

        public T[] GetOutputValues()
        {
            lock (outputPlugsSyncRoot)
                return outputPlugs.Select(o => o.Value).ToArray();
        }

        public T[] PollInputValues()
        {
            bool isCompleted;
            return PollInputValues(out isCompleted);
        }

        public T[] PollInputValues(out bool isCompleted)
        {
            isCompleted = true;
            return PollInputValuesInternal(ref isCompleted);
        }

        private T[] PollInputValuesInternal(ref bool isCompleted)
        {
            lock (inputPlugsSyncRoot)
            {
                if (IsGraphInput == false)
                {
                    var remoteWorkflowItems = new HashSet<WorkflowItem<T>>(Utility.UniquelyIdentifiableEqualityComparer);

                    foreach (var inputPlug in inputPlugs)
                    {
                        if (inputPlug.RemoteOutputPlug == null) // input not plugged
                        {
                        //    isCompleted = false;
                        //    break;
                            continue;
                        }

                        remoteWorkflowItems.Add(inputPlug.RemoteOutputPlug.WorkflowItem);
                    }

                    foreach (var remoteWorkflowItem in remoteWorkflowItems)
                        remoteWorkflowItem.PollInputValuesInternal(ref isCompleted);
                }

                if (isCompleted)
                    OnInputsPolled();

                return inputPlugs.Select(i => i.Value).ToArray();
            }
        }

        public event EventHandler<InputPlugEventArgs<T>> InputPlugAdded;
        public event EventHandler<InputPlugEventArgs<T>> InputPlugRemoved;

        public event EventHandler<OutputPlugEventArgs<T>> OutputPlugAdded;
        public event EventHandler<OutputPlugEventArgs<T>> OutputPlugRemoved;

        public event EventHandler InputsPolled;
        public event EventHandler InputsInvalidated;

        public InputPlug<T>[] InputPlugs { get; private set; }
        public OutputPlug<T>[] OutputPlugs { get; private set; }

        private readonly object inputPlugsSyncRoot = new object();
        private readonly object outputPlugsSyncRoot = new object();

        private readonly List<InputPlug<T>> inputPlugs = new List<InputPlug<T>>();
        private readonly List<OutputPlug<T>> outputPlugs = new List<OutputPlug<T>>();

        protected internal virtual void OnInputsPolled()
        {
            var handler = InputsPolled;
            if (handler != null)
                handler(this, EventArgs.Empty);
        }

        protected internal virtual void OnInputsInvalidated()
        {
            var handler = InputsInvalidated;
            if (handler != null)
                handler(this, EventArgs.Empty);
        }

        private bool CheckAddPlug(PlugBase<T> plug)
        {
            if (plug.WorkflowItem != null)
            {
                if (plug.WorkflowItem.Equals(this))
                    return false;
                if (plug is OutputPlug<T>)
                    Utility.ThrowOutputAlreadyAttachedException(plug as OutputPlug<T>);
                else
                    Utility.ThrowInputAlreadyAttachedException(plug as InputPlug<T>);
            }

            return true;
        }

        #region InputPlugs Add and Remove

        public void AddInputPlug(InputPlug<T> inputPlug)
        {
            if (inputPlug == null)
                throw new ArgumentNullException("inputPlug");

            if (CheckAddPlug(inputPlug) == false)
                return;

            var added = false;
            lock (inputPlugsSyncRoot)
            {
                if (inputPlugs.Contains(inputPlug, Utility.UniquelyIdentifiableEqualityComparer) == false)
                {
                    var index = inputPlugs.Count;
                    inputPlugs.Add(inputPlug);
                    inputPlug.WorkflowItem = this;
                    inputPlug.Index = index;
                    InputPlugs = inputPlugs.ToArray();
                    added = true;
                }
            }

            if (added)
            {
                OnInputPlugAdded(new InputPlugEventArgs<T>(inputPlug));
                OnInputsInvalidated();
            }
        }

        public void RemoveInputPlug(InputPlug<T> inputPlug)
        {
            if (inputPlug == null)
                throw new ArgumentNullException("inputPlug");

            var removed = false;
            lock (inputPlugsSyncRoot)
            {
                if (inputPlugs.Contains(inputPlug, Utility.UniquelyIdentifiableEqualityComparer))
                {
                    inputPlugs.Remove(inputPlug);
                    inputPlug.WorkflowItem = null;
                    inputPlug.Index = -1;
                    ResetInputIndices();
                    InputPlugs = inputPlugs.ToArray();
                    removed = true;
                }
            }

            if (removed)
            {
                OnInputPlugRemoved(new InputPlugEventArgs<T>(inputPlug));
                OnInputsInvalidated();
            }
        }

        protected virtual void OnInputPlugAdded(InputPlugEventArgs<T> e)
        {
            var handler = InputPlugAdded;
            if (handler != null)
                handler(this, e);
        }

        protected virtual void OnInputPlugRemoved(InputPlugEventArgs<T> e)
        {
            var handler = InputPlugRemoved;
            if (handler != null)
                handler(this, e);
        }

        #endregion // InputPlugs Add and Remove

        #region OutputPlugs Add and Remove

        public void AddOutputPlug(OutputPlug<T> outputPlug)
        {
            if (outputPlug == null)
                throw new ArgumentNullException("outputPlug");

            if (CheckAddPlug(outputPlug) == false)
                return;

            var added = false;
            lock (outputPlugsSyncRoot)
            {
                if (outputPlugs.Contains(outputPlug, Utility.UniquelyIdentifiableEqualityComparer) == false)
                {
                    var index = outputPlugs.Count;
                    outputPlugs.Add(outputPlug);
                    outputPlug.WorkflowItem = this;
                    outputPlug.Index = index;
                    OutputPlugs = outputPlugs.ToArray();
                    added = true;
                }
            }

            if (added)
                OnOutputPlugAdded(new OutputPlugEventArgs<T>(outputPlug));
        }

        public void RemoveOutputPlug(OutputPlug<T> outputPlug)
        {
            if (outputPlug == null)
                throw new ArgumentNullException("outputPlug");

            var removed = false;
            lock (outputPlugsSyncRoot)
            {
                if (outputPlugs.Contains(outputPlug, Utility.UniquelyIdentifiableEqualityComparer))
                {
                    outputPlugs.Remove(outputPlug);
                    outputPlug.WorkflowItem = null;
                    outputPlug.Index = -1;
                    ResetOutputIndices();
                    OutputPlugs = outputPlugs.ToArray();
                    removed = true;
                }
            }

            if (removed)
                OnOutputPlugRemoved(new OutputPlugEventArgs<T>(outputPlug));
        }

        protected virtual void OnOutputPlugAdded(OutputPlugEventArgs<T> e)
        {
            var handler = OutputPlugAdded;
            if (handler != null)
                handler(this, e);
        }

        protected virtual void OnOutputPlugRemoved(OutputPlugEventArgs<T> e)
        {
            var handler = OutputPlugRemoved;
            if (handler != null)
                handler(this, e);
        }

        #endregion // OutputPlugs Add and Remove

        private void ResetInputIndices()
        {
            lock (inputPlugsSyncRoot)
            {
                for (var i = 0; i < inputPlugs.Count; i++)
                    inputPlugs[i].Index = i;
            }
        }

        private void ResetOutputIndices()
        {
            lock (outputPlugsSyncRoot)
            {
                for (var i = 0; i < outputPlugs.Count; i++)
                    outputPlugs[i].Index = i;
            }
        }

        public WorkflowItem<T>[] GetInputWorkflowItems()
        {
            lock (inputPlugsSyncRoot)
            {
                return inputPlugs.Select(i => i.RemoteWorkflowItem).ToArray();
            }
        }

        public WorkflowItem<T>[] GetOutputWorkflowItems()
        {
            lock (outputPlugsSyncRoot)
            {
                return outputPlugs.SelectMany(o => o.GetRemoteWorkflowItems()).ToArray();
            }
        }

        public bool Equals(WorkflowItem<T> other)
        {
            return UniquelyIdentifiableExtensions.Equals(this, other);
        }

        /// <summary>
        /// Gets whether the WorkflowItem is an orphan WorkflowItem or not.
        /// </summary>
        /// <remarks>
        /// A WorkflowItem is said to be an orphan when it has no plugs at all or none of its plugs are connected.
        /// </remarks>
        public bool IsOrphan
        {
            get
            {
                lock (inputPlugsSyncRoot)
                {
                    if (inputPlugs.Any())
                    {
                        if (inputPlugs.Any(p => p.RemoteOutputPlug != null))
                            return false;
                    }
                }

                lock (outputPlugsSyncRoot)
                {
                    if (outputPlugs.Any())
                    {
                        if (outputPlugs.Any(p => p.RemoteInputPlugs.Length > 0))
                            return false;
                    }
                }

                return true;
            }
        }

        /// <summary>
        /// Gets whether the WorkflowItem is fully connected or not.
        /// </summary>
        /// <remarks>
        /// A WorkflowItem is said to be fully connected when it has at least one plug, either input or output,
        /// and that all its input plugs, if any, are connected, and that all its output plugs, if any, are connected.
        /// </remarks>
        public bool IsFullyConnected
        {
            get
            {
                lock (inputPlugsSyncRoot)
                {
                    lock (outputPlugsSyncRoot)
                    {
                        if (inputPlugs.Any() == false && outputPlugs.Any() == false)
                            return false;
                    }
                }

                lock (inputPlugsSyncRoot)
                {
                    if (inputPlugs.Any())
                    {
                        if (AreAllInputPlugsConnected() == false)
                            return false;
                    }
                }

                lock (outputPlugsSyncRoot)
                {
                    if (outputPlugs.Any())
                    {
                        if (AreAllOutputPlugsConnected() == false)
                            return false;
                    }
                }

                return true;
            }
        }

        /// <summary>
        /// Gets whether the WorkflowItem is properly connected or not.
        /// </summary>
        /// <remarks>
        /// A WorkflowItem is said to be properly connected when it has at least one plug, either input or output,
        /// and that all its input plugs, if any, are connected, and that at least one output plug, if there is, is connected.
        /// </remarks>
        public bool IsProperlyConnected
        {
            get
            {
                lock (inputPlugsSyncRoot)
                {
                    lock (outputPlugsSyncRoot)
                    {
                        if (inputPlugs.Any() == false && outputPlugs.Any() == false)
                            return false;
                    }
                }

                lock (inputPlugsSyncRoot)
                {
                    if (inputPlugs.Any())
                    {
                        if (AreAllInputPlugsConnected() == false)
                            return false;
                    }
                }

                lock (outputPlugsSyncRoot)
                {
                    if (outputPlugs.Any())
                    {
                        if (AreAnyOutputPlugsConnected() == false)
                            return false;
                    }
                }

                return true;
            }
        }

        /// <summary>
        /// Gets whether the WorkflowItem can represent an input of a graph or not.
        /// </summary>
        /// <remarks>
        /// A WorkflowItem is said to be a graph input when it has no input plug and it has
        /// at least one output plug and at least one of its output plugs is connected.
        /// </remarks>
        public bool IsGraphInput
        {
            get
            {
                lock (inputPlugsSyncRoot)
                {
                    if (inputPlugs.Any())
                        return false;
                }

                lock (outputPlugsSyncRoot)
                {
                    if (outputPlugs.Any() == false)
                        return false;
                    if (AreAnyOutputPlugsConnected() == false)
                        return false;
                }

                return true;
            }
        }

        protected virtual bool AreAnyOutputPlugsConnected()
        {
            return outputPlugs.Any(p => p.RemoteInputPlugs.Length > 0);
        }

        protected virtual bool AreAllOutputPlugsConnected()
        {
            return outputPlugs.All(p => p.RemoteInputPlugs.Length > 0);
        }

        /// <summary>
        /// Gets whether the WorkflowItem can represent an output of a graph or not.
        /// </summary>
        /// <remarks>
        /// A WorkflowItem is said to be a graph output when it has no output plug and
        /// it has at least one input plug and all of its input plugs are connected.
        /// </remarks>
        public bool IsGraphOutput
        {
            get
            {
                lock (outputPlugsSyncRoot)
                {
                    if (outputPlugs.Any())
                        return false;
                }

                lock (inputPlugsSyncRoot)
                {
                    if (inputPlugs.Any() == false)
                        return false;
                    if (AreAllInputPlugsConnected() == false)
                        return false;
                }

                return true;
            }
        }

        protected virtual bool AreAnyInputPlugsConnected()
        {
            return inputPlugs.Any(p => p.RemoteOutputPlug != null);
        }

        protected virtual bool AreAllInputPlugsConnected()
        {
            return inputPlugs.All(p => p.RemoteOutputPlug != null);
        }

        /// <summary>
        /// Checks whether the WorkflowItem set makes a fully connected graph.
        /// This overload excludes orphans.
        /// </summary>
        /// <remarks>For more information about the notion of 'fully connected', please refer to
        /// the IsFullyConnected property of the WorkflowItem class.</remarks>
        /// <param name="workflowItems">The set of WorkflowItem that make up the graph.</param>
        /// <returns>Returns true if the WorkflowItem set produces a fully connected graph, false otherwise.</returns>
        public static bool IsGraphFullyConnected(IEnumerable<WorkflowItem<T>> workflowItems)
        {
            return IsGraphFullyConnected(workflowItems, true);
        }

        /// <summary>
        /// Checks whether the WorkflowItem set makes a fully connected graph.
        /// </summary>
        /// <remarks>For more information about the notion of 'fully connected', please refer to
        /// the IsFullyConnected property of the WorkflowItem class.</remarks>
        /// <param name="workflowItems">The set of WorkflowItem that make up the graph.</param>
        /// <param name="excludeOrphans">Flag indicating whether orphans
        /// (WorkflowItems that are not connected at all) are not taking part of a fully
        /// connected graph or not.</param>
        /// <returns>Returns true if the WorkflowItem set produces a fully connected graph, false otherwise.</returns>
        public static bool IsGraphFullyConnected(
            IEnumerable<WorkflowItem<T>> workflowItems,
            bool excludeOrphans)
        {
            return IsGraphConnected(workflowItems, w => w.IsFullyConnected, excludeOrphans);
        }

        /// <summary>
        /// Checks whether the WorkflowItem set makes a properly connected graph.
        /// This overload excludes orphans.
        /// </summary>
        /// <remarks>For more information about the notion of 'properly connected', please refer to
        /// the IsProperlyConnected property of the WorkflowItem class.</remarks>
        /// <param name="workflowItems">The set of WorkflowItem that make up the graph.</param>
        /// <returns>Returns true if the WorkflowItem set produces a properly connected graph, false otherwise.</returns>
        public static bool IsGraphProperlyConnected(IEnumerable<WorkflowItem<T>> workflowItems)
        {
            return IsGraphProperlyConnected(workflowItems, true);
        }

        /// <summary>
        /// Checks whether the WorkflowItem set makes a properly connected graph.
        /// </summary>
        /// <remarks>For more information about the notion of 'properly connected', please refer to
        /// the IsProperlyConnected property of the WorkflowItem class.</remarks>
        /// <param name="workflowItems">The set of WorkflowItem that make up the graph.</param>
        /// <param name="excludeOrphans">Flag indicating whether orphans
        /// (WorkflowItems that are not connected at all) are not taking part of a properly
        /// connected graph or not.</param>
        /// <returns>Returns true if the WorkflowItem set produces a properly connected graph, false otherwise.</returns>
        public static bool IsGraphProperlyConnected(
            IEnumerable<WorkflowItem<T>> workflowItems,
            bool excludeOrphans)
        {
            return IsGraphConnected(workflowItems, w => w.IsProperlyConnected, excludeOrphans);
        }

        private static bool IsGraphConnected(
            IEnumerable<WorkflowItem<T>> workflowItems,
            Func<WorkflowItem<T>, bool> connectionCheck,
            bool excludeOrphans)
        {
            var connected = false;

            foreach (var workflowItem in workflowItems)
            {
                if (excludeOrphans)
                {
                    if (workflowItem.IsOrphan)
                        continue;
                }

                connected = true;

                if (connectionCheck(workflowItem))
                    continue;

                return false;
            }

            return connected;
        }

        public IEnumerable<WorkflowItem<T>> GetConnectedGroup(bool allConnections)
        {
            var passedWorkflowItems = new HashSet<WorkflowItem<T>>(Utility.UniquelyIdentifiableEqualityComparer);

            var stack = new Stack<WorkflowItem<T>>();
            stack.Push(this);

            while (stack.Count > 0)
            {
                var stackWorkflowItem = stack.Pop();

                if (passedWorkflowItems.Add(stackWorkflowItem) == false)
                    continue;

                if (allConnections)
                {
                    foreach (var targetWorkflowItem in stackWorkflowItem.GetOutputWorkflowItems().Where(w => w != null))
                        stack.Push(targetWorkflowItem);
                }

                foreach (var sourceWorkflowItem in stackWorkflowItem.GetInputWorkflowItems().Where(w => w != null))
                    stack.Push(sourceWorkflowItem);
            }

            return passedWorkflowItems;
        }
    }
}
