﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

namespace Workflow.Core
{
    public class OutputPlug<T> : PlugBase<T>
    {
        public OutputPlug()
        {
            RemoteInputPlugs = new InputPlug<T>[0];
        }

        public T Value { get; private set; }

        internal void SetValue(T value, bool forcePropagate = false)
        {
            if (EqualityComparer<T>.Default.Equals(Value, value))
            {
                if (forcePropagate)
                    Propagate();
                return;
            }

            var oldValue = Value;

            OnValueChanging(new ValueChangedEventArgs<T>(oldValue, value));

            Value = value;
            Propagate();

            OnValueChanged(new ValueChangedEventArgs<T>(oldValue, value));
        }

        private void Propagate()
        {
            lock (remoteInputPlugsSyncRoot)
            {
                foreach (var input in remoteInputPlugs)
                    input.SetValue(Value);
            }
        }

        protected virtual void OnValueChanging(ValueChangedEventArgs<T> e)
        {
            var handler = ValueChanging;
            if (handler != null)
                handler(this, e);
        }

        protected virtual void OnValueChanged(ValueChangedEventArgs<T> e)
        {
            var handler = ValueChanged;
            if (handler != null)
                handler(this, e);
        }

        public event EventHandler<ValueChangedEventArgs<T>> ValueChanging;
        public event EventHandler<ValueChangedEventArgs<T>> ValueChanged;

        private readonly object remoteInputPlugsSyncRoot = new object();
        private readonly HashSet<InputPlug<T>> remoteInputPlugs = new HashSet<InputPlug<T>>(new UniquelyIdentifiableEqualityComparer());

        public InputPlug<T>[] RemoteInputPlugs { get; private set; }

        internal void AddInputPlug(InputPlug<T> inputPlug)
        {
            if (inputPlug == null)
                throw new ArgumentNullException("inputPlug");

            bool added;
            lock (remoteInputPlugsSyncRoot)
            {
                added = remoteInputPlugs.Add(inputPlug);
                RemoteInputPlugs = remoteInputPlugs.ToArray();
            }

            if (added)
                OnInputPlugAdded(new InputPlugEventArgs<T>(inputPlug));
        }

        internal void RemoveInputPlug(InputPlug<T> inputPlug)
        {
            if (inputPlug == null)
                throw new ArgumentNullException("inputPlug");

            bool removed;
            lock (remoteInputPlugsSyncRoot)
            {
                removed = remoteInputPlugs.Remove(inputPlug);
                RemoteInputPlugs = remoteInputPlugs.ToArray();
            }

            if (removed)
                OnInputPlugRemoved(new InputPlugEventArgs<T>(inputPlug));
        }

        protected virtual void OnInputPlugAdded(InputPlugEventArgs<T> e)
        {
            var handler = InputPlugAdded;
            if (handler != null)
                handler(this, e);
        }

        protected virtual void OnInputPlugRemoved(InputPlugEventArgs<T> e)
        {
            var handler = InputPlugRemoved;
            if (handler != null)
                handler(this, e);
        }

        public event EventHandler<InputPlugEventArgs<T>> InputPlugAdded;
        public event EventHandler<InputPlugEventArgs<T>> InputPlugRemoved;

        public WorkflowItem<T>[] GetRemoteWorkflowItems()
        {
            lock (remoteInputPlugsSyncRoot)
            {
                return remoteInputPlugs
                    .Select(i => i.WorkflowItem)
                    .Distinct(Utility.UniquelyIdentifiableEqualityComparer)
                    .OfType<WorkflowItem<T>>()
                    .ToArray();
            }
        }
    }
}
