﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

namespace ShaderGenerator.GLSL
{
    public class PlugValue : IEquatable<PlugValue>
    {
        public PlugValue(ShaderTyping.ShaderTypeDefinition type, string expression, string target)
        {
            SetValues(type, expression, target);
        }

        public PlugValue(PlugValue valueHolder)
        {
            if (valueHolder == null)
                throw new ArgumentNullException("valueHolder");

            SetValues(valueHolder.Type, valueHolder.Expression, valueHolder.Target);
        }

        public PlugValue(Variable variable)
        {
            if (variable == null)
                throw new ArgumentNullException("variable");

            SetValues(variable.Type, variable.Name, variable.Name);
        }

        private void SetValues(ShaderTyping.ShaderTypeDefinition type, string expression, string target)
        {
            if (type == null)
                throw new ArgumentNullException("type");
            if (string.IsNullOrWhiteSpace(expression))
                throw new ArgumentException(string.Format(Messages.EXCEPTION_INVALID_ARGUMENT, "expression"), "expression");
            if (string.IsNullOrWhiteSpace(target))
                throw new ArgumentException(string.Format(Messages.EXCEPTION_INVALID_ARGUMENT, "target"), "target");

            Type = type;
            Expression = expression;
            Target = target;
            BaseTarget = GetTargetBase(target);
        }

        public ShaderTyping.ShaderTypeDefinition Type { get; private set; }
        public string Expression { get; private set; }
        public string Target { get; private set; }
        public string BaseTarget { get; private set; }

        public static string GetTargetBase(string target)
        {
            if (string.IsNullOrWhiteSpace(target))
                throw new ArgumentException(string.Format(Messages.EXCEPTION_INVALID_ARGUMENT, "target"), "target");

            var i = target.IndexOf('.');
            return i < 0 ? target : target.Substring(0, i);
        }

        public static string GetTargetFields(string target)
        {
            if (string.IsNullOrWhiteSpace(target))
                throw new ArgumentException(string.Format(Messages.EXCEPTION_INVALID_ARGUMENT, "target"), "target");

            var i = target.IndexOf('.');
            return i < 0 ? string.Empty : target.Substring(i);
        }

        public override string ToString()
        {
            return string.Format("{0} {1} -- {2}", Type, Expression, Target);
        }

        public override int GetHashCode()
        {
            return string.Format("{0}{1}{2}{3}", Type, Expression, Target, BaseTarget).GetHashCode();
        }

        public override bool Equals(object obj)
        {
            if ((obj is PlugValue) == false)
                return false;

            return Equals((PlugValue)obj);
        }

        public bool Equals(PlugValue other)
        {
            if (other == null)
                return false;

            return Type.Equals(other.Type) &&
                Expression == other.Expression &&
                Target == other.Target &&
                BaseTarget == other.BaseTarget;
        }
    }
}
