﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using EffectCombiner.Core;
using ShaderGenerator.Core;
using ShaderTyping;

namespace ShaderGenerator.GLSL
{
    public enum DependencyMode
    {
        Link,
        Insert
    }

    public class GlslGenerationContext : GenerationContext<FunctionDefinition, PlugValue>
    {
        private readonly List<Uri> sourceFiles = new List<Uri>();
        private readonly List<FunctionCall> functionCalls = new List<FunctionCall>();

        internal readonly Dictionary<Guid, PlugValue[]> nodeOutputs = new Dictionary<Guid, PlugValue[]>();

        public GlslGenerationContext(IResourceLocator resourceLocator = null, IContentRepository contentRepository = null, string shaderMainName = "main")
        {
            if (string.IsNullOrWhiteSpace(shaderMainName))
                throw new ArgumentException(string.Format(Messages.EXCEPTION_INVALID_ARGUMENT, "shaderMainName"), "shaderMainName");

            this.DefinitionMode = DependencyMode.Link;
            this.VariableGenerator = new VariableGenerator();
            this.ResourceLocator = resourceLocator;
            this.ContentRepository = contentRepository;
            this.ShaderMainName = shaderMainName;
        }

        /// <summary>
        /// シェーダコードを取得します。
        /// </summary>
        public string FullShader { get; private set; }

        public DependencyMode DefinitionMode { get; set; }
        public VariableGenerator VariableGenerator { get; private set; }
        public IResourceLocator ResourceLocator { get; private set; }
        public IContentRepository ContentRepository { get; private set; }

        /// <summary>
        /// シェーダコードの最終出力名を取得します。
        /// </summary>
        public string ShaderMainName { get; private set; }

        public override void BeginGeneration()
        {
            base.BeginGeneration();

            if (this.ContentRepository == null && this.DefinitionMode == DependencyMode.Insert)
                throw new InvalidOperationException(Messages.EXCEPTION_CONTENT_REPOSITORY_REQUIRED);
            if (this.ResourceLocator == null && this.DefinitionMode == DependencyMode.Insert)
                throw new InvalidOperationException(Messages.EXCEPTION_RESOURCE_LOCATOR_REQUIRED);

            this.sourceFiles.Clear();
            this.functionCalls.Clear();
            this.VariableGenerator.Reset();
            this.FullShader = string.Empty;
        }

        public void AddFunctionCall(Guid guid, FunctionCall functionCall)
        {
            this.functionCalls.Add(functionCall);
            this.nodeOutputs[guid] = functionCall.Outputs;

            if (this.ResourceLocator != null)
            {
                var source = this.ResourceLocator.GetResourceLocation(functionCall.Function.GetMangling());

                if (this.sourceFiles.Contains(source) == false)
                {
                    this.sourceFiles.Add(source);
                }
            }
        }

        public override void EndGeneration()
        {
            this.FullShader =
                this.GenerateHeader() + Environment.NewLine +
                this.GenerateDefinitionsBlock() + Environment.NewLine +
                this.GenerateCodeBlock() + Environment.NewLine;

            base.EndGeneration();
        }

        #region Code generation

        private string GenerateHeader()
        {
            string shaderCode =
                "//" + Environment.NewLine +
                "// Generated shader code" + Environment.NewLine +
                "//" + Environment.NewLine;
            return shaderCode;
        }

        private string GenerateDefinitionsBlock()
        {
            string shaderCode = string.Empty;

            switch (this.DefinitionMode)
            {
                case DependencyMode.Link:
                    foreach (var functionDefinition in this.functionCalls.Select(x => x.Function).Distinct())
                    {
                        shaderCode += functionDefinition.GetPrototype() + Environment.NewLine;
                    }
                    break;
                case DependencyMode.Insert:
                    foreach (var sourceFile in this.sourceFiles)
                    {
                        shaderCode += this.ContentRepository.GetContent(sourceFile, Encoding.UTF8);
                    }
                    break;
            }

            return shaderCode;
        }

        /// <summary>
        /// シェーダコードのmain関数部分を生成します。
        /// </summary>
        /// <returns>シェーダコードのmain関数部分を返します。</returns>
        private string GenerateCodeBlock()
        {
            string indent = new string(' ', 4);

            string mainCode = string.Join(Environment.NewLine, this.functionCalls.Select(x => x.GetCallCode(indent)));

            return "void " + this.ShaderMainName + "()" + Environment.NewLine + "{" + Environment.NewLine + mainCode + "}";
        }

        #endregion
    }
}
